package com.vungle.ads.internal.model

import com.vungle.ads.internal.Constants
import kotlinx.serialization.SerialName
import kotlinx.serialization.Serializable
import kotlinx.serialization.Transient

@Serializable
data class ConfigPayload(
    @SerialName("reuse_assets") val cleverCache: CleverCache? = null,
    @SerialName("config") val configSettings: ConfigSettings? = null,
    @SerialName("endpoints") val endpoints: Endpoints? = null,
    @SerialName("log_metrics") val logMetricsSettings: LogMetricsSettings? = null,
    @SerialName("placements") val placements: List<Placement>? = null,
    @SerialName("user") val userPrivacy: UserPrivacy? = null,
    @SerialName("config_extension") val configExtension: String? = null,
    @SerialName("disable_ad_id") val disableAdId: Boolean? = true,
    @SerialName("ri_enabled") val isReportIncentivizedEnabled: Boolean? = null,
    @SerialName("session_timeout") val sessionTimeout: Int? = null,
    @SerialName("wait_for_connectivity_for_tpat") val waitForConnectivityForTPAT: Boolean? = null,
    @SerialName("sdk_session_timeout") val signalSessionTimeout: Int? = null,
    @SerialName("signals_disabled") val signalsDisabled: Boolean? = null,
    @SerialName("fpd_enabled") val fpdEnabled: Boolean? = null,
    @SerialName("rta_debugging") val rtaDebugging: Boolean? = null,
    @SerialName("config_last_validated_ts") var configLastValidatedTimestamp: Long? = null,
    @SerialName("auto_redirect") var autoRedirect: AutoRedirect? = null,
    @SerialName("retry_prioritized_tpat") var retryPriorityTPATs: Boolean? = null,
    @SerialName("enable_ot") var enableOT: Boolean? = null,
) {
    @Serializable
    data class Endpoints(
        @SerialName("ads") val adsEndpoint: String? = null,
        @SerialName("ri") val riEndpoint: String? = null,
        @SerialName("error_logs") val errorLogsEndpoint: String? = null,
        @SerialName("metrics") val metricsEndpoint: String? = null,
        @SerialName("mraid_js") val mraidEndpoint: String? = null
    )

    @Serializable
    data class ConfigSettings(@SerialName("refresh_interval") val refreshTime: Long? = null)

    @Serializable
    data class UserPrivacy(
        @SerialName("gdpr") val gdpr: GDPRSettings? = null,
        @SerialName("iab") val iab: IABSettings? = null
    )

    @Serializable
    data class IABSettings(
        @SerialName("tcf_status") val tcfStatus: Int? = null,
    ) {
        enum class TcfStatus(val rawValue: Int) {
            ALLOW_ID(0),
            DISABLE_ID(1),
            LEGACY(2);

            companion object {
                private val rawValueMap = values().associateBy { it.rawValue }

                fun fromRawValue(rawValue: Int?): TcfStatus? {
                    return rawValueMap[rawValue]
                }
            }
        }
    }


    @Serializable
    data class GDPRSettings(
        @SerialName("is_country_data_protected") val isCountryDataProtected: Boolean? = null,
        @SerialName("consent_title") val consentTitle: String? = null,
        @SerialName("consent_message") val consentMessage: String? = null,
        @SerialName("consent_message_version") val consentMessageVersion: String? = null,
        @SerialName("button_accept") val buttonAccept: String? = null,
        @SerialName("button_deny") val buttonDeny: String? = null
    )

    @Serializable
    data class LogMetricsSettings(
        @SerialName("error_log_level") val errorLogLevel: Int? = null,
        @SerialName("metrics_is_enabled") val metricsEnabled: Boolean? = null
    )

    @Serializable
    data class CleverCache(
        @SerialName("enabled") val enabled: Boolean? = false,
        @SerialName("disk_size") val diskSize: Long? = 1 * 1000, //MB
        @SerialName("disk_percentage") val diskPercentage: Int? = 3
    )

    @Serializable
    data class AutoRedirect(
        @SerialName("allow_auto_redirect") val allowAutoRedirect: Boolean? = false,
        @SerialName("after_click_ms") val afterClickDuration: Long? = Long.MAX_VALUE)
}

@Serializable
data class Placement(
    @SerialName("placement_ref_id") val referenceId: String,
    @SerialName("is_hb") val headerBidding: Boolean = false,
    @SerialName("type") val type: String? = null
) {

    @Transient
    var wakeupTime: Long? = null

    fun isNative() = type == Constants.PLACEMENT_TYPE_NATIVE

    fun isInline() = type == Constants.PLACEMENT_TYPE_IN_LINE

    fun isBanner() = type == Constants.PLACEMENT_TYPE_BANNER

    fun isMREC() = type == Constants.PLACEMENT_TYPE_MREC

    fun isInterstitial() = type == Constants.PLACEMENT_TYPE_INTERSTITIAL

    fun isAppOpen() = type == Constants.PLACEMENT_TYPE_APP_OPEN

    fun isRewardedVideo() = type == Constants.PLACEMENT_TYPE_REWARDED

    fun snooze(sleepTime: Long) {
        wakeupTime = System.currentTimeMillis() + sleepTime * 1000
    }
}
