package com.vungle.ads

import android.content.Context
import com.vungle.ads.internal.model.AdPayload
import com.vungle.ads.internal.presenter.AdPlayCallback
import com.vungle.ads.internal.protos.Sdk
import com.vungle.ads.internal.util.ThreadUtil

abstract class BaseFullscreenAd(context: Context, placementId: String, adConfig: AdConfig) :
    BaseAd(context, placementId, adConfig), FullscreenAd {

    override fun load(adMarkup: String?) {
        signaledAd = signalManager.getSignaledAd(placementId)
        super.load(adMarkup)
    }

    override fun onAdLoaded(advertisement: AdPayload) {
        super.onAdLoaded(advertisement)
        signaledAd?.adAvailabilityCallbackTime = System.currentTimeMillis()
    }

    override fun play(context: Context?) {
        AnalyticsClient.logMetric(
            SingleValueMetric(Sdk.SDKMetric.SDKMetricType.PLAY_AD_API), logEntry)
        responseToShowMetric.markEnd()
        AnalyticsClient.logMetric(responseToShowMetric, logEntry)
        showToFailMetric.markStart()
        showToCloseMetric.markStart()
        signaledAd?.let {
            it.playAdTime = System.currentTimeMillis()
            it.calculateTimeBetweenAdAvailabilityAndPlayAd()
            signalManager.registerSignaledAd(context, it)
        }
        adInternal.play(context, object : AdPlayCallback {
            override fun onAdStart(id: String?) {
                signalManager.increaseSessionDepthCounter()
                presentToDisplayMetric.markStart()
                ThreadUtil.runOnUiThread {
                    adListener?.onAdStart(this@BaseFullscreenAd)
                }
            }

            override fun onAdImpression(id: String?) {
                ThreadUtil.runOnUiThread {
                    adListener?.onAdImpression(this@BaseFullscreenAd)
                }

                presentToDisplayMetric.markEnd()
                AnalyticsClient.logMetric(presentToDisplayMetric, logEntry)
                displayToClickMetric.markStart()
            }

            override fun onAdEnd(id: String?) {
                ThreadUtil.runOnUiThread {
                    adListener?.onAdEnd(this@BaseFullscreenAd)
                }
                showToCloseMetric.markEnd()
                AnalyticsClient.logMetric(showToCloseMetric, logEntry)
            }

            override fun onAdClick(id: String?) {
                ThreadUtil.runOnUiThread {
                    adListener?.onAdClicked(this@BaseFullscreenAd)
                }

                displayToClickMetric.markEnd()
                AnalyticsClient.logMetric(displayToClickMetric, logEntry)
            }

            override fun onAdRewarded(id: String?) {
                ThreadUtil.runOnUiThread {
                    val rewardedAdListener: RewardedAdListener? =
                        adListener as? RewardedAdListener
                    rewardedAdListener?.onAdRewarded(this@BaseFullscreenAd)
                }
                AnalyticsClient.logMetric(rewardedMetric, logEntry)
            }

            override fun onAdLeftApplication(id: String?) {
                ThreadUtil.runOnUiThread {
                    adListener?.onAdLeftApplication(this@BaseFullscreenAd)
                }
                AnalyticsClient.logMetric(leaveApplicationMetric, logEntry)
            }

            override fun onFailure(error: VungleError) {
                ThreadUtil.runOnUiThread {
                    adListener?.onAdFailedToPlay(this@BaseFullscreenAd, error)
                }
                showToFailMetric.markEnd()
                AnalyticsClient.logMetric(showToFailMetric, logEntry, "${error.code}")
            }
        })
    }
}
