package com.vungle.ads.internal

import android.Manifest
import android.content.Context
import android.content.pm.PackageManager
import android.os.Build
import androidx.annotation.VisibleForTesting
import androidx.core.content.ContextCompat
import com.vungle.ads.AnalyticsClient
import com.vungle.ads.InitializationListener
import com.vungle.ads.InvalidAppId
import com.vungle.ads.OutOfMemory
import com.vungle.ads.SdkNotInitialized
import com.vungle.ads.SdkVersionTooLow
import com.vungle.ads.ServiceLocator
import com.vungle.ads.ServiceLocator.Companion.inject
import com.vungle.ads.SingleValueMetric
import com.vungle.ads.TimeIntervalMetric
import com.vungle.ads.VungleError
import com.vungle.ads.VungleWrapperFramework
import com.vungle.ads.internal.downloader.Downloader
import com.vungle.ads.internal.executor.Executors
import com.vungle.ads.internal.load.MraidJsLoader
import com.vungle.ads.internal.network.VungleApiClient
import com.vungle.ads.internal.network.VungleHeader
import com.vungle.ads.internal.persistence.FilePreferences
import com.vungle.ads.internal.privacy.PrivacyManager
import com.vungle.ads.internal.protos.Sdk
import com.vungle.ads.internal.task.CleanupJob
import com.vungle.ads.internal.task.JobRunner
import com.vungle.ads.internal.util.Logger
import com.vungle.ads.internal.util.PathProvider
import com.vungle.ads.internal.util.ThreadUtil
import com.vungle.ads.internal.util.Utils
import java.util.concurrent.CopyOnWriteArrayList
import java.util.concurrent.atomic.AtomicBoolean

internal class VungleInitializer {

    companion object {
        private const val TAG = "VungleInitializer"
    }

    @VisibleForTesting
    internal var isInitialized = AtomicBoolean(false)

    @VisibleForTesting
    internal var isInitializing = AtomicBoolean(false)

    private val initializationCallbackArray: CopyOnWriteArrayList<InitializationListener> = CopyOnWriteArrayList()

    private val initDurationMetric =
        TimeIntervalMetric(Sdk.SDKMetric.SDKMetricType.INIT_TO_SUCCESS_CALLBACK_DURATION_MS)

    fun init(
        appId: String,
        context: Context,
        initializationCallback: InitializationListener
    ) {
        AnalyticsClient.logMetric(SingleValueMetric(Sdk.SDKMetric.SDKMetricType.SDK_INIT_API))
        initDurationMetric.markStart()
        initializationCallbackArray.add(initializationCallback)

        if (isAppIdInvalid(appId)) {
            val errorMessage = "App id invalid: $appId, package name: ${context.packageName}"
            onInitError(InvalidAppId(errorMessage).logError())
            return
        }

        if (Utils.isOSVersionInvalid()) {
            val error = "Init: SDK is supported only for API versions ${Build.VERSION_CODES.N_MR1} and above."
            Logger.e(TAG, error)
            onInitError(SdkVersionTooLow(error).logError())
            return
        }

        // Set app id for config manager, file preference will check local cached app id match
        // with this app id or not. If not match, resend config request.
        ConfigManager.setAppId(appId)

        if (isInitialized.get()) {
            Logger.d(TAG, "init already complete")
            onInitSuccess()
            return
        }

        if (isInitializing.getAndSet(true)) {
            Logger.d(TAG, "init already in progress")
            return
        }

        if (ContextCompat.checkSelfPermission(context, Manifest.permission.ACCESS_NETWORK_STATE)
            != PackageManager.PERMISSION_GRANTED
            || ContextCompat.checkSelfPermission(context, Manifest.permission.INTERNET)
            != PackageManager.PERMISSION_GRANTED
        ) {
            Logger.e(TAG, "Network permissions not granted")
            onInitError(
                SdkNotInitialized("Network permissions not granted").logError()
            )
            return
        }

        val sdkExecutors: Executors by inject(context)
        val vungleApiClient: VungleApiClient by inject(context)
        sdkExecutors.backgroundExecutor.execute(
            {
                PrivacyManager.init(context)

                vungleApiClient.initialize(appId)

                /// Configure the instance.
                configure(context, appId)
            }
        ) {
            //failRunnable
            onInitError(
                OutOfMemory("Config: Out of Memory").logError()
            )
        }
    }

    private fun isAppIdInvalid(appId: String) = appId.isBlank() || hasInvalidChar(appId)

    /**
     * Request a configuration from the server. This updates the current list of placements and
     * schedules another configuration job in the future. It also has the side-effect of loading the
     * auto_cached ad if it is not downloaded currently.
     *
     */
    private fun configure(context: Context, appId: String) {
        /// Request a configuration from the server. This happens asynchronously on the network thread.
        try {
            // 1. Check local config cache
            var fromCachedConfig = false
            val filePreferences: FilePreferences by inject(context)
            val initialConfigPayload = ConfigManager.getCachedConfig(filePreferences, appId)

            // if cache exists and not expired
            if (initialConfigPayload != null) {
                fromCachedConfig = true
                ConfigManager.initWithConfig(context, initialConfigPayload, true)
            }

            isInitialized.set(true)
            // Inform the publisher that initialization has succeeded.
            onInitSuccess()

            Logger.d(TAG, "Running cleanup and resend tpat jobs. ${Thread.currentThread().id}")

            // Run cleanup jobs
            val jobRunner: JobRunner by inject(context)
            jobRunner.execute(CleanupJob.makeJobInfo())

            // Call /config asynchronously
            if (!fromCachedConfig) {
                ConfigManager.fetchConfigAsync(context) { result: Boolean ->
                    if (result) {
                        // Download js assets asynchronously
                        downloadMraidJs(context)
                    }
                }
            } else {
                // Download js assets asynchronously
                downloadMraidJs(context)
            }
        } catch (throwable: Throwable) {
            Logger.e(TAG, "Cannot get config", throwable)
        }
    }

    private fun downloadMraidJs(context: Context) {
        val pathProvider: PathProvider by inject(context)
        val downloader: Downloader by inject(context)
        val sdkExecutors: Executors by inject(context)
        MraidJsLoader.downloadJs(pathProvider, downloader, sdkExecutors.backgroundExecutor)
    }

    private fun onInitError(exception: VungleError) {
        isInitializing.set(false)
        val exMsg = exception.localizedMessage ?: "Exception code is ${exception.code}"
        initDurationMetric.metricType = Sdk.SDKMetric.SDKMetricType.INIT_TO_FAIL_CALLBACK_DURATION_MS
        initDurationMetric.markEnd()
        AnalyticsClient.logMetric(initDurationMetric, null, exMsg)
        ThreadUtil.runOnUiThread {
            Logger.e(TAG, "onError")
            initializationCallbackArray.forEach {
                it.onError(exception)
            }
            initializationCallbackArray.clear()
        }
        Logger.e(TAG, exMsg)
    }

    private fun onInitSuccess() {
        isInitializing.set(false)
        initDurationMetric.metricType = Sdk.SDKMetric.SDKMetricType.INIT_TO_SUCCESS_CALLBACK_DURATION_MS
        initDurationMetric.markEnd()
        AnalyticsClient.logMetric(initDurationMetric)
        Logger.d(TAG, "onSuccess ${Thread.currentThread().id}")

        ThreadUtil.runOnUiThread {
            initializationCallbackArray.forEach {
                it.onSuccess()
            }
            initializationCallbackArray.clear()
        }
    }

    private fun hasInvalidChar(value: String): Boolean {
        return value.any { !(it.isLetterOrDigit() || it == '.') }
    }

    fun isInitialized(): Boolean {
        return isInitialized.get()
    }

    /**
     * Lite version of de-initialization, can be used in Unit Tests
     */
    internal fun deInit() {
        ServiceLocator.deInit()
        VungleApiClient.reset()
        isInitialized.set(false)
        isInitializing.set(false)
        initializationCallbackArray.clear()
    }

    fun setIntegrationName(
        wrapperFramework: VungleWrapperFramework,
        wrapperFrameworkVersion: String
    ) {
        if (wrapperFramework == VungleWrapperFramework.none) {
            Logger.e(TAG, "Wrapper is null or is none")
            return
        }
        val originalHeader: String = VungleHeader.headerUa
        val wrapperVersion =
            if (wrapperFrameworkVersion.isNotEmpty()) "/$wrapperFrameworkVersion" else ""
        val newMediation = "${wrapperFramework.name}${wrapperVersion}"
        if (originalHeader.contains(newMediation)) {
            Logger.w(TAG, "Wrapper info already set")
            return
        }
        VungleHeader.headerUa = "$originalHeader;$newMediation"

        if (isInitialized()) {
            Logger.w(
                TAG,
                "VUNGLE WARNING: SDK already initialized, you should've set wrapper info before"
            )
        }
    }

}
