package com.vungle.ads.internal.load

import android.content.Context
import com.vungle.ads.AdMarkupInvalidError
import com.vungle.ads.InvalidBidPayloadError
import com.vungle.ads.ServiceLocator
import com.vungle.ads.SingleValueMetric
import com.vungle.ads.internal.ConfigManager
import com.vungle.ads.internal.Constants.WIN_NOTIFICATION
import com.vungle.ads.internal.downloader.Downloader
import com.vungle.ads.internal.executor.Executors
import com.vungle.ads.internal.network.TpatRequest
import com.vungle.ads.internal.network.TpatSender
import com.vungle.ads.internal.network.VungleApiClient
import com.vungle.ads.internal.omsdk.OMInjector
import com.vungle.ads.internal.protos.Sdk
import com.vungle.ads.internal.util.Logger
import com.vungle.ads.internal.util.PathProvider

class RealtimeAdLoader(
    context: Context,
    vungleApiClient: VungleApiClient,
    sdkExecutors: Executors,
    omInjector: OMInjector,
    downloader: Downloader,
    pathProvider: PathProvider,
    adRequest: AdRequest
) : BaseAdLoader(
    context, vungleApiClient, sdkExecutors, omInjector, downloader, pathProvider, adRequest
) {

    override fun requestAd() {
        val adMarkup = adRequest.adMarkup
        // Support V2 only.
        if (adMarkup == null) {
            onAdLoadFailed(InvalidBidPayloadError().setLogEntry(logEntry).logError())
            return
        }

        if (ConfigManager.rtaDebuggingEnabled()) {
            try {
                val adResponse = adMarkup.getDecodedAdsResponse()
                Logger.d("RTA_DEBUGGER", "$adResponse")
                val vungleApiClient: VungleApiClient by ServiceLocator.inject(context)
                adResponse?.let { RTADebugger(vungleApiClient).reportAdMarkup(it) }
            } catch (ignored: Throwable) {
                // ignore
            }
        }

        val adPayload = adMarkup.getAdPayload()

        if (adMarkup.version != 2 || adPayload == null) {
            onAdLoadFailed(
                AdMarkupInvalidError("The ad response did not contain valid ad markup")
                    .setLogEntry(logEntry).logError()
            )
            return
        }

        handleAdMetaData(
            adPayload,
            SingleValueMetric(Sdk.SDKMetric.SDKMetricType.CONFIG_LOADED_FROM_ADM_LOAD)
        )
    }

    override fun onAdLoadReady() {
        sendWinNotification(advertisement?.getWinNotifications())
    }

    private fun sendWinNotification(notifications: List<String>?) {
        if (notifications?.isEmpty() == true) {
            return
        }
        val tpatSender: TpatSender by ServiceLocator.inject(context)
        notifications?.forEach { notificationUrl ->
            val request = TpatRequest.Builder(notificationUrl)
                .tpatKey(WIN_NOTIFICATION)
                .withLogEntry(logEntry)
                .build()
            tpatSender.sendTpat(request)
        }
    }
}
