package com.vungle.ads.internal

import android.content.Context
import com.vungle.ads.internal.util.Logger
import com.vungle.ads.AdExpiredError
import com.vungle.ads.AdExpiredOnPlayError
import com.vungle.ads.AdMarkupInvalidError
import com.vungle.ads.AdMarkupJsonError
import com.vungle.ads.AdNotLoadedCantPlay
import com.vungle.ads.AnalyticsClient
import com.vungle.ads.InvalidBannerSizeError
import com.vungle.ads.BuildConfig
import com.vungle.ads.InvalidAdStateError
import com.vungle.ads.EmptyBidPayloadError
import com.vungle.ads.PlacementAdTypeMismatchError
import com.vungle.ads.PlacementNotFoundError
import com.vungle.ads.SdkNotInitialized
import com.vungle.ads.ServiceLocator.Companion.inject
import com.vungle.ads.SingleValueMetric
import com.vungle.ads.TimeIntervalMetric
import com.vungle.ads.VungleAdSize
import com.vungle.ads.VungleAds
import com.vungle.ads.VungleError
import com.vungle.ads.internal.ConfigManager.CONFIG_LAST_VALIDATE_TS_DEFAULT
import com.vungle.ads.internal.Constants.AD_VISIBILITY_INVISIBLE
import com.vungle.ads.internal.downloader.Downloader
import com.vungle.ads.internal.executor.SDKExecutors
import com.vungle.ads.internal.load.AdLoaderCallback
import com.vungle.ads.internal.load.AdRequest
import com.vungle.ads.internal.load.BaseAdLoader
import com.vungle.ads.internal.load.DefaultAdLoader
import com.vungle.ads.internal.load.RealtimeAdLoader
import com.vungle.ads.internal.model.AdPayload
import com.vungle.ads.internal.model.BidPayload
import com.vungle.ads.internal.model.Placement
import com.vungle.ads.internal.network.TpatRequest
import com.vungle.ads.internal.network.TpatSender
import com.vungle.ads.internal.network.VungleApiClient
import com.vungle.ads.internal.omsdk.OMInjector
import com.vungle.ads.internal.presenter.AdEventListener
import com.vungle.ads.internal.presenter.AdPlayCallback
import com.vungle.ads.internal.presenter.AdPlayCallbackWrapper
import com.vungle.ads.internal.protos.Sdk
import com.vungle.ads.internal.protos.Sdk.SDKError
import com.vungle.ads.internal.task.CleanupJob
import com.vungle.ads.internal.task.JobRunner
import com.vungle.ads.internal.ui.AdActivity
import com.vungle.ads.internal.util.ActivityManager
import com.vungle.ads.internal.util.LogEntry
import com.vungle.ads.internal.util.PathProvider
import kotlinx.serialization.ExperimentalSerializationApi
import kotlinx.serialization.decodeFromString
import kotlinx.serialization.json.Json
import java.lang.ref.WeakReference

abstract class AdInternal(val context: Context) : AdLoaderCallback {
    companion object {
        private const val TAG = "AdInternal"

        private val THROW_ON_ILLEGAL_TRANSITION = BuildConfig.DEBUG

        @OptIn(ExperimentalSerializationApi::class)
        private val json = Json {
            ignoreUnknownKeys = true
            encodeDefaults = true
            explicitNulls = false
        }
    }

    var adState: AdState = AdState.NEW
        set(value) {
            if (value.isTerminalState()) {
                advertisement?.eventId()?.let {
                    val jobRunner: JobRunner by inject(context)
                    val jobInfo = CleanupJob.makeJobInfo(it)
                    jobRunner.execute(jobInfo)
                }
            }

            field = adState.transitionTo(value)
        }

    var advertisement: AdPayload? = null
    var placement: Placement? = null
    var bidPayload: BidPayload? = null
    private var adLoaderCallback: AdLoaderCallback? = null
    private val vungleApiClient: VungleApiClient by inject(context)

    private var baseAdLoader: BaseAdLoader? = null

    private var requestMetric: TimeIntervalMetric? = null
    private var loadMetric: TimeIntervalMetric? = null
    internal val showToValidationMetric: TimeIntervalMetric =
        TimeIntervalMetric(Sdk.SDKMetric.SDKMetricType.AD_SHOW_TO_VALIDATION_DURATION_MS)
    internal val validationToPresentMetric: TimeIntervalMetric =
        TimeIntervalMetric(Sdk.SDKMetric.SDKMetricType.AD_VALIDATION_TO_PRESENT_DURATION_MS)

    private var playContext: WeakReference<Context>? = null

    internal var logEntry: LogEntry? = null

    fun canPlayAd(onPlay: Boolean = false): VungleError? {
        val error: VungleError =
            when {
                advertisement == null -> AdNotLoadedCantPlay("adv is null on onPlay=$onPlay")

                adState == AdState.PLAYING -> InvalidAdStateError(
                    SDKError.Reason.AD_IS_PLAYING,
                    "Current ad is playing"
                )

                adState != AdState.READY -> InvalidAdStateError(
                    SDKError.Reason.AD_NOT_LOADED,
                    "$adState is not READY"
                )

                advertisement?.hasExpired() == true -> {
                    val extraMessage = "Ad expiry: ${advertisement?.adUnit()?.expiry}, device: ${System.currentTimeMillis()}"
                    if (onPlay) {
                        AdExpiredOnPlayError(extraMessage)
                    } else {
                        AdExpiredError(extraMessage)
                    }
                }

                else -> return null
            }
        if (onPlay) {
            error.setLogEntry(logEntry).logErrorNoReturnValue()
        }
        return error
    }

    abstract fun isValidAdTypeForPlacement(placement: Placement): Boolean

    abstract fun isValidAdSize(adSize: VungleAdSize?): Boolean

    abstract fun getAdSizeForAdRequest(): VungleAdSize?

    fun loadAd(
        placementId: String,
        adMarkup: String?,
        adLoaderCallback: AdLoaderCallback
    ) {
        AnalyticsClient.logMetric(Sdk.SDKMetric.SDKMetricType.LOAD_AD_API, logEntry = logEntry)
        val defaultType = Sdk.SDKMetric.SDKMetricType.AD_LOAD_TO_CALLBACK_ADO_DURATION_MS
        loadMetric = TimeIntervalMetric(defaultType)
        loadMetric?.markStart()

        this.adLoaderCallback = adLoaderCallback
        if (!VungleAds.isInitialized()) {
            adLoaderCallback.onFailure(
                SdkNotInitialized("SDK not initialized").setLogEntry(logEntry).logError()
            )
            return
        }

        // Check the placement validity from the config if it exists
        var pl: Placement? = ConfigManager.getPlacement(placementId)
        if (pl != null) {
            this.placement = pl
            if (!isValidAdTypeForPlacement(pl)) {
                adLoaderCallback.onFailure(
                    PlacementAdTypeMismatchError(pl.referenceId).setLogEntry(logEntry).logError()
                )
                return
            }
            if (pl.headerBidding && adMarkup.isNullOrEmpty()) {
                adLoaderCallback.onFailure(EmptyBidPayloadError(placementId)
                    .setLogEntry(logEntry).logError())
                return
            }
        } else if (ConfigManager.configLastValidatedTimestamp() == CONFIG_LAST_VALIDATE_TS_DEFAULT) {
            // Create a temp placement, do not validate.
            pl = Placement(placementId)
            this.placement = pl
        } else {
            adLoaderCallback.onFailure(
                PlacementNotFoundError(placementId).setLogEntry(logEntry).logError()
            )
            return

        }
        val adSize: VungleAdSize? = getAdSizeForAdRequest()
        if (!isValidAdSize(adSize)) {
            adLoaderCallback.onFailure(
                InvalidBannerSizeError(adSize?.toString())
                    .setLogEntry(logEntry).logError()
            )
            return
        }

        if (adState != AdState.NEW) {
            val error = when (adState) {
                AdState.NEW -> TODO() // Not possible
                AdState.LOADING -> SDKError.Reason.AD_IS_LOADING
                AdState.READY -> SDKError.Reason.AD_ALREADY_LOADED
                AdState.PLAYING -> SDKError.Reason.AD_IS_PLAYING
                AdState.FINISHED -> SDKError.Reason.AD_CONSUMED
                AdState.ERROR -> SDKError.Reason.AD_ALREADY_FAILED
            }
            adLoaderCallback.onFailure(
                InvalidAdStateError(error, "$adState state is incorrect for load")
                    .setLogEntry(logEntry).logError()
            )
            return
        }

        val type = Sdk.SDKMetric.SDKMetricType.AD_REQUEST_TO_CALLBACK_ADO_DURATION_MS
        requestMetric = TimeIntervalMetric(type)
        requestMetric?.markStart()
        if (!adMarkup.isNullOrEmpty()) {
            try {
                bidPayload = json.decodeFromString<BidPayload>(adMarkup)
            } catch (e: IllegalArgumentException) {
                adLoaderCallback.onFailure(
                    AdMarkupInvalidError("Unable to decode payload into BidPayload object. Error: ${e.localizedMessage}")
                        .setLogEntry(logEntry).logError()
                )
                return
            } catch (t: Throwable) {
                adLoaderCallback.onFailure(
                    AdMarkupJsonError(t.localizedMessage)
                        .setLogEntry(logEntry).logError()
                )
                return
            }
        }

        adState = AdState.LOADING

        val omInjector: OMInjector by inject(context)
        val sdkExecutors: SDKExecutors by inject(context)
        val pathProvider: PathProvider by inject(context)
        val downloader: Downloader by inject(context)

        if (adMarkup.isNullOrEmpty()) {
            val adRequest = AdRequest(pl, null, adSize)
            val adLoader = DefaultAdLoader(
                context,
                vungleApiClient,
                sdkExecutors,
                omInjector,
                downloader,
                pathProvider,
                adRequest
            )
            baseAdLoader = adLoader
        } else {
            val adRequest = AdRequest(pl, bidPayload, adSize)
            val adLoader = RealtimeAdLoader(
                context,
                vungleApiClient,
                sdkExecutors,
                omInjector,
                downloader,
                pathProvider,
                adRequest
            )
            baseAdLoader = adLoader
        }
        baseAdLoader?.logEntry = logEntry
        baseAdLoader?.loadAd(this)
    }

    internal fun cancelDownload() {
        if (advertisement?.isPartialDownloadEnabled() == true) {
            Logger.d(TAG, "Skip cancelling download for ads with partial download enabled.")
            return
        }

        baseAdLoader?.cancel()
    }

    fun play(context: Context?, adPlayCallback: AdPlayCallback) {
        showToValidationMetric.markStart()
        playContext = if (context != null) {
            WeakReference(context)
        } else {
            null
        }

        val error = canPlayAd(true)
        if (error != null) {
            adPlayCallback.onFailure(error)
            if (isErrorTerminal(error.code)) {
                adState = AdState.ERROR
            }
            return
        }

        val adv = advertisement ?: return

        val callbackWrapper = object : AdPlayCallbackWrapper(adPlayCallback) {
            override fun onAdStart(id: String?) {
                adState = AdState.PLAYING
                validationToPresentMetric.markEnd()
                AnalyticsClient.logMetric(validationToPresentMetric, logEntry)
                super.onAdStart(id)
            }

            override fun onAdEnd(id: String?) {
                adState = AdState.FINISHED
                super.onAdEnd(id)
            }

            override fun onFailure(error: VungleError) {
                adState = AdState.ERROR
                super.onFailure(error)
            }
        }

        cancelDownload()

        renderAd(callbackWrapper, adv)
    }

    internal open fun renderAd(
        listener: AdPlayCallback?,
        advertisement: AdPayload
    ) {
        /// Subscribe to the event bus of the activity before starting the activity, otherwise
        /// the Publisher notices it has no subscribers and does not emit the start value.
        AdActivity.eventListener = object : AdEventListener(
            listener,
            placement
        ) {}

        AdActivity.advertisement = advertisement
        AdActivity.bidPayload = bidPayload

        // Start the activity, and if there are any extras that have been overridden by the application, apply them.
        // Special case that we allow load ad in A activity and play it in B activity.
        val ctx = playContext?.get() ?: context
        val pl = placement ?: return
        val intent = AdActivity.createIntent(ctx, pl.referenceId, advertisement.eventId())

        // log metric if the ad activity is in background on play
        if (!ActivityManager.isForeground()) {
            Logger.d(TAG, "The ad activity is in background on play, log AD_VISIBILITY_INVISIBLE.")
            intent.apply {
                putExtra(AdActivity.AD_INVISIBLE_LOGGED_KEY, true)
            }
            AnalyticsClient.logMetric(
                SingleValueMetric(Sdk.SDKMetric.SDKMetricType.AD_VISIBILITY).also {
                    it.value = AD_VISIBILITY_INVISIBLE
                }, logEntry
            )
        }
        showToValidationMetric.markEnd()
        AnalyticsClient.logMetric(showToValidationMetric, logEntry)
        validationToPresentMetric.markStart()
        ActivityManager.startWhenForeground(ctx, null, intent, null)
    }

    override fun onSuccess(advertisement: AdPayload) {
        this@AdInternal.advertisement = advertisement
        adState = AdState.READY
        adLoadedAndUpdateConfigure(advertisement)
        adLoaderCallback?.onSuccess(advertisement)
        loadMetric?.let { metric ->
            if (!advertisement.adLoadOptimizationEnabled()) {
                metric.metricType = Sdk.SDKMetric.SDKMetricType.AD_LOAD_TO_CALLBACK_DURATION_MS
            }
            metric.markEnd()
            AnalyticsClient.logMetric(metric, logEntry)
        }
        requestMetric?.let { metric ->
            //update the metric type if ADO disabled
            if (!advertisement.adLoadOptimizationEnabled()) {
                metric.metricType = Sdk.SDKMetric.SDKMetricType.AD_REQUEST_TO_CALLBACK_DURATION_MS
            }
            metric.markEnd()
            AnalyticsClient.logMetric(metric, logEntry)

            val tpatSender: TpatSender by inject(context)
            advertisement.getTpatUrls(
                Constants.AD_LOAD_DURATION,
                metric.getValue().toString()
            )?.forEach { url ->
                val request = TpatRequest.Builder(url)
                    .tpatKey(Constants.AD_LOAD_DURATION)
                    .withLogEntry(logEntry)
                    .build()
                tpatSender.sendTpat(request)
            }
        }
    }

    internal open fun adLoadedAndUpdateConfigure(advertisement: AdPayload) {
    }

    override fun onFailure(error: VungleError) {
        adState = AdState.ERROR
        loadMetric?.let { metric ->
            metric.metricType = Sdk.SDKMetric.SDKMetricType.AD_LOAD_TO_FAIL_CALLBACK_DURATION_MS
            metric.markEnd()
            AnalyticsClient.logMetric(metric, logEntry, "${error.code}-${error.errorMessage}")
        }
        adLoaderCallback?.onFailure(error)
    }

    /*
      We use this method to check if we allowed to change ad state to ERROR. Because we use ERROR
    state to clean up the ad assets in the device.
      There's only one case we need to clean up assets which is pub try to play one READY but has
    expired ad.

    |InstanceState | canPlay() | expired | loadAd() | playAd() | playAd()-to-ERROR? |
    |--------------|-----------|---------|----------|----------|--------------------|
    |    NEW       |      10   |     -   |     Y    |    10    |            N       |
    |    LOADING   |      10   |     -   |     42   |    10    |            N       |
    |    READY     |      0    |     N   |     42   |     Y    |            N       |
    |    READY     |      4    |     Y   |     42   |     4    |            Y       |
    |    PLAYING   |    4,46   |    Y/N  |     42   |   4,46   |            N       |
    |    FINISHED  |    4,42   |    Y/N  |     42   |   4,42   |            N       |
    |    ERROR     |   10,4,42 |    Y/N  |     42   |  10,4,42 |            N       |

     */
    internal fun isErrorTerminal(errorCode: Int): Boolean {
        return adState == AdState.READY && errorCode == SDKError.Reason.AD_EXPIRED_VALUE
    }

    enum class AdState {
        NEW {
            override fun canTransitionTo(adState: AdState): Boolean {
                return adState === LOADING || adState === READY || adState === ERROR
            }
        },
        LOADING {
            override fun canTransitionTo(adState: AdState): Boolean {
                return adState === READY || adState === ERROR
            }
        },
        READY {
            override fun canTransitionTo(adState: AdState): Boolean {
                return adState === PLAYING || adState === FINISHED || adState === ERROR
            }
        },
        PLAYING {
            override fun canTransitionTo(adState: AdState): Boolean {
                return adState === FINISHED || adState === ERROR
            }
        },
        FINISHED {
            override fun canTransitionTo(adState: AdState): Boolean {
                return false
            }
        },
        ERROR {
            override fun canTransitionTo(adState: AdState): Boolean {
                return adState === FINISHED
            }
        };

        abstract fun canTransitionTo(adState: AdState): Boolean

        fun transitionTo(adState: AdState): AdState {
            if (this != adState && !canTransitionTo(adState)) {
                val msg = "Cannot transition from ${this.name} to ${adState.name}"
                if (THROW_ON_ILLEGAL_TRANSITION) {
                    throw IllegalStateException(msg)
                } else {
                    Logger.e(TAG, "Illegal state transition", IllegalStateException(msg))
                }
            }
            return adState
        }

        fun isTerminalState(): Boolean {
            return this in listOf(FINISHED, ERROR)
        }
    }
}
