package com.vungle.ads.internal.network

import androidx.annotation.VisibleForTesting
import com.vungle.ads.TpatError
import com.vungle.ads.internal.ConfigManager
import com.vungle.ads.internal.Constants
import com.vungle.ads.internal.Constants.CHECKPOINT_0
import com.vungle.ads.internal.Constants.CLICK_URL
import com.vungle.ads.internal.Constants.IMPRESSION
import com.vungle.ads.internal.Constants.LOAD_AD
import com.vungle.ads.internal.model.ErrorInfo
import com.vungle.ads.internal.persistence.FilePreferences
import com.vungle.ads.internal.persistence.FilePreferences.Companion.TPAT_FAILED_FILENAME
import com.vungle.ads.internal.protos.Sdk.SDKError
import com.vungle.ads.internal.signals.SignalManager
import com.vungle.ads.internal.util.Logger
import com.vungle.ads.internal.util.PathProvider
import kotlinx.serialization.Serializable
import kotlinx.serialization.decodeFromString
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import java.util.concurrent.Executor
import java.util.regex.Pattern

@Serializable
enum class HttpMethod {
    GET, POST
}

@Serializable
data class FailedTpat(
    val method: HttpMethod = HttpMethod.GET,
    val headers: Map<String, String>? = null,
    val body: String? = null,
    var retryAttempt: Int = 0,
    var retryCount: Int,
    var tpatKey: String? = null,
)

class TpatSender(
    val vungleApiClient: VungleApiClient,
    ioExecutor: Executor,
    val jobExecutor: Executor,
    pathProvider: PathProvider,
    val signalManager: SignalManager? = null
) {
    companion object {
        private const val TAG: String = "TpatSender"
        private const val FAILED_TPATS = "FAILED_TPATS"
    }

    private val tpatFilePreferences =
        FilePreferences.get(ioExecutor, pathProvider, TPAT_FAILED_FILENAME)

    private val tpatLock = Any()

    /**
     * Sends a TPAT request to the server.
     *
     * @param request The `TpatRequest` object containing the details of the TPAT request.
     * @param fromFailedTpat Indicates if this is a resend attempt. Default is false.
     */
    fun sendTpat(request: TpatRequest, fromFailedTpat: Boolean = false) {
        val urlWithSessionId = injectSessionIdToUrl(request.url)
        jobExecutor.execute {
            val error = performPriorityRetry(request, urlWithSessionId)

            // Return if regular retry is disabled
            if (!request.regularRetry) return@execute

            // Return if the error is terminal, no need to later retry
            if (error?.errorIsTerminal == true) return@execute

            // Return if the error is null and not from a stored failed tpat.
            // Serializable performance enhancement: No need to get/update failed tpats.
            if (error == null && !fromFailedTpat) return@execute

            synchronized(tpatLock) {
                val storedTpats = getStoredTpats()
                val regularRetryAttempt = storedTpats[request.url]?.retryAttempt ?: 0
                when {
                    error == null && regularRetryAttempt > 0 -> {
                        // If the request is successful after retries, remove it from shared preferences
                        storedTpats.remove(request.url)
                        saveStoredTpats(storedTpats)
                    }

                    error != null && regularRetryAttempt >= request.regularRetryCount -> {
                        storedTpats.remove(request.url)
                        saveStoredTpats(storedTpats)
                        logTpatError(
                            request,
                            urlWithSessionId,
                            error,
                            SDKError.Reason.TPAT_RETRY_FAILED
                        )
                    }

                    error != null -> {
                        val updated =
                            storedTpats[request.url]?.copy(retryAttempt = regularRetryAttempt + 1)
                                ?: FailedTpat(
                                    method = request.method,
                                    headers = request.headers,
                                    body = request.body,
                                    retryAttempt = 1,
                                    retryCount = request.regularRetryCount,
                                    tpatKey = request.tpatKey
                                )
                        storedTpats[request.url] = updated
                        saveStoredTpats(storedTpats)
                    }
                }
            }
        }
    }

    private fun isPriorityTpat(event: String?): Boolean {
        return event == CHECKPOINT_0 || event == CLICK_URL || event == IMPRESSION
                || event == LOAD_AD
    }

    private fun performPriorityRetry(request: TpatRequest, url: String): ErrorInfo? {
        val priorityTpat = request.priorityRetry ?: isPriorityTpat(request.tpatKey)
        val priorityRetryEnabled = ConfigManager.retryPriorityTPATs() && priorityTpat
        var attempt = 0
        var error: ErrorInfo?

        do {
            error = vungleApiClient.pingTPAT(url, request.headers, request.body, request.method, request.logEntry)
        } while (priorityRetryEnabled && error?.isRetryCode == true && ++attempt < request.priorityRetryCount)

        if (error != null) {
            val reason = if (attempt >= request.priorityRetryCount) {
                SDKError.Reason.TPAT_RETRY_FAILED
            } else {
                SDKError.Reason.TPAT_ERROR
            }
            logTpatError(request, url, error, reason)
        }

        return error
    }

    private fun logTpatError(request: TpatRequest, url: String, error: ErrorInfo, reason: SDKError.Reason) {
        val msg = "tpat key: ${request.tpatKey}, error: ${error.description}, " +
                "errorIsTerminal: ${error.errorIsTerminal} url: $url"
        Logger.e(TAG, msg)
        TpatError(reason, msg).setLogEntry(request.logEntry).logErrorNoReturnValue()
    }

    /**
     * Retrieves a map of TPAT failures stored in shared preferences.
     *
     * @return A `Map` where the key is the TPAT event and the value is the `FailedTpat` object.
     * If no stored TPAT failures are found or if there's an error in decoding the stored TPAT failures,
     * an empty `Map` is returned.
     */
    private fun getStoredTpats(): MutableMap<String, FailedTpat> {
        val storedTpats = tpatFilePreferences.getString(FAILED_TPATS)
        return storedTpats?.let {
            runCatching { Json.decodeFromString<MutableMap<String, FailedTpat>>(it) }.onFailure {
                Logger.e(TAG, "Failed to decode stored tpats: $it")
            }.getOrElse { mutableMapOf() }
        } ?: mutableMapOf()
    }

    /**
     * Stores a map of TPAT failures into shared preferences.
     *
     * @param tpats A `Map` where the key is the TPAT event and the value is the `FailedTpat` object.
     * If there's an error in encoding the TPAT failures for storage, an error message is logged.
     */
    private fun saveStoredTpats(tpats: MutableMap<String, FailedTpat>) {
        runCatching {
            tpatFilePreferences
                .put(FAILED_TPATS, Json.encodeToString(tpats))
                .apply()
        }.onFailure {
            Logger.e(TAG, "Failed to encode the about to storing tpats: $tpats")
        }
    }

    internal fun resendStoredTpats() {
        getStoredTpats().forEach { (url, failedTpat) ->
            val tpatRequest = TpatRequest.Builder(url)
                .regularRetry(true)
                .priorityRetry(false)
                .headers(failedTpat.headers)
                .body(failedTpat.body)
                .regularRetryCount(failedTpat.retryCount)
                .method(failedTpat.method)
                .tpatKey(failedTpat.tpatKey)
                .build()
            sendTpat(tpatRequest, true)
        }
    }

    @VisibleForTesting
    fun injectSessionIdToUrl(url: String): String {
        val sessionId = signalManager?.uuid.orEmpty()
        return if (sessionId.isNotEmpty()) {
            url.replace(
                Pattern.quote(Constants.SESSION_ID).toRegex(), sessionId
            )
        } else {
            url
        }
    }
}
