package com.vungle.ads.internal.omsdk

import android.content.Context
import android.os.Handler
import android.os.Looper
import androidx.annotation.RawRes
import androidx.annotation.WorkerThread
import com.iab.omid.library.vungle.Omid
import com.vungle.ads.R
import com.vungle.ads.internal.Constants.OM_SDK_JS
import com.vungle.ads.internal.Constants.OM_SESSION_JS
import com.vungle.ads.internal.util.Logger
import java.io.File
import java.io.FileWriter
import java.io.IOException
import java.util.concurrent.atomic.AtomicReference

class OMInjector(val context: Context) {

    private val uiHandler = Handler(Looper.getMainLooper())

    fun init() {
        uiHandler.post {
            runCatching {
                if (!Omid.isActive()) {
                    Omid.activate(context)
                }
            }.onFailure { error ->
                Logger.e("OMSDK", "error: ${error.localizedMessage}")
            }
        }
    }

    private var cachedOMSDKJS = AtomicReference<String>(null)
    private var cachedOMSessionJS = AtomicReference<String>(null)

    @WorkerThread
    internal fun injectJsFiles(dir: File): List<File> {
        val ret = mutableListOf<File>()
        val omSdkFile = File(dir, OM_SDK_JS)
        if (!omSdkFile.exists()) {
            Logger.d(TAG, "injecting omsdk.js to ${omSdkFile.absolutePath}")
            getOMSDKJS()?.let { writeToFile(it, omSdkFile) }
            ret.add(omSdkFile)
        }
        val omSessionFile = File(dir, OM_SESSION_JS)
        if (!omSessionFile.exists()) {
            Logger.d(TAG, "injecting omsdk-session.js to ${omSessionFile.absolutePath}")
            getOMSessionJS()?.let { writeToFile(it, omSessionFile) }
            ret.add(omSessionFile)
        }

        return ret
    }

    internal fun getOMSDKJS(): String? {
        return cachedOMSDKJS.get() ?: loadJsFromRaw(R.raw.omsdk_v1_5_7).also {
            cachedOMSDKJS.set(it)
        }
    }

    private fun getOMSessionJS(): String? {
        return cachedOMSessionJS.get() ?: loadJsFromRaw(R.raw.omid_session_client_v1_5_7).also {
            cachedOMSessionJS.set(it)
        }
    }

    private fun loadJsFromRaw(@RawRes resId: Int): String? {
        return runCatching {
            context.resources.openRawResource(resId).bufferedReader().use { it.readText() }
        }.getOrNull()
    }

    private fun writeToFile(lines: String?, outputFile: File): File {
        if (lines == null) throw IOException("omsdk js must not be null")
        return FileWriter(outputFile).use { writer ->
            writer.write(lines)
            writer.flush()
            outputFile
        }
    }

    companion object {
        private const val TAG = "OMInjector"
    }

}
