package com.vungle.ads.internal.platform

import android.content.Context
import android.content.pm.PackageManager
import android.hardware.Sensor
import android.hardware.SensorManager
import android.net.ConnectivityManager
import android.net.NetworkCapabilities
import android.net.ProxyInfo
import android.net.Uri
import android.os.Build
import android.provider.Settings

object DeviceCheckUtils {

    fun isEmulator(): Boolean {
        val fingerprint = Build.FINGERPRINT?.lowercase() ?: ""
        val model = Build.MODEL?.lowercase() ?: ""
        val product = Build.PRODUCT?.lowercase() ?: ""
        val brand = Build.BRAND?.lowercase() ?: ""
        val manufacturer = Build.MANUFACTURER?.lowercase() ?: ""
        val board = Build.BOARD?.lowercase() ?: ""
        val cpuAbi = Build.SUPPORTED_ABIS.joinToString().lowercase()

        return (fingerprint.contains("generic") ||
                fingerprint.contains("emulator") ||
                model.contains("google_sdk") ||
                model.contains("emulator") ||
                product.contains("sdk") ||
                product.contains("vbox") ||
                brand.contains("generic") ||
                manufacturer.contains("genymotion") ||
                board.contains("nox") ||
                cpuAbi.contains("x86"))
    }

    fun isVpnConnected(context: Context): Boolean {
        return runCatching {
            val cm = context.getSystemService(Context.CONNECTIVITY_SERVICE) as ConnectivityManager
            cm.allNetworks.any { network ->
                cm.getNetworkCapabilities(network)
                    ?.hasTransport(NetworkCapabilities.TRANSPORT_VPN) == true
            }
        }.getOrDefault(false)
    }

    fun hasSystemAlertWindowPermission(context: Context): Boolean {
        return if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            Settings.canDrawOverlays(context)
        } else {
            context.checkCallingOrSelfPermission(
                android.Manifest.permission.SYSTEM_ALERT_WINDOW
            ) == PackageManager.PERMISSION_GRANTED
        }
    }

    fun getSensorCount(context: Context): Int {
        val sensorManager = context.getSystemService(Context.SENSOR_SERVICE) as SensorManager
        return sensorManager.getSensorList(Sensor.TYPE_ALL).size
    }

    private fun ProxyInfo?.isEnabled(): Boolean {
        if (this == null) return false
        val hasHostPort = host?.isNotEmpty() == true && port > 0
        val hasPac = pacFileUrl != null && pacFileUrl != Uri.EMPTY
        return hasHostPort || hasPac
    }

    /**
     * Checks if there is any active HTTP proxy configured (system or Wi-Fi).
     */
    fun isProxyEnabled(context: Context): Boolean {
        return runCatching {
            val cm = context.getSystemService(Context.CONNECTIVITY_SERVICE) as ConnectivityManager
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                val net = cm.activeNetwork ?: return@runCatching false
                cm.getLinkProperties(net)?.httpProxy.isEnabled()
            } else {
                cm.allNetworks.any { n ->
                    cm.getNetworkInfo(n)?.isConnected == true &&
                            cm.getLinkProperties(n)?.httpProxy.isEnabled()
                }
            }
        }.getOrDefault(false)
    }

}
