package com.vungle.ads.internal.task

import android.content.Context
import android.os.Build
import android.os.Bundle
import com.vungle.ads.internal.util.Logger
import com.vungle.ads.BuildConfig
import com.vungle.ads.ServiceLocator
import com.vungle.ads.internal.persistence.FilePreferences
import com.vungle.ads.internal.util.FileUtility.delete
import com.vungle.ads.internal.util.FileUtility.printDirectoryTree
import com.vungle.ads.internal.util.PathProvider
import java.io.File
import java.io.IOException
import androidx.core.content.edit
import com.vungle.ads.internal.Constants.OM_SDK_JS
import com.vungle.ads.internal.Constants.OM_SESSION_JS
import com.vungle.ads.internal.util.FileUtility

private const val VERSION_CODE_KEY = "VERSION_CODE"

private const val NO_VALUE = -1

/**
 * Job that will remove any stale or expired data.
 */
class CleanupJob
internal constructor(val context: Context, val pathProvider: PathProvider) : Job {

    override fun onRunJob(bundle: Bundle, jobRunner: JobRunner): Int {
        val downloadsDir = pathProvider.getVmDir()
        var dirToDelete = downloadsDir
        bundle.getString(AD_ID_KEY)?.also { adId ->
            pathProvider.getDownloadsDirForAd(adId)?.also {
                dirToDelete = it
            }
        }
        Logger.d(TAG, "CleanupJob: Current directory snapshot")
        if (BuildConfig.DEBUG) {
            printDirectoryTree(dirToDelete)
        }
        try {
            if (dirToDelete == downloadsDir) {
                checkIfSdkUpgraded()
                val excludeFiles = setOf(OM_SDK_JS, OM_SESSION_JS)
                FileUtility.deleteContents(dirToDelete, excludeFiles)
            } else {
                delete(dirToDelete)
            }
        } catch (_: IOException) {
            return Job.Result.FAILURE
        }
        return Job.Result.SUCCESS
    }

    private fun dropV6Data() {
        Logger.d(TAG, "CleanupJob: drop old files data")

        // DB
        val oldDb = File(context.noBackupFilesDir, "vungle_db")
        if (oldDb.exists()) {
            delete(oldDb)
            delete(File(oldDb.path + "-journal"))
        } else {
            context.deleteDatabase("vungle_db")
        }

        // SP
        val sp = context.getSharedPreferences("com.vungle.sdk", Context.MODE_PRIVATE)
        val vungleCachePath = sp.getString("cache_path", null)
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            context.deleteSharedPreferences("com.vungle.sdk")
        } else {
            sp.edit { clear() }
        }

        // vungleSetting
        val dir: File = context.noBackupFilesDir
        val vungleSettings = File(dir, "vungle_settings")
        delete(vungleSettings)

        // vungle_cache
        vungleCachePath?.let {
            delete(File(it))
        }
    }

    private fun dropV700Data() {
        val vungleDirV700 = File(
            context.applicationInfo.dataDir,
            "vungle"
        )
        delete(vungleDirV700)
    }

    private fun dropV730TempData() {
        try {
            delete(File(pathProvider.getSharedPrefsDir(), "vungleSettings"))
            delete(File(pathProvider.getSharedPrefsDir(), "failedTpatSet"))
        } catch (e: Exception) {
            Logger.e(TAG, "Failed to delete temp data", e)
        }
    }

    private fun dropV742TpatData() {
        // We should use hardcode sharedPrefsDir instead of calling getSharedPrefsDir() function
        // because we need to delete the old data and the dir or filename could changed in the new version.
        val sharedPrefsDir = context.noBackupFilesDir
        try {
            delete(File(sharedPrefsDir, "failedTpats"))
            delete(File(sharedPrefsDir, "failedGenericTpats"))
        } catch (e: Exception) {
            Logger.e(TAG, "Failed to delete 742 tpat data", e)
        }
    }

    private fun dropV750Data() {
        val sharedPrefsDir = context.noBackupFilesDir
        try {
            delete(File(sharedPrefsDir, "vungle_cache/downloads"))
            delete(File(sharedPrefsDir, "vungle_cache/js"))
        } catch (e: Exception) {
            Logger.e(TAG, "Failed to delete 750 data", e)
        }
    }

    private fun checkIfSdkUpgraded() {
        val filePreferences: FilePreferences by ServiceLocator.inject(context)
        val lastInstalledVersion = filePreferences.getInt(VERSION_CODE_KEY, NO_VALUE)
        if (lastInstalledVersion < BuildConfig.VERSION_CODE) {
            if (lastInstalledVersion < 70000) {
                dropV6Data()
            }
            if (lastInstalledVersion < 70100) {
                dropV700Data()
            }
            if (lastInstalledVersion < 70301) {
                dropV730TempData()
            }
            if (lastInstalledVersion < 70500) {
                dropV742TpatData()
            }
            if (lastInstalledVersion < 70600) {
                dropV750Data()
            }
            filePreferences.put(VERSION_CODE_KEY, BuildConfig.VERSION_CODE).apply()
        }
    }

    companion object {
        const val TAG: String = "CleanupJob"
        private const val AD_ID_KEY = "AD_ID_KEY"

        fun makeJobInfo(adId: String? = null): JobInfo {
            return JobInfo(TAG)
                .setPriority(JobInfo.Priority.LOWEST)
                .setExtras(Bundle().apply {
                    adId?.let { putString(AD_ID_KEY, it) }
                })
                .setUpdateCurrent(adId == null) /*only update "erase everything" type of job*/
        }
    }
}
