package com.vungle.ads.internal.ui

import android.os.Build
import android.os.Handler
import android.os.Looper
import android.view.View
import android.webkit.*
import androidx.annotation.RequiresApi
import androidx.annotation.VisibleForTesting
import com.vungle.ads.AnalyticsClient
import com.vungle.ads.BuildConfig
import com.vungle.ads.EvaluateJsError
import com.vungle.ads.OutOfMemory
import com.vungle.ads.internal.model.AdPayload
import com.vungle.ads.internal.model.Placement
import com.vungle.ads.internal.omsdk.WebViewObserver
import com.vungle.ads.internal.platform.Platform
import com.vungle.ads.internal.presenter.PreloadDelegate
import com.vungle.ads.internal.protos.Sdk
import com.vungle.ads.internal.ui.view.WebViewAPI
import com.vungle.ads.internal.util.Logger
import kotlinx.serialization.json.buildJsonObject
import kotlinx.serialization.json.put
import java.io.BufferedInputStream
import java.io.File
import java.io.FileInputStream
import java.util.Locale
import java.util.concurrent.ExecutorService
import androidx.core.net.toUri
import com.vungle.ads.OneShotSingleValueMetric
import com.vungle.ads.SingleValueMetric
import kotlinx.serialization.json.JsonObject

class VungleWebClient(
    private val advertisement: AdPayload,
    private val placement: Placement,
    private val offloadExecutor: ExecutorService,
    private val platform: Platform? = null,
    private val delegate: PreloadDelegate? = null,
    private val loadDuration: Long? = null,

) : WebViewClient(), WebViewAPI {

    companion object {
        private const val TAG = "VungleWebClient"
        private const val COMMAND_COMPLETE = "window.vungle.mraidBridge.notifyCommandComplete()"
    }

    @VisibleForTesting internal var collectConsent = false
    @VisibleForTesting internal var gdprTitle: String? = null
    @VisibleForTesting internal var gdprBody: String? = null
    @VisibleForTesting internal var gdprAccept: String? = null
    @VisibleForTesting internal var gdprDeny: String? = null

    @VisibleForTesting internal var loadedWebView: WebView? = null
    @VisibleForTesting internal var ready = false

    @VisibleForTesting internal var mraidDelegate: WebViewAPI.MraidDelegate? = null
    @VisibleForTesting internal var errorHandler: WebViewAPI.WebClientErrorHandler? = null
    internal val handler = Handler(Looper.getMainLooper())

    @VisibleForTesting internal var webViewObserver: WebViewObserver? = null
    @VisibleForTesting internal var isViewable: Boolean? = null

    private val partialDownloadMetric =
        OneShotSingleValueMetric(Sdk.SDKMetric.SDKMetricType.AD_PLAY_WITH_PARTIAL_DOWNLOAD_ASSET)
    private val partialDownloadErrorMetric =
        OneShotSingleValueMetric(Sdk.SDKMetric.SDKMetricType.AD_PLAY_WITH_PARTIAL_DOWNLOAD_ASSET)
    private val skippedUrlMetric =
        SingleValueMetric(Sdk.SDKMetric.SDKMetricType.BANNER_AUTO_REDIRECT_NOT_OVERRIDE_URL)

    override fun setConsentStatus(
        collectedConsent: Boolean, title: String?, message: String?,
        accept: String?, deny: String?
    ) {
        this.collectConsent = collectedConsent
        gdprTitle = title
        gdprBody = message
        gdprAccept = accept
        gdprDeny = deny
    }

    override fun setMraidDelegate(mraidDelegate: WebViewAPI.MraidDelegate?) {
        this.mraidDelegate = mraidDelegate
    }

    override fun shouldInterceptRequest(
        view: WebView?,
        request: WebResourceRequest?
    ): WebResourceResponse? {
        val uri = request?.url ?: return null
        val scheme = uri.scheme?.lowercase(Locale.ROOT) ?: return null
        if (scheme != "http" && scheme != "https") {
            return null
        }

        if (!advertisement.isPartialDownloadEnabled()) {
            Logger.w(TAG, "shouldInterceptRequest called but partial download is disabled.")
            return null
        }

        val url = uri.toString()
        val asset = advertisement.getLocalPartialDownloadAssets(url)
        val localFilePath = asset?.localPath
        if (localFilePath.isNullOrEmpty()) return null
        val localFile = File(localFilePath)
        if (!localFile.exists()) return null
        val contentLength = asset.contentLength
        if (contentLength <= 0) return null

        var cachedFileLength = localFile.length()
        val rangeHeader = request.requestHeaders["Range"]

        partialDownloadMetric.meta = "$rangeHeader cached:$cachedFileLength $url"
        AnalyticsClient.logMetric(partialDownloadMetric, advertisement.logEntry)

        val (rangeStart, rangeEnd) = parseRange(rangeHeader, contentLength).also {
            asset.rangeStart = it.first
            asset.rangeEnd = it.second
        }
        val availableBytes = cachedFileLength - rangeStart
        Logger.i(
            TAG,
            ">>request: $rangeHeader rangeStart=$rangeStart rangeEnd=$rangeEnd cachedFileLength=$cachedFileLength availableBytes=$availableBytes contentLength=$contentLength "
        )
        if (availableBytes <= 0) {
            Logger.w(TAG, "Requested range exceeds cached file: $rangeHeader")

            asset.waitForDownload()

            // Refresh the cached file length after waiting for download
            cachedFileLength = localFile.length()
        }

        val availableRangeEnd = rangeEnd ?: (cachedFileLength - 1)
        val availableContentLength = availableRangeEnd - rangeStart + 1

        runCatching {
            val inputStream = FileInputStream(localFile)
            val response = WebResourceResponse(
                "video/mp4", "UTF-8", 206, "Partial Content",
                mapOf(
                    "Content-Type" to "video/mp4",
                    "Accept-Ranges" to "bytes",
                    "Content-Length" to availableContentLength.toString(),
                    "Content-Range" to "bytes $rangeStart-$availableRangeEnd/$contentLength"
                ),
                BufferedInputStream(inputStream, 1024)
            )

            Logger.i(TAG, "<<Return:${response.responseHeaders}")
            return response
        }.onFailure { e ->
            Logger.e(TAG, "Error serving local range video: ${e.message}", e)
            partialDownloadErrorMetric.meta = "$url ${e.message}"
            AnalyticsClient.logMetric(partialDownloadErrorMetric, advertisement.logEntry)
        }

        return null
    }

    /**
     * Do not support multipart byte ranges request.
     * Parses the Range header to extract the start and end byte positions.
     * Returns a pair of Longs, where the first is the start position and the second is the end position (nullable).
     * If the header is malformed or not present, returns (0L, null).
     *
     * https://www.rfc-editor.org/rfc/rfc9110#section-14.1.2-8
     * Example of valid Range headers, assuming a file with length of 10000 bytes:
     * "bytes=100-499": Pair(100L, 499L)
     * "bytes=500-": Pair(500L, null)
     * "bytes=-500": Pair(9500L, null)
     */
    internal fun parseRange(rangeHeader: String?, contentLength: Long): Pair<Long, Long?> {
        return runCatching {
            if (rangeHeader?.startsWith("bytes=") == true) {
                val rangePart = rangeHeader.removePrefix("bytes=")
                val parts = rangePart.split("-")
                var start = parts.getOrNull(0)?.toLongOrNull()
                var end = parts.getOrNull(1)?.toLongOrNull()

                if (start == null) {
                    if (end == null) {
                        start = 0
                    } else {
                        start = contentLength - end
                        end = null
                    }
                }

                Pair(start, end)
            } else {
                Pair(0L, null)
            }
        }.getOrElse {
            Pair(0L, null)
        }
    }

    @Deprecated("Deprecated in Java")
    override fun shouldOverrideUrlLoading(view: WebView?, url: String?): Boolean {
        try {
            Logger.d(TAG, "MRAID Command $url")

            if (url.isNullOrEmpty()) {
                Logger.e(TAG, "Invalid URL ")
                return false
            }

            val uri = url.toUri()
            if (uri.scheme.isNullOrEmpty()) {
                return false
            }

            when (val scheme = uri.scheme) {
                "mraid" -> {
                    uri.host?.let { command ->
                        when (command) {
                            "propertiesChangeCompleted" -> {
                                if (!ready) {
                                    ready = true
                                    offloadExecutor.submit {
                                        /// Pass the URL for the assets to the webview. This can be handled by the web client.
                                        val mraidArgs: JsonObject = advertisement.createMRAIDArgs()
                                        val injectJs =
                                            "window.vungle.mraidBridge.notifyReadyEvent($mraidArgs)"
                                        handler.post {
                                            runJavascriptOnWebView(view, injectJs)
                                        }
                                    }
                                }
                            }
                            "readyToPlay" -> {
                                offloadExecutor.submit {
                                    delegate?.onAdReadyToPlay()
                                    handler.post {
                                        runJavascriptOnWebView(view, COMMAND_COMPLETE)
                                    }
                                }

                            }
                            "failToLoad" -> {
                                offloadExecutor.submit {
                                    delegate?.onAdFailedToPlay()
                                    handler.post {
                                        runJavascriptOnWebView(view, COMMAND_COMPLETE)
                                    }
                                }
                            }
                            else -> {
                                if (mraidDelegate == null) {
                                    handler.post {
                                        runJavascriptOnWebView(view, COMMAND_COMPLETE)
                                    }
                                    return true
                                }
                                mraidDelegate?.let {
                                    val args = buildJsonObject {
                                        for (param in uri.queryParameterNames) {
                                            put(param, uri.getQueryParameter(param))
                                        }
                                    }

                                    offloadExecutor.submit {
                                        if (it.processCommand(command, args)) {
                                            handler.post {
                                                runJavascriptOnWebView(view, COMMAND_COMPLETE)
                                            }
                                        }
                                    }
                                }
                            }
                        }
                        return true
                    }
                }

                else -> {
                    if ("http".equals(scheme, ignoreCase = true) ||
                        "https".equals(scheme, ignoreCase = true)
                    ) {
                        Logger.d(TAG, "Open URL$url")
                        mraidDelegate?.let {
                            val args = buildJsonObject {
                                put("url", url)
                            }
                            it.processCommand("openNonMraid", args)
                        }
                        return true
                    }
                }
            }

            skippedUrlMetric.meta = "url: $url"
            AnalyticsClient.logMetric(skippedUrlMetric, advertisement.logEntry)
            Logger.w(TAG, "skipped url: $url")

            return false
        } catch (e: Throwable) {
            if (e is OutOfMemoryError) {
                OutOfMemory("mraid:$url").logErrorNoReturnValue()
            }
            return false
        }
    }

    override fun onPageFinished(view: WebView?, url: String?) {
        super.onPageFinished(view, url)
        Logger.w("VungleWebClient", "onPageFinished.")

        if (view == null) {
            return
        }

        loadedWebView = view
        loadedWebView?.visibility = View.VISIBLE
        notifyPropertiesChange(true)

        /* The Banner and MRAID both uses the webview. let us catch the issue with both of them */
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
            view.webViewRenderProcessClient = VungleWebViewRenderProcessClient(errorHandler)
        }

        webViewObserver?.onPageFinished(view)
    }

    override fun notifyPropertiesChange(skipCmdQueue: Boolean) {
        offloadExecutor.submit {
            val screenJson = buildJsonObject {
                put("placementType", advertisement.templateType())
                isViewable?.let { visible -> put("isViewable", visible) }
                put("os", "android")
                put("osVersion", Build.VERSION.SDK_INT.toString())
                put("incentivized", placement.isRewardedVideo())
                platform?.let { put("isSilent", platform.isSilentModeEnabled) }
                loadDuration?.let { put("timeLoaded", loadDuration) }
                if (collectConsent) {
                    put("consentRequired", true)
                    put("consentTitleText", gdprTitle)
                    put("consentBodyText", gdprBody)
                    put("consentAcceptButtonText", gdprAccept)
                    put("consentDenyButtonText", gdprDeny)
                } else {
                    put("consentRequired", false)
                }

                put("sdkVersion", BuildConfig.VERSION_NAME)
            }

            val injectJs =
                "window.vungle.mraidBridge.notifyPropertiesChange($screenJson,$skipCmdQueue)"
            handler.post {
                loadedWebView?.let { runJavascriptOnWebView(it, injectJs) }
            }
        }
    }

    fun notifyDiskAvailableSize(availableDiskSize: Long, appFolderSize: Long) {
        loadedWebView?.let {
            val injectJs = "window.vungle.mraidBridgeExt.notifyAvailableDiskSpace(${availableDiskSize}-$appFolderSize)"
            runJavascriptOnWebView(it, injectJs)
        }
    }

    fun notifySilentModeChange(silentModeEnabled: Boolean) {
        loadedWebView?.let {
            val silentModeJson = buildJsonObject {
                put("isSilent", silentModeEnabled)
            }

            val injectJs =
                "window.vungle.mraidBridge.notifyPropertiesChange($silentModeJson)"
            runJavascriptOnWebView(it, injectJs)
        }
    }

    override fun setAdVisibility(isViewable: Boolean) {
        this.isViewable = isViewable
        notifyPropertiesChange(false)
    }

    override fun setErrorHandler(errorHandler: WebViewAPI.WebClientErrorHandler) {
        this.errorHandler = errorHandler
    }

    override fun setWebViewObserver(webViewObserver: WebViewObserver?) {
        this.webViewObserver = webViewObserver
    }

    @Deprecated("Deprecated in Java")
    override fun onReceivedError(
        view: WebView?,
        errorCode: Int,
        description: String,
        failingUrl: String
    ) {
        super.onReceivedError(view, errorCode, description, failingUrl)
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.M) {
            val criticalAsset = isCriticalAsset(failingUrl)
            Logger.e(TAG, "Error desc $description for URL $failingUrl")

            handleWebViewError(description, failingUrl, criticalAsset)
        }
    }

    override fun onReceivedHttpError(
        view: WebView?,
        request: WebResourceRequest?,
        errorResponse: WebResourceResponse?
    ) {
        super.onReceivedHttpError(view, request, errorResponse)
        val statusCode = errorResponse?.statusCode.toString()
        val url = request?.url.toString()
        val isMainFrame = request?.isForMainFrame == true
        Logger.e(TAG, "Http Error desc $statusCode $isMainFrame for URL $url")

        val criticalAsset = isCriticalAsset(url)
        val crash = criticalAsset && isMainFrame
        handleWebViewError(statusCode, url, crash)
    }

    override fun onReceivedError(
        view: WebView?,
        request: WebResourceRequest?,
        error: WebResourceError?
    ) {
        super.onReceivedError(view, request, error)
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            val description = error?.description.toString()
            val url = request?.url.toString()
            val isMainFrame = request?.isForMainFrame == true
            Logger.e(TAG, "Error desc $description $isMainFrame for URL $url")

            val criticalAsset = isCriticalAsset(url)
            val crash = criticalAsset && isMainFrame
            handleWebViewError(description, url, crash)
        }
    }

    override fun onRenderProcessGone(view: WebView?, detail: RenderProcessGoneDetail?): Boolean {
        //Invalidate the local webview reference as the same is used with the notifyPropertiesChange()
        loadedWebView = null

        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.O) {
            Logger.w(TAG, "onRenderProcessGone url: " + view?.url)
            return errorHandler?.onWebRenderingProcessGone(view, true) ?: true
        }

        Logger.w(TAG, "onRenderProcessGone url: " + view?.url + ", did crash: " + detail?.didCrash())

        return errorHandler?.onWebRenderingProcessGone(
            view, detail?.didCrash()
        ) ?: super.onRenderProcessGone(view, detail)
    }

    private fun handleWebViewError(errorMsg: String, url: String, didCrash: Boolean) {
        val errorDesc = "$url $errorMsg"

        //Notify error
        errorHandler?.onReceivedError(errorDesc, didCrash)
    }

    private fun isCriticalAsset(url: String): Boolean {
        // if this url belongs to cacheable_replacement, we will think it's a critical asset.
        if (url.isNotEmpty()) {
            return advertisement.isCriticalAsset(url)
        }
        return false
    }

    private fun runJavascriptOnWebView(webView: WebView?, injectJs: String) {
        try {
            Logger.w(TAG, "mraid Injecting JS $injectJs")
            webView?.evaluateJavascript(injectJs, null)
        } catch (t : Throwable) {
            EvaluateJsError("Evaluate js failed ${t.localizedMessage}")
                .setLogEntry(advertisement.logEntry).logErrorNoReturnValue()
        }
    }

    @RequiresApi(Build.VERSION_CODES.Q)
    internal class VungleWebViewRenderProcessClient(private var errorHandler: WebViewAPI.WebClientErrorHandler?) :
        WebViewRenderProcessClient() {
        override fun onRenderProcessUnresponsive(
            webView: WebView,
            webViewRenderProcess: WebViewRenderProcess?
        ) {
            Logger.w(TAG,
                "onRenderProcessUnresponsive(Title = " + webView.title + ", URL = " +
                        webView.originalUrl + ", (webViewRenderProcess != null) = " + (webViewRenderProcess != null)
            )

            errorHandler?.onRenderProcessUnresponsive(webView, webViewRenderProcess)
        }

        override fun onRenderProcessResponsive(
            webView: WebView,
            webViewRenderProcess: WebViewRenderProcess?
        ) {
            /* DO NOTHING */
        }
    }
}
