/*
 * Decompiled with CFR 0.152.
 */
package com.wavefront.agent.listeners.otlp;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Lists;
import com.wavefront.agent.handlers.ReportableEntityHandler;
import com.wavefront.agent.listeners.otlp.OtlpTraceUtils;
import com.wavefront.agent.preprocessor.ReportableEntityPreprocessor;
import com.wavefront.common.MetricConstants;
import com.wavefront.sdk.common.Pair;
import com.wavefront.sdk.entities.histograms.HistogramGranularity;
import io.opentelemetry.proto.collector.metrics.v1.ExportMetricsServiceRequest;
import io.opentelemetry.proto.common.v1.AnyValue;
import io.opentelemetry.proto.common.v1.KeyValue;
import io.opentelemetry.proto.metrics.v1.AggregationTemporality;
import io.opentelemetry.proto.metrics.v1.ExponentialHistogram;
import io.opentelemetry.proto.metrics.v1.ExponentialHistogramDataPoint;
import io.opentelemetry.proto.metrics.v1.Gauge;
import io.opentelemetry.proto.metrics.v1.HistogramDataPoint;
import io.opentelemetry.proto.metrics.v1.Metric;
import io.opentelemetry.proto.metrics.v1.NumberDataPoint;
import io.opentelemetry.proto.metrics.v1.ResourceMetrics;
import io.opentelemetry.proto.metrics.v1.ScopeMetrics;
import io.opentelemetry.proto.metrics.v1.Sum;
import io.opentelemetry.proto.metrics.v1.Summary;
import io.opentelemetry.proto.metrics.v1.SummaryDataPoint;
import io.opentelemetry.proto.resource.v1.Resource;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.annotation.Nullable;
import org.jetbrains.annotations.NotNull;
import wavefront.report.Annotation;
import wavefront.report.Histogram;
import wavefront.report.HistogramType;
import wavefront.report.ReportPoint;

public class OtlpMetricsUtils {
    public static final Logger OTLP_DATA_LOGGER = Logger.getLogger("OTLPDataLogger");
    public static final int MILLIS_IN_MINUTE = 60000;
    public static final int MILLIS_IN_HOUR = 3600000;
    public static final int MILLIS_IN_DAY = 86400000;

    public static void exportToWavefront(ExportMetricsServiceRequest request, ReportableEntityHandler<ReportPoint, String> pointHandler, ReportableEntityHandler<ReportPoint, String> histogramHandler, @Nullable Supplier<ReportableEntityPreprocessor> preprocessorSupplier, String defaultSource, boolean includeResourceAttrsForMetrics, boolean includeOtlpAppTagsOnMetrics) {
        ReportableEntityPreprocessor preprocessor = null;
        if (preprocessorSupplier != null) {
            preprocessor = preprocessorSupplier.get();
        }
        for (ReportPoint point : OtlpMetricsUtils.fromOtlpRequest(request, preprocessor, defaultSource, includeResourceAttrsForMetrics, includeOtlpAppTagsOnMetrics)) {
            if (point.getValue() instanceof Histogram) {
                if (OtlpMetricsUtils.wasFilteredByPreprocessor(point, histogramHandler, preprocessor)) continue;
                histogramHandler.report(point);
                continue;
            }
            if (OtlpMetricsUtils.wasFilteredByPreprocessor(point, pointHandler, preprocessor)) continue;
            pointHandler.report(point);
        }
    }

    private static List<ReportPoint> fromOtlpRequest(ExportMetricsServiceRequest request, @Nullable ReportableEntityPreprocessor preprocessor, String defaultSource, boolean includeResourceAttrsForMetrics, boolean includeAppTagsOnMetrics) {
        ArrayList wfPoints = Lists.newArrayList();
        for (ResourceMetrics resourceMetrics : request.getResourceMetricsList()) {
            Resource resource = resourceMetrics.getResource();
            OTLP_DATA_LOGGER.finest(() -> "Inbound OTLP Resource: " + resource);
            Pair<String, List<KeyValue>> sourceAndResourceAttrs = OtlpTraceUtils.sourceFromAttributes(resource.getAttributesList(), defaultSource);
            String source = (String)sourceAndResourceAttrs._1;
            List<KeyValue> resourceAttributes = Collections.EMPTY_LIST;
            if (includeResourceAttrsForMetrics) {
                resourceAttributes = OtlpMetricsUtils.replaceServiceNameKeyWithServiceKey((List)sourceAndResourceAttrs._2);
            } else if (includeAppTagsOnMetrics) {
                resourceAttributes = OtlpMetricsUtils.appTagsFromResourceAttrs((List)sourceAndResourceAttrs._2);
            }
            for (ScopeMetrics scopeMetrics : resourceMetrics.getScopeMetricsList()) {
                OTLP_DATA_LOGGER.finest(() -> "Inbound OTLP Instrumentation Scope: " + scopeMetrics.getScope());
                for (Metric otlpMetric : scopeMetrics.getMetricsList()) {
                    OTLP_DATA_LOGGER.finest(() -> "Inbound OTLP Metric: " + otlpMetric);
                    List<ReportPoint> points = OtlpMetricsUtils.transform(otlpMetric, resourceAttributes, preprocessor, source);
                    OTLP_DATA_LOGGER.finest(() -> "Converted Wavefront Metric: " + points);
                    wfPoints.addAll(points);
                }
            }
        }
        return wfPoints;
    }

    @VisibleForTesting
    static List<KeyValue> replaceServiceNameKeyWithServiceKey(List<KeyValue> attributes) {
        KeyValue serviceNameAttr = OtlpTraceUtils.getAttrByKey(attributes, "service.name");
        KeyValue serviceAttr = OtlpTraceUtils.getAttrByKey(attributes, "service");
        if (serviceNameAttr != null && serviceAttr == null) {
            ArrayList<KeyValue> attributesWithServiceKey = new ArrayList<KeyValue>(attributes);
            attributesWithServiceKey.remove(serviceNameAttr);
            attributesWithServiceKey.add(OtlpTraceUtils.buildKeyValue("service", serviceNameAttr.getValue().getStringValue()));
            return attributesWithServiceKey;
        }
        return attributes;
    }

    @VisibleForTesting
    static List<KeyValue> appTagsFromResourceAttrs(List<KeyValue> resourceAttrs) {
        List<KeyValue> attrList = new ArrayList<KeyValue>();
        attrList.add(OtlpTraceUtils.getAttrByKey(resourceAttrs, "application"));
        KeyValue serviceAttr = OtlpTraceUtils.getAttrByKey(resourceAttrs, "service");
        if (serviceAttr != null) {
            attrList.add(serviceAttr);
        } else {
            attrList.add(OtlpTraceUtils.getAttrByKey(resourceAttrs, "service.name"));
        }
        attrList.add(OtlpTraceUtils.getAttrByKey(resourceAttrs, "cluster"));
        attrList.add(OtlpTraceUtils.getAttrByKey(resourceAttrs, "shard"));
        attrList.removeAll(Collections.singleton(null));
        attrList = OtlpMetricsUtils.replaceServiceNameKeyWithServiceKey(attrList);
        return attrList;
    }

    @VisibleForTesting
    static boolean wasFilteredByPreprocessor(ReportPoint wfReportPoint, ReportableEntityHandler<ReportPoint, String> pointHandler, @Nullable ReportableEntityPreprocessor preprocessor) {
        if (preprocessor == null) {
            return false;
        }
        String[] messageHolder = new String[1];
        if (!preprocessor.forReportPoint().filter(wfReportPoint, messageHolder)) {
            if (messageHolder[0] != null) {
                pointHandler.reject(wfReportPoint, messageHolder[0]);
            } else {
                pointHandler.block(wfReportPoint);
            }
            return true;
        }
        return false;
    }

    @VisibleForTesting
    public static List<ReportPoint> transform(Metric otlpMetric, List<KeyValue> resourceAttrs, ReportableEntityPreprocessor preprocessor, String source) {
        ArrayList<ReportPoint> points = new ArrayList<ReportPoint>();
        if (otlpMetric.hasGauge()) {
            points.addAll(OtlpMetricsUtils.transformGauge(otlpMetric.getName(), otlpMetric.getGauge(), resourceAttrs));
        } else if (otlpMetric.hasSum()) {
            points.addAll(OtlpMetricsUtils.transformSum(otlpMetric.getName(), otlpMetric.getSum(), resourceAttrs));
        } else if (otlpMetric.hasSummary()) {
            points.addAll(OtlpMetricsUtils.transformSummary(otlpMetric.getName(), otlpMetric.getSummary(), resourceAttrs));
        } else if (otlpMetric.hasHistogram()) {
            points.addAll(OtlpMetricsUtils.transformHistogram(otlpMetric.getName(), OtlpMetricsUtils.fromOtelHistogram(otlpMetric.getName(), otlpMetric.getHistogram()), otlpMetric.getHistogram().getAggregationTemporality(), resourceAttrs));
        } else if (otlpMetric.hasExponentialHistogram()) {
            points.addAll(OtlpMetricsUtils.transformHistogram(otlpMetric.getName(), OtlpMetricsUtils.fromOtelExponentialHistogram(otlpMetric.getExponentialHistogram()), otlpMetric.getExponentialHistogram().getAggregationTemporality(), resourceAttrs));
        } else {
            throw new IllegalArgumentException("Otel: unsupported metric type for " + otlpMetric.getName());
        }
        for (ReportPoint point : points) {
            point.setHost(source);
            if (preprocessor == null) continue;
            preprocessor.forReportPoint().transform(point);
        }
        return points;
    }

    private static List<ReportPoint> transformSummary(String name, Summary summary, List<KeyValue> resourceAttrs) {
        ArrayList<ReportPoint> points = new ArrayList<ReportPoint>(summary.getDataPointsCount());
        for (SummaryDataPoint p : summary.getDataPointsList()) {
            points.addAll(OtlpMetricsUtils.transformSummaryDataPoint(name, p, resourceAttrs));
        }
        return points;
    }

    private static List<ReportPoint> transformSum(String name, Sum sum, List<KeyValue> resourceAttrs) {
        if (sum.getDataPointsCount() == 0) {
            throw new IllegalArgumentException("OTel: sum with no data points");
        }
        String prefix = "";
        switch (sum.getAggregationTemporality()) {
            case AGGREGATION_TEMPORALITY_CUMULATIVE: {
                break;
            }
            case AGGREGATION_TEMPORALITY_DELTA: {
                prefix = MetricConstants.DELTA_PREFIX;
                break;
            }
            default: {
                throw new IllegalArgumentException("OTel: sum with unsupported aggregation temporality " + sum.getAggregationTemporality().name());
            }
        }
        ArrayList<ReportPoint> points = new ArrayList<ReportPoint>(sum.getDataPointsCount());
        for (NumberDataPoint p : sum.getDataPointsList()) {
            points.add(OtlpMetricsUtils.transformNumberDataPoint(prefix + name, p, resourceAttrs));
        }
        return points;
    }

    private static List<ReportPoint> transformHistogram(String name, List<BucketHistogramDataPoint> dataPoints, AggregationTemporality aggregationTemporality, List<KeyValue> resourceAttrs) {
        switch (aggregationTemporality) {
            case AGGREGATION_TEMPORALITY_CUMULATIVE: {
                return OtlpMetricsUtils.transformCumulativeHistogram(name, dataPoints, resourceAttrs);
            }
            case AGGREGATION_TEMPORALITY_DELTA: {
                return OtlpMetricsUtils.transformDeltaHistogram(name, dataPoints, resourceAttrs);
            }
        }
        throw new IllegalArgumentException("OTel: histogram with unsupported aggregation temporality " + aggregationTemporality.name());
    }

    private static List<ReportPoint> transformDeltaHistogram(String name, List<BucketHistogramDataPoint> dataPoints, List<KeyValue> resourceAttrs) {
        ArrayList<ReportPoint> reportPoints = new ArrayList<ReportPoint>();
        for (BucketHistogramDataPoint dataPoint : dataPoints) {
            reportPoints.addAll(OtlpMetricsUtils.transformDeltaHistogramDataPoint(name, dataPoint, resourceAttrs));
        }
        return reportPoints;
    }

    private static List<ReportPoint> transformCumulativeHistogram(String name, List<BucketHistogramDataPoint> dataPoints, List<KeyValue> resourceAttrs) {
        ArrayList<ReportPoint> reportPoints = new ArrayList<ReportPoint>();
        for (BucketHistogramDataPoint dataPoint : dataPoints) {
            reportPoints.addAll(OtlpMetricsUtils.transformCumulativeHistogramDataPoint(name, dataPoint, resourceAttrs));
        }
        return reportPoints;
    }

    private static List<ReportPoint> transformDeltaHistogramDataPoint(String name, BucketHistogramDataPoint point, List<KeyValue> resourceAttrs) {
        ArrayList<ReportPoint> reportPoints = new ArrayList<ReportPoint>();
        BinsAndCounts binsAndCounts = point.asDelta();
        for (HistogramGranularity granularity : HistogramGranularity.values()) {
            int duration;
            switch (granularity) {
                case MINUTE: {
                    duration = 60000;
                    break;
                }
                case HOUR: {
                    duration = 3600000;
                    break;
                }
                case DAY: {
                    duration = 86400000;
                    break;
                }
                default: {
                    throw new IllegalArgumentException("Unknown granularity: " + granularity);
                }
            }
            Histogram histogram = Histogram.newBuilder().setType(HistogramType.TDIGEST).setBins(binsAndCounts.getBins()).setCounts(binsAndCounts.getCounts()).setDuration(duration).build();
            ReportPoint rp = OtlpMetricsUtils.pointWithAnnotations(name, point.getAttributesList(), resourceAttrs, point.getTimeUnixNano()).setValue(histogram).build();
            reportPoints.add(rp);
        }
        return reportPoints;
    }

    private static List<ReportPoint> transformCumulativeHistogramDataPoint(String name, BucketHistogramDataPoint point, List<KeyValue> resourceAttrs) {
        List<CumulativeBucket> buckets = point.asCumulative();
        ArrayList<ReportPoint> reportPoints = new ArrayList<ReportPoint>(buckets.size());
        for (CumulativeBucket bucket : buckets) {
            ReportPoint rp = OtlpMetricsUtils.pointWithAnnotations(name, point.getAttributesList(), resourceAttrs, point.getTimeUnixNano()).setValue(bucket.getCount()).build();
            OtlpMetricsUtils.handleDupAnnotation(rp);
            rp.getAnnotations().put("le", bucket.getTag());
            reportPoints.add(rp);
        }
        return reportPoints;
    }

    private static void handleDupAnnotation(ReportPoint rp) {
        if (rp.getAnnotations().containsKey("le")) {
            String val = (String)rp.getAnnotations().get("le");
            rp.getAnnotations().remove("le");
            rp.getAnnotations().put("_le", val);
        }
    }

    private static Collection<ReportPoint> transformGauge(String name, Gauge gauge, List<KeyValue> resourceAttrs) {
        if (gauge.getDataPointsCount() == 0) {
            throw new IllegalArgumentException("OTel: gauge with no data points");
        }
        ArrayList<ReportPoint> points = new ArrayList<ReportPoint>(gauge.getDataPointsCount());
        for (NumberDataPoint p : gauge.getDataPointsList()) {
            points.add(OtlpMetricsUtils.transformNumberDataPoint(name, p, resourceAttrs));
        }
        return points;
    }

    @NotNull
    private static ReportPoint transformNumberDataPoint(String name, NumberDataPoint point, List<KeyValue> resourceAttrs) {
        ReportPoint.Builder rp = OtlpMetricsUtils.pointWithAnnotations(name, point.getAttributesList(), resourceAttrs, point.getTimeUnixNano());
        if (point.hasAsInt()) {
            return rp.setValue(point.getAsInt()).build();
        }
        return rp.setValue(point.getAsDouble()).build();
    }

    @NotNull
    private static List<ReportPoint> transformSummaryDataPoint(String name, SummaryDataPoint point, List<KeyValue> resourceAttrs) {
        ArrayList<ReportPoint> toReturn = new ArrayList<ReportPoint>();
        List<KeyValue> pointAttributes = OtlpMetricsUtils.replaceQuantileTag(point.getAttributesList());
        toReturn.add(OtlpMetricsUtils.pointWithAnnotations(name + "_sum", pointAttributes, resourceAttrs, point.getTimeUnixNano()).setValue(point.getSum()).build());
        toReturn.add(OtlpMetricsUtils.pointWithAnnotations(name + "_count", pointAttributes, resourceAttrs, point.getTimeUnixNano()).setValue(point.getCount()).build());
        for (SummaryDataPoint.ValueAtQuantile q : point.getQuantileValuesList()) {
            ArrayList<KeyValue> attributes = new ArrayList<KeyValue>(pointAttributes);
            KeyValue quantileTag = KeyValue.newBuilder().setKey("quantile").setValue(AnyValue.newBuilder().setDoubleValue(q.getQuantile()).build()).build();
            attributes.add(quantileTag);
            toReturn.add(OtlpMetricsUtils.pointWithAnnotations(name, attributes, resourceAttrs, point.getTimeUnixNano()).setValue(q.getValue()).build());
        }
        return toReturn;
    }

    @NotNull
    private static List<KeyValue> replaceQuantileTag(List<KeyValue> pointAttributes) {
        if (pointAttributes.isEmpty()) {
            return pointAttributes;
        }
        ArrayList<KeyValue> modifiableAttributes = new ArrayList<KeyValue>();
        for (KeyValue pointAttribute : pointAttributes) {
            if (pointAttribute.getKey().equals("quantile")) {
                modifiableAttributes.add(KeyValue.newBuilder().setKey("_quantile").setValue(pointAttribute.getValue()).build());
                continue;
            }
            modifiableAttributes.add(pointAttribute);
        }
        return modifiableAttributes;
    }

    @NotNull
    private static ReportPoint.Builder pointWithAnnotations(String name, List<KeyValue> pointAttributes, List<KeyValue> resourceAttrs, long timeInNs) {
        ReportPoint.Builder builder = ReportPoint.newBuilder().setMetric(name);
        HashMap<String, String> annotations = new HashMap<String, String>();
        List<KeyValue> otlpAttributes = Stream.of(resourceAttrs, pointAttributes).flatMap(Collection::stream).collect(Collectors.toList());
        for (Annotation a : OtlpTraceUtils.annotationsFromAttributes(otlpAttributes)) {
            annotations.put(a.getKey(), a.getValue());
        }
        builder.setAnnotations(annotations);
        builder.setTimestamp(TimeUnit.NANOSECONDS.toMillis(timeInNs));
        return builder;
    }

    static List<BucketHistogramDataPoint> fromOtelHistogram(String name, io.opentelemetry.proto.metrics.v1.Histogram histogram) {
        ArrayList<BucketHistogramDataPoint> result = new ArrayList<BucketHistogramDataPoint>(histogram.getDataPointsCount());
        for (HistogramDataPoint dataPoint : histogram.getDataPointsList()) {
            result.add(OtlpMetricsUtils.fromOtelHistogramDataPoint(name, dataPoint));
        }
        return result;
    }

    static BucketHistogramDataPoint fromOtelHistogramDataPoint(String name, HistogramDataPoint dataPoint) {
        if (dataPoint.getExplicitBoundsCount() != dataPoint.getBucketCountsCount() - 1) {
            throw new IllegalArgumentException("OTel: histogram " + name + ": Explicit bounds count should be one less than bucket count. ExplicitBounds: " + dataPoint.getExplicitBoundsCount() + ", BucketCounts: " + dataPoint.getBucketCountsCount());
        }
        return new BucketHistogramDataPoint(dataPoint.getBucketCountsList(), dataPoint.getExplicitBoundsList(), dataPoint.getAttributesList(), dataPoint.getTimeUnixNano(), false);
    }

    static List<BucketHistogramDataPoint> fromOtelExponentialHistogram(ExponentialHistogram histogram) {
        ArrayList<BucketHistogramDataPoint> result = new ArrayList<BucketHistogramDataPoint>(histogram.getDataPointsCount());
        for (ExponentialHistogramDataPoint dataPoint : histogram.getDataPointsList()) {
            result.add(OtlpMetricsUtils.fromOtelExponentialHistogramDataPoint(dataPoint));
        }
        return result;
    }

    static BucketHistogramDataPoint fromOtelExponentialHistogramDataPoint(ExponentialHistogramDataPoint dataPoint) {
        double base = Math.pow(2.0, Math.pow(2.0, -dataPoint.getScale()));
        List negativeBucketCounts = dataPoint.getNegative().getBucketCountsList();
        List positiveBucketCounts = dataPoint.getPositive().getBucketCountsList();
        int numBucketCounts = 1 + negativeBucketCounts.size() + 1 + positiveBucketCounts.size() + 1;
        ArrayList<Long> bucketCounts = new ArrayList<Long>(numBucketCounts);
        ArrayList<Double> explicitBounds = new ArrayList<Double>(numBucketCounts - 1);
        OtlpMetricsUtils.appendNegativeBucketsAndExplicitBounds(dataPoint.getNegative().getOffset(), base, negativeBucketCounts, bucketCounts, explicitBounds);
        OtlpMetricsUtils.appendZeroBucketAndExplicitBound(dataPoint.getPositive().getOffset(), base, dataPoint.getZeroCount(), bucketCounts, explicitBounds);
        OtlpMetricsUtils.appendPositiveBucketsAndExplicitBounds(dataPoint.getPositive().getOffset(), base, positiveBucketCounts, bucketCounts, explicitBounds);
        return new BucketHistogramDataPoint(bucketCounts, explicitBounds, dataPoint.getAttributesList(), dataPoint.getTimeUnixNano(), true);
    }

    static void appendNegativeBucketsAndExplicitBounds(int negativeOffset, double base, List<Long> negativeBucketCounts, List<Long> bucketCounts, List<Double> explicitBounds) {
        bucketCounts.add(0L);
        double le = -Math.pow(base, (double)negativeOffset + (double)negativeBucketCounts.size());
        explicitBounds.add(le);
        for (int i = negativeBucketCounts.size() - 1; i >= 0; --i) {
            bucketCounts.add(negativeBucketCounts.get(i));
            explicitBounds.add(le /= base);
        }
    }

    static void appendZeroBucketAndExplicitBound(int positiveOffset, double base, long zeroBucketCount, List<Long> bucketCounts, List<Double> explicitBounds) {
        bucketCounts.add(zeroBucketCount);
        explicitBounds.add(Math.pow(base, positiveOffset));
    }

    static void appendPositiveBucketsAndExplicitBounds(int positiveOffset, double base, List<Long> positiveBucketCounts, List<Long> bucketCounts, List<Double> explicitBounds) {
        double le = Math.pow(base, positiveOffset);
        for (Long positiveBucketCount : positiveBucketCounts) {
            bucketCounts.add(positiveBucketCount);
            explicitBounds.add(le *= base);
        }
        bucketCounts.add(0L);
    }

    static class BucketHistogramDataPoint {
        private final List<Long> bucketCounts;
        private final List<Double> explicitBounds;
        private final List<KeyValue> attributesList;
        private final long timeUnixNano;
        private final boolean isExponential;

        private BucketHistogramDataPoint(List<Long> bucketCounts, List<Double> explicitBounds, List<KeyValue> attributesList, long timeUnixNano, boolean isExponential) {
            this.bucketCounts = bucketCounts;
            this.explicitBounds = explicitBounds;
            this.attributesList = attributesList;
            this.timeUnixNano = timeUnixNano;
            this.isExponential = isExponential;
        }

        List<CumulativeBucket> asCumulative() {
            if (this.isExponential) {
                return this.asCumulative(1, this.bucketCounts.size());
            }
            return this.asCumulative(0, this.bucketCounts.size());
        }

        BinsAndCounts asDelta() {
            if (this.isExponential) {
                return this.asDelta(1, this.bucketCounts.size() - 1);
            }
            return this.asDelta(0, this.bucketCounts.size());
        }

        private List<CumulativeBucket> asCumulative(int startBucketIndex, int endBucketIndex) {
            ArrayList<CumulativeBucket> result = new ArrayList<CumulativeBucket>(endBucketIndex - startBucketIndex);
            long leCount = 0L;
            for (int i = startBucketIndex; i < endBucketIndex; ++i) {
                result.add(new CumulativeBucket(this.leTagValue(i), leCount += this.bucketCounts.get(i).longValue()));
            }
            return result;
        }

        private String leTagValue(int index) {
            if (index == this.explicitBounds.size()) {
                return "+Inf";
            }
            return String.valueOf(this.explicitBounds.get(index));
        }

        private BinsAndCounts asDelta(int startBucketIndex, int endBucketIndex) {
            ArrayList<Double> bins = new ArrayList<Double>(endBucketIndex - startBucketIndex);
            ArrayList<Integer> counts = new ArrayList<Integer>(endBucketIndex - startBucketIndex);
            for (int i = startBucketIndex; i < endBucketIndex; ++i) {
                bins.add(this.centroidValue(i));
                counts.add(this.bucketCounts.get(i).intValue());
            }
            return new BinsAndCounts(bins, counts);
        }

        private double centroidValue(int index) {
            int length = this.explicitBounds.size();
            if (length == 0) {
                return 0.0;
            }
            if (index == 0) {
                return this.explicitBounds.get(0);
            }
            if (index == length) {
                return this.explicitBounds.get(length - 1);
            }
            return (this.explicitBounds.get(index - 1) + this.explicitBounds.get(index)) / 2.0;
        }

        List<KeyValue> getAttributesList() {
            return this.attributesList;
        }

        long getTimeUnixNano() {
            return this.timeUnixNano;
        }
    }

    static class BinsAndCounts {
        private final List<Double> bins;
        private final List<Integer> counts;

        BinsAndCounts(List<Double> bins, List<Integer> counts) {
            this.bins = bins;
            this.counts = counts;
        }

        List<Double> getBins() {
            return this.bins;
        }

        List<Integer> getCounts() {
            return this.counts;
        }
    }

    static class CumulativeBucket {
        private final String tag;
        private final long count;

        CumulativeBucket(String tag, long count) {
            this.tag = tag;
            this.count = count;
        }

        String getTag() {
            return this.tag;
        }

        long getCount() {
            return this.count;
        }
    }
}

