/*
 * Copyright 2002-2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.webauthn4j.data;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonValue;
import com.fasterxml.jackson.databind.exc.InvalidFormatException;

/**
 * WebAuthn Relying Parties may use AttestationConveyancePreference to specify their preference
 * regarding attestation conveyance during credential generation.
 *
 * @see <a href="https://www.w3.org/TR/webauthn-1/#enumdef-attestationconveyancepreference">
 * §5.4.6. Attestation Conveyance Preference Enumeration (enum AttestationConveyancePreference)</a>
 */
public enum AttestationConveyancePreference {

    /**
     * This value indicates that the Relying Party is not interested in authenticator attestation.
     * For example, in order to potentially avoid having to obtain user consent to relay identifying
     * information to the Relying Party, or to save a roundtrip to an Attestation CA.
     */
    NONE("none"),

    /**
     * This value indicates that the Relying Party prefers an attestation conveyance yielding
     * verifiable attestation statements, but allows the client to decide how to obtain such
     * attestation statements. The client MAY replace the authenticator-generated attestation
     * statements with attestation statements generated by an Anonymization CA, in order to protect
     * the user’s privacy, or to assist Relying Parties with attestation verification in a
     * heterogeneous ecosystem.
     */
    INDIRECT("indirect"),

    /**
     * This value indicates that the Relying Party wants to receive the attestation statement as
     * generated by the authenticator.
     */
    DIRECT("direct");

    private String value;

    AttestationConveyancePreference(String value) {
        this.value = value;
    }

    public static AttestationConveyancePreference create(String value) {
        if (value == null) {
            return null;
        }
        switch (value) {
            case "none":
                return NONE;
            case "indirect":
                return INDIRECT;
            case "direct":
                return DIRECT;
            default:
                throw new IllegalArgumentException("value '" + value + "' is out of range");
        }
    }

    @JsonCreator
    private static AttestationConveyancePreference deserialize(String value) throws InvalidFormatException {
        try {
            return create(value);
        } catch (IllegalArgumentException e) {
            throw new InvalidFormatException(null, "value is out of range", value, AttestationConveyancePreference.class);
        }
    }

    @JsonValue
    public String getValue() {
        return value;
    }

}
