/*
 * Copyright 2002-2019 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.webauthn4j.springframework.security.config.configurers;

import com.webauthn4j.converter.util.ObjectConverter;
import com.webauthn4j.data.*;
import com.webauthn4j.data.extension.client.AuthenticationExtensionClientInput;
import com.webauthn4j.data.extension.client.AuthenticationExtensionsClientInputs;
import com.webauthn4j.data.extension.client.RegistrationExtensionClientInput;
import com.webauthn4j.springframework.security.UserVerificationStrategy;
import com.webauthn4j.springframework.security.WebAuthnProcessingFilter;
import com.webauthn4j.springframework.security.challenge.ChallengeRepository;
import com.webauthn4j.springframework.security.endpoint.AssertionOptionsEndpointFilter;
import com.webauthn4j.springframework.security.endpoint.AttestationOptionsEndpointFilter;
import com.webauthn4j.springframework.security.extension.AuthenticationExtensionProvider;
import com.webauthn4j.springframework.security.extension.AuthenticationExtensionsClientInputsProvider;
import com.webauthn4j.springframework.security.extension.RegistrationExtensionProvider;
import com.webauthn4j.springframework.security.options.*;
import com.webauthn4j.springframework.security.server.ServerPropertyProvider;
import com.webauthn4j.springframework.security.server.ServerPropertyProviderImpl;
import org.springframework.security.authentication.AuthenticationTrustResolver;
import org.springframework.security.config.annotation.web.HttpSecurityBuilder;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.configurers.AbstractAuthenticationFilterConfigurer;
import org.springframework.security.web.authentication.ForwardAuthenticationFailureHandler;
import org.springframework.security.web.authentication.ForwardAuthenticationSuccessHandler;
import org.springframework.security.web.session.SessionManagementFilter;
import org.springframework.security.web.util.matcher.AntPathRequestMatcher;
import org.springframework.security.web.util.matcher.RequestMatcher;
import org.springframework.util.Assert;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

/**
 * Adds WebAuthn authentication. All attributes have reasonable defaults making all
 * parameters are optional. If no {@link #loginPage(String)} is specified, a default login
 * page will be generated by the framework.
 *
 * <h2>Security Filters</h2>
 * <p>
 * The following Filters are populated
 *
 * <ul>
 * <li>{@link WebAuthnProcessingFilter}</li>
 * <li>{@link AttestationOptionsEndpointFilter}</li>
 * </ul>
 *
 * <h2>Shared Objects Created</h2>
 * <p>
 * The following shared objects are populated
 * <ul>
 * <li>{@link ChallengeRepository}</li>
 * <li>{@link AssertionOptionsProvider}</li>
 * <li>{@link ServerPropertyProvider}</li>
 * </ul>
 *
 * <h2>Shared Objects Used</h2>
 * <p>
 * The following shared objects are used:
 *
 * <ul>
 * <li>{@link org.springframework.security.authentication.AuthenticationManager}</li>
 * </ul>
 *
 * @see WebAuthnAuthenticationProviderConfigurer
 */
public final class WebAuthnLoginConfigurer<H extends HttpSecurityBuilder<H>> extends
        AbstractAuthenticationFilterConfigurer<H, WebAuthnLoginConfigurer<H>, WebAuthnProcessingFilter> {

    //~ Instance fields
    // ================================================================================================
    private RpIdProvider rpIdProvider = null;
    private ObjectConverter objectConverter = null;
    private ServerPropertyProvider serverPropertyProvider = null;
    private AuthenticationTrustResolver trustResolver = null;
    private UserVerificationStrategy userVerificationStrategy = null;

    private final AttestationOptionsEndpointConfig attestationOptionsEndpointConfig = new AttestationOptionsEndpointConfig();
    private final AssertionOptionsEndpointConfig assertionOptionsEndpointConfig = new AssertionOptionsEndpointConfig();

    private String rpId = null;

    private String usernameParameter = null;
    private String passwordParameter = null;
    private String credentialIdParameter = null;
    private String clientDataJSONParameter = null;
    private String authenticatorDataParameter = null;
    private String signatureParameter = null;
    private String clientExtensionsJSONParameter = null;


    public WebAuthnLoginConfigurer() {
        super(new WebAuthnProcessingFilter(), null);
    }

    public static WebAuthnLoginConfigurer<HttpSecurity> webAuthnLogin() {
        return new WebAuthnLoginConfigurer<>();
    }

    // ~ Methods
    // ========================================================================================================

    /**
     * {@inheritDoc}
     */
    @Override
    public void configure(H http) throws Exception {
        super.configure(http);

        if (objectConverter == null) {
            objectConverter = WebAuthnConfigurerUtil.getObjectConverterOrCreateNew(http);
        }
        http.setSharedObject(ObjectConverter.class, objectConverter);

        if(rpIdProvider == null){
            rpIdProvider = WebAuthnConfigurerUtil.getRpIdProviderOrNull(http);
        }
        http.setSharedObject(RpIdProvider.class, rpIdProvider);

        if (serverPropertyProvider == null) {
            serverPropertyProvider = WebAuthnConfigurerUtil.getServerPropertyProviderOrCreateNew(http);
        }
        if(serverPropertyProvider instanceof ServerPropertyProviderImpl){
            ServerPropertyProviderImpl serverPropertyProviderImpl = (ServerPropertyProviderImpl)serverPropertyProvider;
            if(rpId != null){
                serverPropertyProviderImpl.setRpId(rpId);
            }
            if(rpIdProvider != null){
                serverPropertyProviderImpl.setRpIdProvider(rpIdProvider);
            }
        }
        http.setSharedObject(ServerPropertyProvider.class, serverPropertyProvider);

        if(trustResolver == null){
            trustResolver = WebAuthnConfigurerUtil.getAuthenticationTrustResolverOrCreateNew(http);
        }
        http.setSharedObject(AuthenticationTrustResolver.class, trustResolver);

        if(userVerificationStrategy == null){
            userVerificationStrategy = WebAuthnConfigurerUtil.getUserVerificationStrategyOrCreateNew(http);
        }
        http.setSharedObject(UserVerificationStrategy.class, userVerificationStrategy);


        // Configure AuthenticationFilter
        this.getAuthenticationFilter().setServerPropertyProvider(serverPropertyProvider);
        this.getAuthenticationFilter().setUserVerificationStrategy(userVerificationStrategy);
        configureParameters();

        // Configure OptionsEndPointFilters
        this.attestationOptionsEndpointConfig.configure(http);
        this.assertionOptionsEndpointConfig.configure(http);

    }

    private void configureParameters() {
        if (usernameParameter != null) {
            this.getAuthenticationFilter().setUsernameParameter(usernameParameter);
        }
        if (passwordParameter != null) {
            this.getAuthenticationFilter().setPasswordParameter(passwordParameter);
        }
        if (credentialIdParameter != null) {
            this.getAuthenticationFilter().setCredentialIdParameter(credentialIdParameter);
        }
        if (clientDataJSONParameter != null) {
            this.getAuthenticationFilter().setClientDataJSONParameter(clientDataJSONParameter);
        }
        if (authenticatorDataParameter != null) {
            this.getAuthenticationFilter().setAuthenticatorDataParameter(authenticatorDataParameter);
        }
        if (signatureParameter != null) {
            this.getAuthenticationFilter().setSignatureParameter(signatureParameter);
        }
        if (clientExtensionsJSONParameter != null) {
            this.getAuthenticationFilter().setClientExtensionsJSONParameter(clientExtensionsJSONParameter);
        }
    }

    /**
     * Sets the {@link ObjectConverter} to be used.
     *
     * @param objectConverter the {@link ObjectConverter}
     * @return the {@link WebAuthnLoginConfigurer} for additional customization
     */
    public WebAuthnLoginConfigurer<H> objectConverter(ObjectConverter objectConverter) {
        Assert.notNull(objectConverter, "objectConverter must not be null");
        this.objectConverter = objectConverter;
        return this;
    }

    /**
     * Sets the {@link ServerPropertyProvider} to be used.
     *
     * @param serverPropertyProvider the {@link ServerPropertyProvider}
     * @return the {@link WebAuthnLoginConfigurer} for additional customization
     */
    public WebAuthnLoginConfigurer<H> serverPropertyProvider(ServerPropertyProvider serverPropertyProvider) {
        Assert.notNull(serverPropertyProvider, "serverPropertyProvider must not be null");
        this.serverPropertyProvider = serverPropertyProvider;
        return this;
    }

    /**
     * Sets the {@link AuthenticationTrustResolver} to be used.
     *
     * @param trustResolver the {@link AuthenticationTrustResolver}
     * @return the {@link WebAuthnLoginConfigurer} for additional customization
     */
    public WebAuthnLoginConfigurer<H> trustResolver(AuthenticationTrustResolver trustResolver) {
        Assert.notNull(trustResolver, "trustResolver must not be null");
        this.trustResolver = trustResolver;
        return this;
    }

    /**
     * Sets the {@link UserVerificationStrategy} to be used.
     *
     * @param userVerificationStrategy the {@link UserVerificationStrategy}
     * @return the {@link WebAuthnLoginConfigurer} for additional customization
     */
    public WebAuthnLoginConfigurer<H> userVerificationStrategy(UserVerificationStrategy userVerificationStrategy) {
        Assert.notNull(userVerificationStrategy, "userVerificationStrategy must not be null");
        this.userVerificationStrategy = userVerificationStrategy;
        return this;
    }

    /**
     * Sets the {@link ServerPropertyProvider} to be used.
     * If both rpId and rpIdProvider are set, value from rpIdProvider is used.
     *
     * @param rpId the relying party Id
     * @return the {@link WebAuthnLoginConfigurer} for additional customization
     */
    public WebAuthnLoginConfigurer<H> rpId(String rpId) {
        this.rpId = rpId;
        return this;
    }

    /**
     * Sets the {@link RpIdProvider} to be used.
     * If both rpId and rpIdProvider are set, value from rpIdProvider is used.
     *
     * @param rpIdProvider the relying party id provider
     * @return the {@link WebAuthnLoginConfigurer} for additional customization
     */
    public WebAuthnLoginConfigurer<H> rpIdProvider(RpIdProvider rpIdProvider) {
        Assert.notNull(rpIdProvider, "rpIdProvider must not be null");
        this.rpIdProvider = rpIdProvider;
        return this;
    }

    /**
     * Returns the {@link AttestationOptionsEndpointConfig} for configuring the {@link AttestationOptionsEndpointFilter}
     *
     * @return the {@link AttestationOptionsEndpointConfig}
     */
    public AttestationOptionsEndpointConfig attestationOptionsEndpoint() {
        return attestationOptionsEndpointConfig;
    }

    /**
     * Returns the {@link AssertionOptionsEndpointConfig} for configuring the {@link AssertionOptionsEndpointFilter}
     *
     * @return the {@link AssertionOptionsEndpointConfig}
     */
    public AssertionOptionsEndpointConfig assertionOptionsEndpoint() {
        return assertionOptionsEndpointConfig;
    }

    /**
     * The HTTP parameter to look for the username when performing authentication. Default
     * is "username".
     *
     * @param usernameParameter the HTTP parameter to look for the username when
     *                              performing authentication
     * @return the {@link WebAuthnLoginConfigurer} for additional customization
     */
    public WebAuthnLoginConfigurer<H> usernameParameter(String usernameParameter) {
        Assert.hasText(usernameParameter, "usernameParameter must not be null or empty");
        this.usernameParameter = usernameParameter;
        return this;
    }

    /**
     * The HTTP parameter to look for the password when performing authentication. Default
     * is "password".
     *
     * @param passwordParameter the HTTP parameter to look for the password when
     *                              performing authentication
     * @return the {@link WebAuthnLoginConfigurer} for additional customization
     */
    public WebAuthnLoginConfigurer<H> passwordParameter(String passwordParameter) {
        Assert.hasText(passwordParameter, "passwordParameter must not be null or empty");
        this.passwordParameter = passwordParameter;
        return this;
    }

    /**
     * The HTTP parameter to look for the credentialId when performing authentication. Default
     * is "credentialId".
     *
     * @param credentialIdParameter the HTTP parameter to look for the credentialId when
     *                              performing authentication
     * @return the {@link WebAuthnLoginConfigurer} for additional customization
     */
    public WebAuthnLoginConfigurer<H> credentialIdParameter(String credentialIdParameter) {
        Assert.hasText(credentialIdParameter, "credentialIdParameter must not be null or empty");
        this.credentialIdParameter = credentialIdParameter;
        return this;
    }

    /**
     * The HTTP parameter to look for the clientData when performing authentication. Default
     * is "clientDataJSON".
     *
     * @param clientDataJSONParameter the HTTP parameter to look for the clientDataJSON when
     *                                performing authentication
     * @return the {@link WebAuthnLoginConfigurer} for additional customization
     */
    public WebAuthnLoginConfigurer<H> clientDataJSONParameter(String clientDataJSONParameter) {
        Assert.hasText(clientDataJSONParameter, "clientDataJSONParameter must not be null or empty");
        this.clientDataJSONParameter = clientDataJSONParameter;
        return this;
    }

    /**
     * The HTTP parameter to look for the authenticatorData when performing authentication. Default
     * is "authenticatorData".
     *
     * @param authenticatorDataParameter the HTTP parameter to look for the authenticatorData when
     *                                   performing authentication
     * @return the {@link WebAuthnLoginConfigurer} for additional customization
     */
    public WebAuthnLoginConfigurer<H> authenticatorDataParameter(String authenticatorDataParameter) {
        Assert.hasText(authenticatorDataParameter, "authenticatorDataParameter must not be null or empty");
        this.authenticatorDataParameter = authenticatorDataParameter;
        return this;
    }

    /**
     * The HTTP parameter to look for the signature when performing authentication. Default
     * is "signature".
     *
     * @param signatureParameter the HTTP parameter to look for the signature when
     *                           performing authentication
     * @return the {@link WebAuthnLoginConfigurer} for additional customization
     */
    public WebAuthnLoginConfigurer<H> signatureParameter(String signatureParameter) {
        Assert.hasText(signatureParameter, "signatureParameter must not be null or empty");
        this.signatureParameter = signatureParameter;
        return this;
    }

    /**
     * The HTTP parameter to look for the clientExtensionsJSON when performing authentication. Default
     * is "clientExtensionsJSON".
     *
     * @param clientExtensionsJSONParameter the HTTP parameter to look for the clientExtensionsJSON when
     *                                      performing authentication
     * @return the {@link WebAuthnLoginConfigurer} for additional customization
     */
    public WebAuthnLoginConfigurer<H> clientExtensionsJSONParameter(String clientExtensionsJSONParameter) {
        Assert.hasText(clientExtensionsJSONParameter, "clientExtensionsJSONParameter must not be null or empty");
        this.clientExtensionsJSONParameter = clientExtensionsJSONParameter;
        return this;
    }


    /**
     * Forward Authentication Success Handler
     *
     * @param forwardUrl the target URL in case of success
     * @return he {@link WebAuthnLoginConfigurer} for additional customization
     */
    public WebAuthnLoginConfigurer<H> successForwardUrl(String forwardUrl) {
        successHandler(new ForwardAuthenticationSuccessHandler(forwardUrl));
        return this;
    }

    /**
     * Forward Authentication Failure Handler
     *
     * @param forwardUrl the target URL in case of failure
     * @return he {@link WebAuthnLoginConfigurer} for additional customization
     */
    public WebAuthnLoginConfigurer<H> failureForwardUrl(String forwardUrl) {
        failureHandler(new ForwardAuthenticationFailureHandler(forwardUrl));
        return this;
    }

    /**
     * <p>
     * Specifies the URL to send users to if login is required. If used with
     * WebSecurityConfigurerAdapter a default login page will be generated when
     * this attribute is not specified.
     * </p>
     *
     * @param loginPage login page
     * @return the {@link WebAuthnLoginConfigurer} for additional customization
     */
    @Override
    public WebAuthnLoginConfigurer<H> loginPage(String loginPage) {
        return super.loginPage(loginPage);
    }

    /**
     * Create the {@link RequestMatcher} given a loginProcessingUrl
     *
     * @param loginProcessingUrl creates the {@link RequestMatcher} based upon the
     *                           loginProcessingUrl
     * @return the {@link RequestMatcher} to use based upon the loginProcessingUrl
     */
    @Override
    protected RequestMatcher createLoginProcessingUrlMatcher(String loginProcessingUrl) {
        return new AntPathRequestMatcher(loginProcessingUrl, "POST");
    }

    /**
     * Configuration options for the {@link AttestationOptionsEndpointFilter}
     */
    public class AttestationOptionsEndpointConfig {

        private String processingUrl = AttestationOptionsEndpointFilter.FILTER_URL;

        private AttestationOptionsProvider attestationOptionsProvider = null;

        private PublicKeyCredentialUserEntityProvider userProvider;
        private final RpConfig rpConfig = new RpConfig();
        private PublicKeyCredentialParameters[] pubKeyCredParams = null;
        private final AuthenticatorSelectionConfig authenticatorSelectionConfig = new AuthenticatorSelectionConfig();
        private AttestationConveyancePreference attestation = null;
        private Long timeout;
        private final RegistrationExtensionsClientInputsConfig extensions = new RegistrationExtensionsClientInputsConfig();

        private AttestationOptionsEndpointConfig() {
        }

        private void configure(H http) {
            AttestationOptionsEndpointFilter optionsEndpointFilter = WebAuthnConfigurerUtil.getAttestationOptionsEndpointFilterOrCreateNew(http);
            optionsEndpointFilter.setFilterProcessesUrl(processingUrl);

            if (attestationOptionsProvider == null) {
                attestationOptionsProvider = WebAuthnConfigurerUtil.getAttestationOptionsProviderOrCreateNew(http);
            }
            http.setSharedObject(AttestationOptionsProvider.class, attestationOptionsProvider);

            optionsEndpointFilter.setTrustResolver(trustResolver);

            if(attestationOptionsProvider instanceof AttestationOptionsProviderImpl){
                AttestationOptionsProviderImpl attestationOptionsProviderImpl = (AttestationOptionsProviderImpl)attestationOptionsProvider;

                if (userProvider != null){
                    attestationOptionsProviderImpl.setPublicKeyCredentialUserEntityProvider(userProvider);
                }

                this.rpConfig.configure(http);

                attestationOptionsProviderImpl.setPubKeyCredParams(pubKeyCredParams == null ? null : Arrays.asList(pubKeyCredParams));

                if (timeout != null) {
                    attestationOptionsProviderImpl.setRegistrationTimeout(timeout);
                }

                attestationOptionsProviderImpl.setRegistrationAuthenticatorSelection(authenticatorSelectionConfig.getAuthenticatorSelectionCriteria());
                if (attestation != null){
                    attestationOptionsProviderImpl.setAttestation(attestation);
                }
                attestationOptionsProviderImpl.setRegistrationExtensionsProvider(extensions.getExtensionsProvider());
            }

            http.addFilterAfter(optionsEndpointFilter, SessionManagementFilter.class);
        }

        /**
         * Specifies the {@link AttestationOptionsProvider} to be used.
         *
         * @param attestationOptionsProvider the {@link AttestationOptionsProvider}
         * @return the {@link WebAuthnLoginConfigurer} for additional customization
         */
        public AttestationOptionsEndpointConfig attestationOptionsProvider(AttestationOptionsProvider attestationOptionsProvider) {
            Assert.notNull(attestationOptionsProvider, "attestationOptionsProvider must not be null");
            this.attestationOptionsProvider = attestationOptionsProvider;
            return this;
        }

        /**
         * Sets the URL for the options endpoint
         *
         * @param processingUrl the URL for the options endpoint
         * @return the {@link AttestationOptionsEndpointConfig} for additional customization
         */
        public AttestationOptionsEndpointConfig processingUrl(String processingUrl) {
            this.processingUrl = processingUrl;
            return this;
        }

        /**
         * Sets the relying party name
         *
         * @param name the relying party name
         * @return the {@link AttestationOptionsEndpointConfig} for additional customization
         */
        public AttestationOptionsEndpointConfig rp(String name) {
            this.rpConfig.name = name;
            return this;
        }

        /**
         * Returns the {@link RpConfig} for configuring PublicKeyCredParams
         *
         * @return the {@link RpConfig}
         */
        public RpConfig rp(){
            return this.rpConfig;
        }

        public AttestationOptionsEndpointConfig userProvider(PublicKeyCredentialUserEntityProvider userProvider) {
            this.userProvider = userProvider;
            return this;
        }

        /**
         * Returns the {@link AttestationOptionsEndpointConfig} for configuring PublicKeyCredParams
         * @param pubKeyCredParams Desired properties of the credential to be created
         * @return the {@link AttestationOptionsEndpointConfig}
         */
        public AttestationOptionsEndpointConfig pubKeyCredParams(PublicKeyCredentialParameters... pubKeyCredParams) {
            this.pubKeyCredParams = pubKeyCredParams;
            return this;
        }

        /**
         * The timeout for registration ceremony
         *
         * @param timeout the timeout for registration ceremony
         * @return the {@link AttestationOptionsEndpointConfig} for additional customization
         */
        public AttestationOptionsEndpointConfig timeout(Long timeout) {
            this.timeout = timeout;
            return this;
        }


        public AuthenticatorSelectionConfig authenticatorSelection() {
            return authenticatorSelectionConfig;
        }

        /**
         * The attestation conveyance preference
         *
         * @param attestation the attestation conveyance preference
         * @return the {@link AttestationOptionsEndpointConfig} for additional customization
         */
        public AttestationOptionsEndpointConfig attestation(AttestationConveyancePreference attestation) {
            this.attestation = attestation;
            return this;
        }

        /**
         * Returns the {@link RegistrationExtensionsClientInputsConfig } for configuring registration extensions
         *
         * @return the {@link RegistrationExtensionsClientInputsConfig}
         */
        public RegistrationExtensionsClientInputsConfig extensions() {
            return this.extensions;
        }

        /**
         * Returns the {@link WebAuthnLoginConfigurer} for further configuration.
         *
         * @return the {@link WebAuthnLoginConfigurer}
         */
        public WebAuthnLoginConfigurer<H> and() {
            return WebAuthnLoginConfigurer.this;
        }


        /**
         * Configuration options for PublicKeyCredParams
         */
        public class RpConfig {

            private String id = null;
            private String name = null;

            private RpIdProvider idProvider = null;

            private RpConfig() {
            }

            @SuppressWarnings("unused")
            private void configure(H http) {
                if(attestationOptionsProvider instanceof AttestationOptionsProviderImpl){
                    AttestationOptionsProviderImpl attestationOptionsProviderImpl = (AttestationOptionsProviderImpl)attestationOptionsProvider;

                    if (rpConfig.id != null) {
                        attestationOptionsProviderImpl.setRpId(rpConfig.id);
                    }
                    else if(WebAuthnLoginConfigurer.this.rpId != null){
                        attestationOptionsProviderImpl.setRpId(WebAuthnLoginConfigurer.this.rpId);
                    }
                    if (rpConfig.name != null) {
                        attestationOptionsProviderImpl.setRpName(rpConfig.name);
                    }

                    if(rpConfig.idProvider != null){
                        attestationOptionsProviderImpl.setRpIdProvider(rpConfig.idProvider);
                    }
                    else if(WebAuthnLoginConfigurer.this.rpIdProvider != null){
                        attestationOptionsProviderImpl.setRpIdProvider(WebAuthnLoginConfigurer.this.rpIdProvider);
                    }

                }
            }

            /**
             * Sets relying party id for credential scoping. If both id and idProvider are set, value from idProvider is used.
             * If neither of them are set, inherits value from {@link WebAuthnLoginConfigurer#rpId} or {@link WebAuthnLoginConfigurer#rpIdProvider}
             *
             * @param id the relying party id
             * @return the {@link RpConfig} for additional customization
             */
            public RpConfig id(String id) {
                Assert.hasText(id, "id parameter must not be null or empty");
                this.id = id;
                return this;
            }

            /**
             * Sets relying party name
             *
             * @param name the relying party name
             * @return the {@link RpConfig} for additional customization
             */
            public RpConfig name(String name) {
                Assert.hasText(name, "name parameter must not be null or empty");
                this.name = name;
                return this;
            }

            /**
             * Sets relying party id provider for credential scoping
             *
             * @param idProvider the relying party id provider
             * @return the {@link RpConfig} for additional customization
             */
            public RpConfig idProvider(RpIdProvider idProvider) {
                Assert.notNull(idProvider, "idProvider parameter must not be null");
                this.idProvider = idProvider;
                return this;
            }


            /**
             * Returns the {@link AttestationOptionsEndpointConfig} for further configuration.
             *
             * @return the {@link AttestationOptionsEndpointConfig}
             */
            public AttestationOptionsEndpointConfig and() {
                return AttestationOptionsEndpointConfig.this;
            }

        }


        public class AuthenticatorSelectionConfig {

            private AuthenticatorAttachment authenticatorAttachment;
            private ResidentKeyRequirement residentKey;
            private UserVerificationRequirement userVerification;

            private AuthenticatorSelectionConfig() {
            }

            /**
             * Authenticator attachment requirement for authenticator filtering
             *
             * @param authenticatorAttachment authenticator attachment requirement
             * @return the {@link AuthenticatorSelectionConfig} for additional customization
             */
            public AuthenticatorSelectionConfig authenticatorAttachment(AuthenticatorAttachment authenticatorAttachment) {
                Assert.notNull(authenticatorAttachment, "authenticatorAttachment parameter must not be null or empty");
                this.authenticatorAttachment = authenticatorAttachment;
                return this;
            }

            /**
             * Requirement for client-side discoverable credentials filtering
             *
             * @param residentKey requirement for client-side discoverable credentials
             * @return the {@link AuthenticatorSelectionConfig} for additional customization
             */
            public AuthenticatorSelectionConfig residentKey(ResidentKeyRequirement residentKey) {
                Assert.notNull(residentKey, "residentKey parameter must not be null or empty");
                this.residentKey = residentKey;
                return this;
            }

            /**
             * user verification requirement for authenticator filtering
             *
             * @param userVerification user verification requirement
             * @return the {@link AuthenticatorSelectionConfig} for additional customization
             */
            public AuthenticatorSelectionConfig userVerification(UserVerificationRequirement userVerification) {
                Assert.notNull(userVerification, "userVerification parameter must not be null or empty");
                this.userVerification = userVerification;
                return this;
            }

            /**
             * Returns the {@link AttestationOptionsEndpointConfig} for further configuration.
             *
             * @return the {@link AttestationOptionsEndpointConfig}
             */
            public AttestationOptionsEndpointConfig and() {
                return AttestationOptionsEndpointConfig.this;
            }


            private AuthenticatorSelectionCriteria getAuthenticatorSelectionCriteria() {
                return new AuthenticatorSelectionCriteria(authenticatorAttachment, residentKey == ResidentKeyRequirement.REQUIRED, residentKey, userVerification);
            }
        }


        /**
         * Configuration options for AuthenticationExtensionsClientInputs
         */
        public class RegistrationExtensionsClientInputsConfig {

            private final AuthenticationExtensionsClientInputs.BuilderForRegistration builder = new AuthenticationExtensionsClientInputs.BuilderForRegistration();
            @SuppressWarnings("java:S1450")
            private List<RegistrationExtensionProvider> providers = Collections.emptyList();

            private RegistrationExtensionsClientInputsConfig() {
            }

            private AuthenticationExtensionsClientInputsProvider<RegistrationExtensionClientInput> getExtensionsProvider() {
                return httpServletRequest -> {
                    providers.forEach(provider -> provider.provide(builder, httpServletRequest));
                    return builder.build();
                };
            }

            /**
             * Configure uvm extension
             *
             * @param uvm flag to enable uvm extension
             * @return the {@link RegistrationExtensionsClientInputsConfig}
             */
            public RegistrationExtensionsClientInputsConfig uvm(Boolean uvm) {
                Assert.notNull(uvm, "uvm must not be null");
                builder.setUvm(uvm);
                return this;
            }

            /**
             * Configure credProps extension
             *
             * @param credProps flag to enable uvm extension
             * @return the {@link RegistrationExtensionsClientInputsConfig}
             */
            public RegistrationExtensionsClientInputsConfig credProps(Boolean credProps){
                Assert.notNull(credProps, "credProps must not be null");
                builder.setCredProps(credProps);
                return this;
            }

            /**
             * Add custom entry
             *
             * @param key key
             * @param value value
             * @return the {@link RegistrationExtensionsClientInputsConfig}
             */
            public RegistrationExtensionsClientInputsConfig entry(String key, Serializable value) {
                Assert.notNull(key, "key must not be null");
                Assert.notNull(value, "value must not be null");
                builder.set(key, value);
                return this;
            }

            public RegistrationExtensionsClientInputsConfig extensionProviders(RegistrationExtensionProvider... providers){
                this.providers = Arrays.asList(providers);
                return this;
            }

            /**
             * Returns the {@link WebAuthnLoginConfigurer} for further configuration.
             *
             * @return the {@link WebAuthnLoginConfigurer}
             */
            public WebAuthnLoginConfigurer<H> and() {
                return WebAuthnLoginConfigurer.this;
            }

        }

    }

    /**
     * Configuration options for the {@link AssertionOptionsEndpointFilter}
     */
    public class AssertionOptionsEndpointConfig {

        private String processingUrl = AssertionOptionsEndpointFilter.FILTER_URL;

        private AssertionOptionsProvider assertionOptionsProvider;

        private String rpId = null;
        private Long timeout;
        private final AuthenticationExtensionsClientInputsConfig extensions = new AuthenticationExtensionsClientInputsConfig();
        private UserVerificationRequirement userVerification;

        private RpIdProvider rpIdProvider = null;

        private AssertionOptionsEndpointConfig() {
        }

        private void configure(H http) {
            AssertionOptionsEndpointFilter optionsEndpointFilter = WebAuthnConfigurerUtil.getAssertionOptionsEndpointFilterOrCreateNew(http);
            optionsEndpointFilter.setFilterProcessesUrl(processingUrl);

            if (assertionOptionsProvider == null) {
                assertionOptionsProvider = WebAuthnConfigurerUtil.getAssertionOptionsProviderOrCreateNew(http);
            }
            http.setSharedObject(AssertionOptionsProvider.class, assertionOptionsProvider);
            optionsEndpointFilter.setTrustResolver(trustResolver);

            if(assertionOptionsProvider instanceof AssertionOptionsProviderImpl){
                AssertionOptionsProviderImpl optionsProviderImpl = (AssertionOptionsProviderImpl)assertionOptionsProvider;

                if(rpId != null){
                    optionsProviderImpl.setRpId(rpId);
                }
                else if(WebAuthnLoginConfigurer.this.rpId != null){
                    optionsProviderImpl.setRpId(WebAuthnLoginConfigurer.this.rpId);
                }

                if(rpIdProvider != null){
                    optionsProviderImpl.setRpIdProvider(rpIdProvider);
                }
                else if(WebAuthnLoginConfigurer.this.rpIdProvider != null){
                    optionsProviderImpl.setRpIdProvider(WebAuthnLoginConfigurer.this.rpIdProvider);
                }

                if (timeout != null) {
                    optionsProviderImpl.setAuthenticationTimeout(timeout);
                }
                optionsProviderImpl.setAuthenticationUserVerification(userVerification);
                optionsProviderImpl.setAuthenticationExtensionsProvider(extensions.getExtensionsProvider());
            }

            http.addFilterAfter(optionsEndpointFilter, SessionManagementFilter.class);
        }

        /**
         * Specifies the {@link AssertionOptionsEndpointConfig} to be used.
         *
         * @param assertionOptionsProvider the {@link AssertionOptionsProvider}
         * @return the {@link WebAuthnLoginConfigurer} for additional customization
         */
        public AssertionOptionsEndpointConfig assertionOptionsProvider(AssertionOptionsProvider assertionOptionsProvider) {
            Assert.notNull(assertionOptionsProvider, "assertionOptionsProvider must not be null");
            this.assertionOptionsProvider = assertionOptionsProvider;
            return this;
        }

        /**
         * Sets the URL for the options endpoint
         *
         * @param processingUrl the URL for the options endpoint
         * @return the {@link AssertionOptionsEndpointConfig} for additional customization
         */
        public AssertionOptionsEndpointConfig processingUrl(String processingUrl) {
            this.processingUrl = processingUrl;
            return this;
        }


        /**
         * Sets relying party id
         *
         * @param rpId  rpId
         * @return the {@link AssertionOptionsEndpointConfig} for additional customization
         */
        public AssertionOptionsEndpointConfig rpId(String rpId) {
            this.rpId = rpId;
            return this;
        }


        /**
         * Sets timeout for authentication ceremony
         *
         * @param timeout timeout for authentication ceremony
         * @return the {@link AssertionOptionsEndpointConfig} for additional customization
         */
        public AssertionOptionsEndpointConfig timeout(Long timeout) {
            this.timeout = timeout;
            return this;
        }

        /**
         * Sets user verification requirement for authentication ceremony
         *
         * @param userVerification user verification requirement for authentication ceremony
         * @return the {@link AssertionOptionsEndpointConfig} for additional customization
         */
        public AssertionOptionsEndpointConfig userVerification(UserVerificationRequirement userVerification) {
            this.userVerification = userVerification;
            return this;
        }

        /**
         * Returns the {@link AuthenticationExtensionsClientInputsConfig} for configuring authentication extensions
         *
         * @return the {@link AuthenticationExtensionsClientInputsConfig}
         */
        public AuthenticationExtensionsClientInputsConfig extensions() {
            return extensions;
        }

        /**
         * Sets relying party id provider for authentication ceremony
         *
         * @param rpIdProvider relying party id provider for authentication ceremony
         * @return the {@link AssertionOptionsEndpointConfig} for additional customization
         */
        public AssertionOptionsEndpointConfig rpIdProvider(RpIdProvider rpIdProvider) {
            this.rpIdProvider = rpIdProvider;
            return this;
        }

        /**
         * Returns the {@link WebAuthnLoginConfigurer} for further configuration.
         *
         * @return the {@link WebAuthnLoginConfigurer}
         */
        public WebAuthnLoginConfigurer<H> and() {
            return WebAuthnLoginConfigurer.this;
        }


        /**
         * Configuration options for AuthenticationExtensionsClientInputs
         */
        public class AuthenticationExtensionsClientInputsConfig {

            private final AuthenticationExtensionsClientInputs.BuilderForAuthentication builder = new AuthenticationExtensionsClientInputs.BuilderForAuthentication();
            @SuppressWarnings("java:S1450")
            private List<AuthenticationExtensionProvider> providers = Collections.emptyList();

            private AuthenticationExtensionsClientInputsConfig() {
            }

            private AuthenticationExtensionsClientInputsProvider<AuthenticationExtensionClientInput> getExtensionsProvider() {
                return httpServletRequest -> {
                    providers.forEach(provider -> provider.provide(builder, httpServletRequest));
                    return builder.build();
                };
            }

            /**
             * Configure appid extension
             *
             * @param appid appid
             * @return the {@link AuthenticationExtensionsClientInputsConfig}
             */
            public AuthenticationExtensionsClientInputsConfig appid(String appid) {
                Assert.notNull(appid, "appid must not be null");
                builder.setAppid(appid);
                return this;
            }

            /**
             * Configure appidExclude extension
             *
             * @param appidExclude appid
             * @return the {@link AuthenticationExtensionsClientInputsConfig}
             */
            public AuthenticationExtensionsClientInputsConfig appidExclude(String appidExclude) {
                Assert.notNull(appidExclude, "appidExclude must not be null");
                builder.setAppidExclude(appidExclude);
                return this;
            }

            /**
             * Configure uvm extension
             *
             * @param uvm flag to enable uvm extension
             * @return the {@link AuthenticationExtensionsClientInputsConfig}
             */
            public AuthenticationExtensionsClientInputsConfig uvm(Boolean uvm) {
                Assert.notNull(uvm, "uvm must not be null");
                builder.setUvm(uvm);
                return this;
            }

            /**
             * Add custom entry
             *
             * @param key key
             * @param value value
             * @return the {@link AuthenticationExtensionsClientInputsConfig}
             */
            public AuthenticationExtensionsClientInputsConfig entry(String key, Serializable value) {
                Assert.notNull(key, "key must not be null");
                Assert.notNull(value, "value must not be null");
                builder.set(key, value);
                return this;
            }

            public AuthenticationExtensionsClientInputsConfig extensionProviders(AuthenticationExtensionProvider... providers){
                this.providers = Arrays.asList(providers);
                return this;
            }

            /**
             * Returns the {@link AssertionOptionsEndpointConfig} for further configuration.
             *
             * @return the {@link AssertionOptionsEndpointConfig}
             */
            public WebAuthnLoginConfigurer<H>.AssertionOptionsEndpointConfig and() {
                return WebAuthnLoginConfigurer.AssertionOptionsEndpointConfig.this;
            }

        }


    }

}
