package com.webengage.sdk.android.actions.deeplink;


import android.annotation.SuppressLint;
import android.app.AlarmManager;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.content.pm.ResolveInfo;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.text.TextUtils;
import android.util.Pair;

import com.webengage.sdk.android.Action;
import com.webengage.sdk.android.Logger;
import com.webengage.sdk.android.PendingIntentFactory;
import com.webengage.sdk.android.WebEngage;
import com.webengage.sdk.android.actions.render.CallToAction;
import com.webengage.sdk.android.actions.render.PushNotificationData;
import com.webengage.sdk.android.utils.WebEngageConstant;
import com.webengage.sdk.android.utils.WebEngageUtils;

import org.json.JSONObject;

import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

class DeepLinkAction extends Action {

    private Context applicationContext = null;
    Bundle extras = null;
    CallToAction.TYPE type = null;
    boolean launchAppIfInvalid = false;
    String actionType = null;

    DeepLinkAction(Context context) {
        super(context);
        applicationContext = context.getApplicationContext();
    }

    @Override
    protected Object preExecute(Map<String, Object> actionAttributes) {
        Intent intent = (Intent) actionAttributes.get(DeepLinkActionController.ACTION_DATA);
        extras = intent.getExtras();
        if (extras != null) {
            launchAppIfInvalid = extras.getBoolean(WebEngageConstant.LAUNCH_APP_IF_INVALID, false);
            String type = extras.getString(WebEngageConstant.TYPE);
            if (CallToAction.TYPE.CUSTOM.toString().equals(type)) {
                this.type = CallToAction.TYPE.CUSTOM;
                try {

                    return new Pair<>(extras.getString(WebEngageConstant.CUSTOM_ACTION_JSON), extras.getString(WebEngageConstant.PUSH_NOTIFICATION_DATA));
                } catch (Exception e) {
                    Logger.e(WebEngageConstant.TAG, "Error while parsing custom action");
                    return null;
                }
            } else {
                String uri = extras.getString(WebEngageConstant.URI);
                List<String> params = null;
                try {
                    params = Uri.parse(uri).getPathSegments();
                } catch (Exception e) {

                }
                if (params != null) {
                    if (params.size() > 0) {
                        this.type = CallToAction.TYPE.valueFromString(params.get(0));
                    }
                    if (this.type != null && params.size() > 1) {
                        return params.get(1);
                    }
                }
            }
        }
        return null;
    }

    @Override
    protected Object execute(Object data) {
        if (data == null) {
            if (launchAppIfInvalid) {
                launchApp();
            }
        } else {
            if (this.type != null) {
                switch (this.type) {
                    case LAUNCH_ACTIVITY:
                        String activityPath = (String) data;
                        String packageName = this.applicationContext.getPackageName();
                        Bundle customData = null;
                        if (extras != null) {
                            customData = extras.getBundle(WebEngageConstant.CUSTOM_DATA);
                        }

                        Intent activityIntent = new Intent();
                        if (customData != null) {
                            activityIntent.putExtras(customData);
                        }
                        activityIntent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                        activityIntent.setClassName(packageName, activityPath);
                        if (activityIntent.resolveActivityInfo(this.applicationContext.getPackageManager(), 0) == null) {
                            if (launchAppIfInvalid) {
                                launchApp();
                            } else {
                                throw new IllegalArgumentException("Received activity path is not valid");
                            }
                        } else {
                            this.applicationContext.startActivity(activityIntent);
                        }

                        break;


                    case LINK:
                        Uri uri = Uri.parse((String) data);
                        Intent deeplinkIntent = new Intent(Intent.ACTION_VIEW, uri);
                        customData = null;
                        if (extras != null) {
                            customData = extras.getBundle(WebEngageConstant.CUSTOM_DATA);
                        }
                        if (customData != null) {
                            deeplinkIntent.putExtras(customData);
                        }
                        deeplinkIntent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                        List<ResolveInfo> resolveInfoList = this.applicationContext.getPackageManager().queryIntentActivities(deeplinkIntent, PackageManager.MATCH_DEFAULT_ONLY);
                        if (resolveInfoList != null && !resolveInfoList.isEmpty()) {
                            for (ResolveInfo resolveInfo : resolveInfoList) {
                                if (resolveInfo != null && resolveInfo.activityInfo != null && this.applicationContext.getPackageName().equals(resolveInfo.activityInfo.packageName)) {
                                    deeplinkIntent.setPackage(this.applicationContext.getPackageName());
                                    break;
                                }
                            }
                        }
                        if (resolveInfoList != null && !resolveInfoList.isEmpty()) {
                            this.applicationContext.startActivity(deeplinkIntent);
                        } else {
                            if (launchAppIfInvalid) {
                                launchApp();
                            } else {
                                throw new IllegalArgumentException("No App can handle implicit intent with link : " + data);
                            }
                        }
                        break;

                    case CUSTOM:
                        performCustomAction(data);
                        break;

                    default:
                        if (launchAppIfInvalid) {
                            launchApp();
                        }
                        break;
                }
            } else {
                if (launchAppIfInvalid) {
                    launchApp();
                }
            }
        }
        return null;
    }

    @Override
    protected void postExecute(Object data) {

    }

    private void launchApp() {
        Intent launchIntent = this.applicationContext.getPackageManager().getLaunchIntentForPackage(this.applicationContext.getPackageName());
        if (extras != null && extras.getBundle(WebEngageConstant.CUSTOM_DATA) != null) {
            launchIntent.putExtras(extras.getBundle(WebEngageConstant.CUSTOM_DATA));
        }
        this.applicationContext.startActivity(launchIntent);
    }

    private void performCustomAction(Object data) {
        try {
            Pair<String, String> pair = (Pair<String, String>) data;
            JSONObject customActionJSON = new JSONObject(pair.first);
            PushNotificationData pushNotificationData = new PushNotificationData(new JSONObject(pair.second), applicationContext);
            String actionType = "";
            String actionValue = "";
            String attributeName = "";
            String attributeValue = "";
            String attributeType = "";
            boolean isAttributeNullOrEmpty = customActionJSON.optJSONArray(WebEngageConstant.CUSTOM_ACTION) != null &&
                    Objects.requireNonNull(customActionJSON.optJSONArray(WebEngageConstant.CUSTOM_ACTION)).getJSONObject(0) != null;

            if (customActionJSON.has(WebEngageConstant.ACTION_TYPE) && !TextUtils.isEmpty(customActionJSON.getString(WebEngageConstant.ACTION_TYPE)))
                actionType = customActionJSON.getString(WebEngageConstant.ACTION_TYPE);

            if (customActionJSON.has(WebEngageConstant.CUSTOM_ACTION_VALUE) && !TextUtils.isEmpty(customActionJSON.getString(WebEngageConstant.CUSTOM_ACTION_VALUE)))
                actionValue = customActionJSON.getString(WebEngageConstant.CUSTOM_ACTION_VALUE);

            if (isAttributeNullOrEmpty && containsNotNullKey(customActionJSON, WebEngageConstant.CUSTOM_ACTION_ATTRIBUTE_NAME))
                attributeName = Objects.requireNonNull(customActionJSON.optJSONArray(WebEngageConstant.CUSTOM_ACTION))
                        .getJSONObject(0).getString(WebEngageConstant.CUSTOM_ACTION_ATTRIBUTE_NAME);

            if (isAttributeNullOrEmpty && containsNotNullKey(customActionJSON, WebEngageConstant.CUSTOM_ACTION_ATTRIBUTE_VALUE))
                attributeValue = Objects.requireNonNull(customActionJSON.optJSONArray(WebEngageConstant.CUSTOM_ACTION))
                        .getJSONObject(0).getString(WebEngageConstant.CUSTOM_ACTION_ATTRIBUTE_VALUE);

            if (isAttributeNullOrEmpty && containsNotNullKey(customActionJSON, WebEngageConstant.CUSTOM_ACTION_ATTRIBUTE_TYPE))
                attributeType = Objects.requireNonNull(customActionJSON.optJSONArray(WebEngageConstant.CUSTOM_ACTION))
                        .getJSONObject(0).getString(WebEngageConstant.CUSTOM_ACTION_ATTRIBUTE_TYPE);

            handleCustomActions(actionType, actionValue, attributeName, attributeValue, attributeType, pushNotificationData, pair.second);

        } catch (Exception e) {

        }
    }

    private void handleCustomActions(String actionType, String actionValue, String attributeName,
                                     String attributeValue, String attributeType, PushNotificationData pushNotificationData,
                                     String pushNotificationDataJson) {
        switch (actionType) {
            case WebEngageConstant.COPY:
                WebEngageUtils.copyToClipboard(applicationContext, actionValue);
                break;

            case WebEngageConstant.CUSTOM_ACTION_EVENT:
                HashMap<String, Object> map = new HashMap<>();
                map.put(attributeName, attributeValue);
                WebEngage.get().analytics().track(actionValue, map);
                break;

            case WebEngageConstant.USER_ATTRIBUTE:
                setUserAttribute(attributeName, attributeValue, attributeType);
                break;

            case WebEngageConstant.PHONE_CALL:
                WebEngageUtils.makePhoneCall(applicationContext, actionValue);
                break;

            case WebEngageConstant.SHARE:
                WebEngageUtils.shareContent(applicationContext, actionValue);
                break;

            case WebEngageConstant.SNOOZE:
                String id = pushNotificationData.getVariationId();
                saveVolatileData(id, pushNotificationDataJson);
                long snoozeInterval = calculateSnoozeInterval(actionValue);
                if (snoozeInterval != -1)
                    scheduleNotificationReRender(pushNotificationData, snoozeInterval);
                break;
            case WebEngageConstant.DYNAMIC_ACTION:
                break;

            default:

        }
    }

    private boolean containsNotNullKey(JSONObject customActionJSON, String key) {
        try {
            return Objects.requireNonNull(customActionJSON.optJSONArray(WebEngageConstant.CUSTOM_ACTION)).length() > 0
                    && Objects.requireNonNull(customActionJSON.optJSONArray(WebEngageConstant.CUSTOM_ACTION)).getJSONObject(0).has(key)
                    && !TextUtils.isEmpty(Objects.requireNonNull(customActionJSON.optJSONArray(WebEngageConstant.CUSTOM_ACTION)).getJSONObject(0).getString(key));
        } catch (Exception e) {
            return false;
        }

    }

    private void setUserAttribute(String attributeName, String attributeValue, String attributeType) {
        switch (attributeType) {
            case WebEngageConstant.STRING:
                WebEngage.get().user().setAttribute(attributeName, attributeValue);
                break;
            case WebEngageConstant.NUMBER:
                if (isInteger(attributeValue))
                    WebEngage.get().user().setAttribute(attributeName, Integer.parseInt(attributeValue));
                else if (isDouble(attributeValue))
                    WebEngage.get().user().setAttribute(attributeName, Double.parseDouble(attributeValue));
                else if (isLong(attributeValue))
                    WebEngage.get().user().setAttribute(attributeName, Long.parseLong(attributeValue));
                break;
            case WebEngageConstant.DATE:
                WebEngage.get().user().setAttribute(attributeName, new Date(Long.parseLong(attributeValue)));
                break;
            case WebEngageConstant.BOOLEAN:
                WebEngage.get().user().setAttribute(attributeName, Boolean.valueOf(attributeValue));
                break;
        }
    }

    private boolean isLong(String attributeValue) {
        try {
            Long.parseLong(attributeValue);
            return true;
        } catch (NumberFormatException e) {
            return false;
        }
    }

    private boolean isDouble(String attributeValue) {
        try {
            Double.parseDouble(attributeValue);
            return true;
        } catch (NumberFormatException e) {
            return false;
        }
    }

    public static boolean isInteger(String attributeValue) {
        try {
            Integer.parseInt(attributeValue);
            return true;
        } catch (NumberFormatException e) {
            return false;
        }
    }

    private long calculateSnoozeInterval(String actionValue) {
        int minHrValue = Integer.parseInt(actionValue.substring(0, actionValue.length() - 1));
        if (actionValue.endsWith("m")) {
            return minHrValue * WebEngageConstant.ONE_MINUTE;
        } else if (actionValue.endsWith("h")) {
            return minHrValue * WebEngageConstant.ONE_HOUR;
        } else {
            return -1;
        }
    }

    /**
     * for scheduling alarm for notification re render
     */
    @SuppressLint("MissingPermission")
    void scheduleNotificationReRender(PushNotificationData pushNotificationData, long snoozeInterval) {
        long alarmInterval = snoozeInterval;
        //setting alarm interval to 5 sec if less than 5 second(min interval for subsequent alarms)
        if (alarmInterval < WebEngageConstant.MINIMUM_UPDATE_INTERVAL_ALARM) {
            alarmInterval = WebEngageConstant.MINIMUM_UPDATE_INTERVAL_ALARM;
        }
        AlarmManager alarmManager = (AlarmManager) applicationContext.getSystemService(Context.ALARM_SERVICE);
        Bundle bundle = new Bundle();
        bundle.putBoolean(WebEngageConstant.SNOOZE, true);
        bundle.putBoolean(WebEngageConstant.WE_RENDER, true);
        PendingIntent pendingIntent = PendingIntentFactory.constructSnoozeReRenderPendingIntent(applicationContext, pushNotificationData, "snooze", bundle);
        //setting exact and in exact based on the permissions and OS version
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.S && !alarmManager.canScheduleExactAlarms()) {
            alarmManager.set(AlarmManager.RTC, System.currentTimeMillis() + alarmInterval, pendingIntent);
        } else {
            alarmManager.setExact(AlarmManager.RTC, System.currentTimeMillis() + alarmInterval, pendingIntent);
        }
    }
}
