package com.webengage.sdk.android.actions.render;

import android.annotation.SuppressLint;
import android.app.AlarmManager;
import android.app.Notification;
import android.app.NotificationChannel;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.content.Context;
import android.content.pm.PackageManager;
import android.content.res.ColorStateList;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Color;
import android.graphics.drawable.Icon;
import android.os.Build;
import android.os.Bundle;
import android.os.SystemClock;
import android.text.TextUtils;
import android.util.TypedValue;
import android.view.View;
import android.widget.RemoteViews;

import androidx.core.app.NotificationManagerCompat;

import com.webengage.sdk.android.Logger;
import com.webengage.sdk.android.PendingIntentFactory;
import com.webengage.sdk.android.PushChannelManager;
import com.webengage.sdk.android.PushUtils;
import com.webengage.sdk.android.R;
import com.webengage.sdk.android.utils.ManifestUtils;
import com.webengage.sdk.android.utils.WebEngageConstant;
import com.webengage.sdk.android.utils.htmlspanner.WEHtmlParserInterface;
import com.webengage.sdk.android.utils.http.CachePolicy;
import com.webengage.sdk.android.utils.http.RequestMethod;
import com.webengage.sdk.android.utils.http.RequestObject;
import com.webengage.sdk.android.utils.http.WENetworkUtil;

import org.json.JSONException;

import java.io.IOException;
import java.io.InputStream;
import java.text.DateFormat;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Contains all common methods to create custom notification
 */
public class NotificationConfigurator {

    /**
     * user for setting up notification builder
     *
     * @param mBuilder             notification builder
     * @param pushNotificationData push data
     * @param applicationContext   context
     */
    void buildNotificationBuilder(Notification.Builder mBuilder, PushNotificationData pushNotificationData, Context applicationContext) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            mBuilder.setVisibility(Notification.VISIBILITY_PRIVATE);
        }
        mBuilder.setSmallIcon(pushNotificationData.getSmallIcon());
        if (!TextUtils.isEmpty(pushNotificationData.getTitle()))
            mBuilder.setContentTitle(new WEHtmlParserInterface().fromHtml(pushNotificationData.getTitle()));
        if (!TextUtils.isEmpty(pushNotificationData.getContentText()) && !pushNotificationData.getContentText().equals("null"))
            mBuilder.setContentText(new WEHtmlParserInterface().fromHtml(pushNotificationData.getContentText()));
        if (pushNotificationData.getContentSummary() != null) {
            mBuilder.setSubText(new WEHtmlParserInterface().fromHtml(pushNotificationData.getContentSummary()));
        }
        if (pushNotificationData.getLargeIcon() != null) {
            mBuilder.setLargeIcon(pushNotificationData.getLargeIcon());
        } else if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            mBuilder.setLargeIcon(Icon.createWithResource(applicationContext,
                    applicationContext.getApplicationInfo().icon));
        }
        if (pushNotificationData.isSticky())
            mBuilder.setOngoing(true);
        // set push configurations
        if (Build.VERSION.SDK_INT < 26) {
            int[] ar = {Notification.PRIORITY_MIN, Notification.PRIORITY_LOW,
                    Notification.PRIORITY_DEFAULT, Notification.PRIORITY_HIGH,
                    Notification.PRIORITY_MAX};
            if (ar.length < pushNotificationData.getPriority() + 2) {
                mBuilder.setPriority(ar[pushNotificationData.getPriority() + 2]);
            }
        }
        if (Build.VERSION.SDK_INT < 26) {
            if (pushNotificationData.getVibrateFlag()) {
                if (ManifestUtils.checkPermission(ManifestUtils.VIBRATE, applicationContext)) {
                    mBuilder.setDefaults(Notification.DEFAULT_VIBRATE);
                }
            }
            if (pushNotificationData.getSound() != null) {
                mBuilder.setSound(pushNotificationData.getSound());
            }
            if (pushNotificationData.getLedColor() != 0) {
                mBuilder.setLights(pushNotificationData.getLedColor(), 500, 1000);
            }
        }

    }

    /**
     * @param applicationContext   context
     * @param pushNotificationData data
     * @param when                 when time in milliseconds
     * @return common push base
     */
    RemoteViews getCustomPushBase(Context applicationContext, PushNotificationData pushNotificationData, long when, int layoutId) {
        RemoteViews pushBase = new RemoteViews(applicationContext.getPackageName(), layoutId);
        int targetSdkVersion = applicationContext.getApplicationInfo().targetSdkVersion;
        boolean hasBackgroundColor = pushNotificationData.getBackgroundColor() != Color.parseColor(WebEngageConstant.WE_TRANSPARENT_COLOR);
        boolean isAndroid12 = Build.VERSION.SDK_INT >= Build.VERSION_CODES.S && applicationContext.getApplicationInfo().targetSdkVersion >= Build.VERSION_CODES.S;
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N && targetSdkVersion >= Build.VERSION_CODES.N) {
            //For Android 12 , template is provided for all notifications.
            // So hide the View used for showing the app name in custom notifications
            if (isAndroid12) {
                pushBase.setViewVisibility(R.id.custom_small_head_container, View.GONE);
            } else {
                pushBase.setViewVisibility(R.id.custom_small_head_container, View.VISIBLE);
                int smallIcon = pushNotificationData.getSmallIcon();
                if (smallIcon != -1) {
                    pushBase.setImageViewResource(R.id.small_icon, smallIcon);
                } else {
                    pushBase.setImageViewIcon(R.id.small_icon, Icon.createWithResource(applicationContext, applicationContext.getApplicationInfo().icon));
                }
                String appName = pushNotificationData.getAppName();
                if (appName != null) {
                    pushBase.setTextViewText(R.id.app_name, appName);
                    pushBase.setTextViewText(R.id.app_name_native, appName);
                }
            }
            if (isAndroid12) {
                int leftPadding = applicationContext.getResources().getDimensionPixelSize(R.dimen.we_push_content_margin_colorbg);
                if (!hasBackgroundColor)
                    leftPadding = 0;
                pushBase.setViewPadding(R.id.custom_head_container, leftPadding, 0, leftPadding, 0);
            }
            if (hasBackgroundColor) {
                pushBase.setInt(R.id.push_base_container, "setBackgroundColor", pushNotificationData.getBackgroundColor());
            }
        } else {
            pushBase.setViewVisibility(R.id.custom_small_head_container, View.GONE);
        }
        if(pushNotificationData.getLargeIcon() != null && Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            pushBase.setImageViewIcon(R.id.large_icon, Icon.createWithBitmap(pushNotificationData.getLargeIcon()));
        } else if (pushNotificationData.getLargeIcon() != null) {
            pushBase.setImageViewBitmap(R.id.large_icon, pushNotificationData.getLargeIcon());
        } else if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            pushBase.setImageViewIcon(R.id.large_icon, Icon.createWithResource(applicationContext, applicationContext.getApplicationInfo().icon));
        }
        pushBase.setViewVisibility(R.id.push_base_margin_view, View.GONE);
        if (!hasBackgroundColor) {
            showTextViews(pushBase);
        } else {
            showNativeTextViews(pushBase);
        }
        if (!TextUtils.isEmpty(pushNotificationData.getTitle())) {
            pushBase.setTextViewText(R.id.custom_title, new WEHtmlParserInterface().fromHtml(pushNotificationData.getTitle()));
            pushBase.setTextViewText(R.id.custom_title_native, new WEHtmlParserInterface().fromHtml(pushNotificationData.getTitle()));
        } else {
            pushBase.setViewVisibility(R.id.custom_title, View.GONE);
            pushBase.setViewVisibility(R.id.custom_title_native, View.GONE);
        }

        if (!TextUtils.isEmpty(pushNotificationData.getContentText()) && !pushNotificationData.getContentText().equals("null")) {
            pushBase.setTextViewText(R.id.custom_message, new WEHtmlParserInterface().fromHtml(pushNotificationData.getContentText()));
            pushBase.setTextViewText(R.id.custom_message_native, new WEHtmlParserInterface().fromHtml(pushNotificationData.getContentText()));
        } else {
            pushBase.setViewVisibility(R.id.custom_message, View.GONE);
            pushBase.setViewVisibility(R.id.custom_message_native, View.GONE);
        }

        if (!TextUtils.isEmpty(pushNotificationData.getContentSummary())) {
            pushBase.setTextViewText(R.id.custom_summary, new WEHtmlParserInterface().fromHtml(pushNotificationData.getContentSummary()));
            pushBase.setTextViewText(R.id.custom_summary_native, new WEHtmlParserInterface().fromHtml(pushNotificationData.getContentSummary()));
        } else {
            pushBase.setViewVisibility(R.id.custom_summary, View.GONE);
            pushBase.setViewVisibility(R.id.custom_summary_native, View.GONE);
        }
        pushBase.setImageViewResource(R.id.small_icon, pushNotificationData.getSmallIcon());
        DateFormat dateFormat = android.text.format.DateFormat.getTimeFormat(applicationContext);
        String time = dateFormat.format(when);
        pushBase.setTextViewText(R.id.custom_notification_time, time);
        pushBase.setTextViewText(R.id.custom_notification_time_native, time);

        return pushBase;
    }


    void addIntents(RemoteViews remoteViews, Notification.Builder mBuilder, PushNotificationData pushNotificationData, Context context) {
        List<CallToAction> callToActions = pushNotificationData.getCallToActions();
        if (callToActions != null && callToActions.size() > 0) {
            int actionNumber = 0;
            for (CallToAction callToAction : callToActions) {
                if (!callToAction.isPrimeAction()) {
                    if (callToAction.isNative()) {
                        PendingIntent ctaPendingIntent = PendingIntentFactory.constructPushClickPendingIntent(context, pushNotificationData, callToAction, true);
                        //added for dismiss cta
                        try {
                            if (callToAction.getType() != null && callToAction.getType().equals(CallToAction.TYPE.CUSTOM) &&
                                    callToAction.getCustomActionJSON() != null && WebEngageConstant.WE_DISMISS.equals(callToAction.getCustomActionJSON().getString(WebEngageConstant.ACTION_TYPE))) {
                                ctaPendingIntent = PendingIntentFactory.constructPushDeletePendingIntent(context, pushNotificationData);
                            }
                        } catch (JSONException e) {
                        }
                        actionNumber++;
                        int actionId = -1;
                        switch (actionNumber) {
                            case 1:
                                if (Build.VERSION.SDK_INT <= Build.VERSION_CODES.O_MR1) {
                                    remoteViews.setInt(R.id.action_list,
                                            "setBackgroundColor", Color.parseColor("#e8e8e8"));
                                }
                                remoteViews.setViewVisibility(R.id.action_list, View.VISIBLE);
                                if (pushNotificationData.getBackgroundColor() != Color.parseColor(WebEngageConstant.WE_TRANSPARENT_COLOR)) {
                                    actionId = R.id.action1_native;
                                } else {
                                    actionId = R.id.action1_adaptive;
                                }
                                break;
                            case 2:
                                if (pushNotificationData.getBackgroundColor() != Color.parseColor(WebEngageConstant.WE_TRANSPARENT_COLOR)) {
                                    actionId = R.id.action2_native;
                                } else {
                                    actionId = R.id.action2_adaptive;
                                }
                                break;
                            case 3:
                                if (pushNotificationData.getBackgroundColor() != Color.parseColor(WebEngageConstant.WE_TRANSPARENT_COLOR)) {
                                    actionId = R.id.action3_native;
                                } else {
                                    actionId = R.id.action3_adaptive;
                                }
                                break;
                        }
                        if (actionId != -1) {
                            remoteViews.setViewVisibility(actionId, View.VISIBLE);
                            remoteViews.setTextViewText(actionId, new WEHtmlParserInterface().fromHtml(callToAction.getText()));
                            remoteViews.setOnClickPendingIntent(actionId, ctaPendingIntent);
                        }
                    }
                }
            }
        }
        // delete intent
        PendingIntent deletePendingIntent = PendingIntentFactory.constructPushDeletePendingIntent(context, pushNotificationData);
        mBuilder.setDeleteIntent(deletePendingIntent);
    }

    Bitmap loadImage(String imageUrl, Context applicationContext) {
        if (imageUrl != null) {
            HashMap<String, String> headers = new HashMap<>();
            headers.put("accept", "image/webp");
            RequestObject requestObject = new RequestObject.Builder(imageUrl, RequestMethod.GET, applicationContext)
                    .setCachePolicy(CachePolicy.GET_DATA_FROM_CACHE_FIRST_ELSE_DOWNLOAD_AND_CACHE)
                    .setTag(WebEngageConstant.LANDSCAPE)
                    .setHeaders(headers)
                    .build();
            InputStream responseStream = null;
            try {
                Map<String, Object> result = WENetworkUtil.makeRequest(applicationContext, requestObject, true, true);
                responseStream = (InputStream) result.get("data");
                Bitmap bitmap = loadImage(responseStream);
                responseStream.close();
                if (bitmap != null) {
                    return bitmap;
                }
            } catch (Exception e) {
                if (responseStream != null) {
                    try {
                        responseStream.close();
                    } catch (IOException ioException) {
                        ioException.printStackTrace();
                    }
                }
                e.printStackTrace();
                return null;
            }
        }
        return null;
    }

    Bitmap loadImage(InputStream inputStream) {
        if (inputStream != null) {
            try {
                return BitmapFactory.decodeStream(inputStream);
            } catch (Exception e) {
                Logger.e(WebEngageConstant.TAG, "Exception while decoding input stream to bitmap.", e);
            } catch (OutOfMemoryError error) {
                Logger.e(WebEngageConstant.TAG, "Error while decoding input stream to bitmap.", error);
            }
        }
        return null;
    }

    void showTextViews(RemoteViews pushBase) {
        pushBase.setViewVisibility(R.id.app_name_native, View.GONE);
        pushBase.setViewVisibility(R.id.custom_summary_native, View.GONE);
        pushBase.setViewVisibility(R.id.custom_notification_time_native, View.GONE);
        pushBase.setViewVisibility(R.id.custom_title_native, View.GONE);
        pushBase.setViewVisibility(R.id.custom_message_native, View.GONE);

    }

    void showNativeTextViews(RemoteViews pushBase) {
        pushBase.setViewVisibility(R.id.app_name, View.GONE);
        pushBase.setViewVisibility(R.id.custom_summary, View.GONE);
        pushBase.setViewVisibility(R.id.custom_notification_time, View.GONE);
        pushBase.setViewVisibility(R.id.custom_title, View.GONE);
        pushBase.setViewVisibility(R.id.custom_message, View.GONE);

    }

    /**
     * for getting channel id
     *
     * @param pushNotificationData push data
     * @param context              context
     * @return channel id
     */
    String getChannelId(PushNotificationData pushNotificationData, Context context) {
        String channelId = pushNotificationData.getChannelId();
        //added for default channel ID support in Push Channel Configuration
        String defaultChannelId = PushChannelManager.getDefaultChannelId();
        if (Build.VERSION.SDK_INT >= 26) {
            if (channelId != null) {
                if (!PushChannelManager.isChannelPresent(channelId, context)) {
                    channelId = defaultChannelId;
                }
            } else {
                channelId = defaultChannelId;
            }
        }
        return channelId;
    }

    public void setChronometer(RemoteViews remoteViews, long timerEndTimeInMillis, PushNotificationData pushNotificationData) {
        if (WebEngageConstant.STYLE.PROGRESS_BAR.equals(pushNotificationData.getStyle())) {
            remoteViews.setViewVisibility(R.id.we_push_right_layout, View.GONE);
            remoteViews.setViewVisibility(R.id.we_progress_bar_timer, View.VISIBLE);
            setTimer(remoteViews, timerEndTimeInMillis, R.id.we_progress_bar_timer);
        } else {
            remoteViews.setViewVisibility(R.id.large_icon, View.GONE);
            remoteViews.setViewVisibility(R.id.we_notification_timer, View.VISIBLE);
            setTimer(remoteViews, timerEndTimeInMillis, R.id.we_notification_timer);
        }

    }

    public void setTimer(RemoteViews remoteViews, long timerEndTimeInMillis, int chronometerId) {
        long timeDiff = timerEndTimeInMillis - System.currentTimeMillis() + SystemClock.elapsedRealtime();
        remoteViews.setChronometer(chronometerId, timeDiff, WebEngageConstant.TIMER_FORMAT, true);
    }

    public void setProgressBar(RemoteViews remoteViews, long timerEndTimeInMillis, long when) {
        remoteViews.setViewVisibility(R.id.we_notification_progressBar, View.VISIBLE);
        remoteViews.setProgressBar(
                R.id.we_notification_progressBar,
                (int) (timerEndTimeInMillis - when),
                (int) (System.currentTimeMillis() - when),
                false
        );
    }

    public void setChronometerViewColor(RemoteViews remoteViews, Integer textColor, int chronometerId) {
        if (textColor != null) {
            remoteViews.setInt(chronometerId, "setTextColor", textColor);
        }
    }

    public void setBigTimerColor(RemoteViews remoteViews, Integer textColor) {
        if (textColor != null) {
            remoteViews.setInt(R.id.we_hr_1, "setTextColor", textColor);
            remoteViews.setInt(R.id.we_hr_2, "setTextColor", textColor);
            remoteViews.setInt(R.id.hr_min_delimiter, "setTextColor", textColor);
            remoteViews.setInt(R.id.we_min_1, "setTextColor", textColor);
            remoteViews.setInt(R.id.we_min_2, "setTextColor", textColor);
            remoteViews.setInt(R.id.min_sec_delimiter, "setTextColor", textColor);
            remoteViews.setInt(R.id.we_sec_1, "setTextColor", textColor);
        }
    }


    public void setProgressBarForegroundColor(RemoteViews remoteViews, Integer progressColor) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.S) {
            if (progressColor != null) {
                remoteViews.setColorStateList(R.id.we_notification_progressBar, "setProgressTintList", ColorStateList.valueOf(progressColor));
            }
        }
    }

    public void setProgressBarBackgroundColor(RemoteViews remoteViews, Integer progressColor) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.S) {
            if (progressColor != null) {
                remoteViews.setColorStateList(R.id.we_notification_progressBar, "setProgressBackgroundTintList", ColorStateList.valueOf(progressColor));
            }
        }
    }

    /**
     * For building notification
     *
     * @param mBuilder notification builder
     */
    Notification buildNotification(Notification.Builder mBuilder) {
        mBuilder.setAutoCancel(true);
        mBuilder.setOnlyAlertOnce(true);
        mBuilder.setShowWhen(true);
        Notification notification;
        notification = mBuilder.build();
        return notification;
    }

    /**
     * For showing timer notification
     *
     * @param timerEndTimeInMillis final epoch time
     * @param notificationId       notification id
     */
    void showNotification(Notification notification, Context context, long timerEndTimeInMillis, int notificationId) {
        NotificationManager notificationManager = (NotificationManager) context.getSystemService(Context.NOTIFICATION_SERVICE);
        if (System.currentTimeMillis() < timerEndTimeInMillis) {
            notificationManager.notify(notificationId, notification);
        }
    }

    /**
     * For showing notification
     *
     * @param notificationId notification id
     */
    void showNotification(Notification notification, Context context, int notificationId) {
        NotificationManager notificationManager = (NotificationManager) context.getSystemService(Context.NOTIFICATION_SERVICE);
        notificationManager.notify(notificationId, notification);
    }

    void setTitleMaxLines(RemoteViews remoteViews, int maxLines) {
        remoteViews.setBoolean(R.id.custom_title, "setSingleLine", false);
        remoteViews.setBoolean(R.id.custom_title_native, "setSingleLine", false);
        remoteViews.setInt(R.id.custom_title, "setMaxLines", maxLines);
        remoteViews.setInt(R.id.custom_title_native, "setMaxLines", maxLines);
    }

    void setDescriptionMaxLines(RemoteViews remoteViews, int maxLines) {
        remoteViews.setInt(R.id.custom_message, "setMaxLines", maxLines);
        remoteViews.setInt(R.id.custom_message_native, "setMaxLines", maxLines);
    }

    /**
     * @param timerEndTimeInMillis timer end time in millisecond
     * @return variance between nearest 9 second at end eg- timer 34 seconds return variance -5
     */
    int calculateVariance(long timerEndTimeInMillis) {
        long timeDiff = timerEndTimeInMillis - System.currentTimeMillis();
        //for getting digit place second
        int lastSecond = (int) ((timeDiff % WebEngageConstant.ONE_MINUTE) / WebEngageConstant.ONE_SECOND) % 10;
        int variance = 0;
        if (lastSecond != 9) {
            if (lastSecond < 5) {
                variance = -1 * (lastSecond + 1);
            } else {
                variance = 9 - lastSecond;
            }
        }
        return variance;
    }

    void setBigTimer(RemoteViews bigTimerView, boolean isCollapsed, long timerEndTimeInMillis, Context context, PushNotificationData pushNotificationData) {
        long timeDiff = timerEndTimeInMillis - System.currentTimeMillis();
        int hours = (int) (timeDiff / WebEngageConstant.ONE_HOUR);
        int minutes = (int) ((timeDiff % WebEngageConstant.ONE_HOUR) / WebEngageConstant.ONE_MINUTE);
        int seconds = (int) ((timeDiff % WebEngageConstant.ONE_MINUTE) / WebEngageConstant.ONE_SECOND);
        if (hours > 0) {
            bigTimerView.setTextViewText(R.id.we_hr_1, Integer.toString(hours / 10));
            bigTimerView.setTextViewText(R.id.we_hr_2, Integer.toString(hours % 10));
        } else {
            bigTimerView.setViewVisibility(R.id.we_hr_1, View.GONE);
            bigTimerView.setViewVisibility(R.id.we_hr_2, View.GONE);
            bigTimerView.setViewVisibility(R.id.hr_min_delimiter, View.GONE);
        }

        bigTimerView.setTextViewText(R.id.we_min_1, Integer.toString(minutes / 10));
        bigTimerView.setTextViewText(R.id.we_min_2, Integer.toString(minutes % 10));
        bigTimerView.setTextViewText(R.id.we_sec_1, Integer.toString(seconds / 10));

        setUpViewFlipperWithTime(
                isCollapsed,
                bigTimerView,
                R.id.we_sec_2,
                context,
                pushNotificationData
        );
    }

    private void setUpViewFlipperWithTime(
            boolean isCollapsed,
            RemoteViews expandedView,
            int layoutID, Context context, PushNotificationData pushNotificationData) {
        for (int i = 9; i >= 0; i--) {
            int flipperItemId;
            flipperItemId = R.layout.vf_expanded_layout;
            RemoteViews flipperItem = new RemoteViews(context.getPackageName(), flipperItemId);
            flipperItem.setTextViewText(R.id.tv_item, String.valueOf(i));
            if (isCollapsed) {
                flipperItem.setTextViewTextSize(R.id.tv_item, TypedValue.COMPLEX_UNIT_SP, 25);
            }
            if (pushNotificationData.getTimerStyleData().getTimerColor() != Color.parseColor(WebEngageConstant.WE_TRANSPARENT_COLOR)) {
                flipperItem.setInt(R.id.tv_item, "setTextColor", pushNotificationData.getTimerStyleData().getTimerColor());
            }
            expandedView.addView(layoutID, flipperItem);
        }
        expandedView.setInt(layoutID, "setFlipInterval", 1000);
    }

    //returns true if buttons are present and vice versa
    boolean areButtonsPresent(PushNotificationData pushNotificationData) {
        List<CallToAction> callToActions = pushNotificationData.getCallToActions();
        if (callToActions != null && callToActions.size() > 0) {
            for (CallToAction callToAction : callToActions) {
                if (!callToAction.isPrimeAction() && callToAction.isNative()) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * for scheduling alarm for notification re render
     */
    @SuppressLint("MissingPermission")
    void scheduleProgressBarNotification(Context applicationContext, PushNotificationData pushNotificationData, long when, long timerEndTimeInMillis, boolean dueToServiceFailure) {

        long alarmInterval = (timerEndTimeInMillis - when) / 100;
        //setting alarm interval to 5 sec if less than 5 second(min interval for subsequent alarms)
        if (alarmInterval < WebEngageConstant.MINIMUM_UPDATE_INTERVAL_ALARM) {
            alarmInterval = WebEngageConstant.MINIMUM_UPDATE_INTERVAL_ALARM;
        }
        AlarmManager alarmManager = (AlarmManager) applicationContext.getSystemService(Context.ALARM_SERVICE);
        Bundle bundle = new Bundle();
        bundle.putLong(WebEngageConstant.WHEN, when);
        bundle.putBoolean(WebEngageConstant.WE_RENDER, true);
        if (dueToServiceFailure) {
            Logger.d(WebEngageConstant.TAG, "Schedule Progress Bar Notification using Alarm due to Service Failure");
            bundle.putBoolean("SERVICE_FAILED", true);
        }
        PendingIntent pendingIntent = PendingIntentFactory.constructProgressBarRerenderPendingIntent(applicationContext, pushNotificationData, "progressbar_update", bundle);
        //setting exact and in exact based on the permissions and OS version
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.S
                && (!ManifestUtils.checkPermission(ManifestUtils.SCHEDULE_EXACT_ALARM, applicationContext)
                || applicationContext.checkSelfPermission(ManifestUtils.SCHEDULE_EXACT_ALARM) != PackageManager.PERMISSION_GRANTED)) {
            alarmManager.set(AlarmManager.RTC, System.currentTimeMillis() + alarmInterval, pendingIntent);
        } else {
            alarmManager.setExact(AlarmManager.RTC, System.currentTimeMillis() + alarmInterval, pendingIntent);
        }
    }

    public void setBitmapToRemoteView(Bitmap bitmap, RemoteViews remoteView, int remoteViewId) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            remoteView.setImageViewIcon(remoteViewId, Icon.createWithBitmap(bitmap));
        } else {
            remoteView.setImageViewBitmap(remoteViewId, bitmap);
        }
    }

    public boolean isNotificationChannelEnabled(Context context, String channelId){
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            if(!TextUtils.isEmpty(channelId)) {
                NotificationManager manager = (NotificationManager) context.getSystemService(Context.NOTIFICATION_SERVICE);
                NotificationChannel channel = manager.getNotificationChannel(channelId);
                return channel.getImportance() != NotificationManager.IMPORTANCE_NONE;
            }
            return false;
        } else {
            return NotificationManagerCompat.from(context).areNotificationsEnabled();
        }
    }

}
