package com.webengage.sdk.android.actions.render;

import android.app.Notification;
import android.content.Context;
import android.content.Intent;
import android.graphics.Bitmap;
import android.graphics.Color;
import android.os.Build;
import android.os.Bundle;
import android.os.SystemClock;
import android.view.View;
import android.widget.RemoteViews;

import com.webengage.sdk.android.EventFactory;
import com.webengage.sdk.android.EventName;
import com.webengage.sdk.android.IntentFactory;
import com.webengage.sdk.android.Logger;
import com.webengage.sdk.android.PushUtils;
import com.webengage.sdk.android.R;
import com.webengage.sdk.android.Topic;
import com.webengage.sdk.android.WebEngage;
import com.webengage.sdk.android.callbacks.CustomPushRender;
import com.webengage.sdk.android.callbacks.CustomPushRerender;
import com.webengage.sdk.android.utils.ManifestUtils;
import com.webengage.sdk.android.utils.WebEngageConstant;
import com.webengage.sdk.android.utils.htmlspanner.WEHtmlParserInterface;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;

public class TimerRenderer extends PushRenderer implements CustomPushRender, CustomPushRerender {
    private long timerEndTimeInMillis = 0;
    long timeDiff = 0;
    NotificationConfigurator notificationConfigurator = null;

    @Override
    void downloadImages() {
    }

    @Override
    void loadImages() {
        if (pushNotificationData.getTimerStyleData() != null && pushNotificationData.getTimerStyleData().getImageUrl() != null) {
            try {
                Bitmap bitmap = notificationConfigurator.loadImage(pushNotificationData.getTimerStyleData().getImageUrl(), applicationContext);
                if (bitmap != null) {
                    validImages.add(bitmap);
                }
            } catch (Exception e) {
                Logger.d(WebEngageConstant.TAG, "Timer renderer can not load image with exception ->" + e);
            }
        }
    }

    @Override
    void buildExpandedPush() {
        if (pushNotificationData.getStyle() != null && pushNotificationData.getTimerStyleData() != null) {
            mBuilder.setWhen(when);
            customBigView = constructExpandedPushBase();
            RemoteViews bigPictureView = null;
            if (WebEngageConstant.STYLE.TIMER.equals(pushNotificationData.getStyle()) ||
                    WebEngageConstant.STYLE.PROGRESS_BAR.equals(pushNotificationData.getStyle())) {
                bigPictureView = new RemoteViews(this.applicationContext.getPackageName(), R.layout.timer_layout);
                if (validImages.size() > 0) {
                    bigPictureView.setViewVisibility(R.id.big_picture_image, View.VISIBLE);
                    setBitmapToRemoteView(validImages.get(0), bigPictureView, R.id.big_picture_image);
                }
                if (WebEngageConstant.STYLE.PROGRESS_BAR.equals(pushNotificationData.getStyle()))
                    notificationConfigurator.setDescriptionMaxLines(customBigView, 1);
            } else if (WebEngageConstant.STYLE.BIG_TIMER.equals(pushNotificationData.getStyle())) {
                bigPictureView = new RemoteViews(this.applicationContext.getPackageName(), R.layout.big_timer);
                //setting image to big icon
                if (validImages.size() > 0) {
                    setBitmapToRemoteView(validImages.get(0), customBigView, R.id.large_icon);
                    if (applicationContext.getApplicationInfo().targetSdkVersion >= Build.VERSION_CODES.S && Build.VERSION.SDK_INT >= Build.VERSION_CODES.S
                            && pushNotificationData.getTimerStyleData().getProgressBarColor() != Color.parseColor(WebEngageConstant.WE_TRANSPARENT_COLOR)) {
                        int padding = applicationContext.getResources().getDimensionPixelSize(R.dimen.we_push_image_margin_colorbg);
                        customBigView.setViewPadding(R.id.large_icon, 0, padding, 0, 0);
                    }
                }
                bigPictureView.setViewVisibility(R.id.we_big_timer_expanded_layout, View.GONE);
                bigPictureView.setViewVisibility(R.id.we_notification_big_timer_expanded, View.VISIBLE);
                notificationConfigurator.setTimer(bigPictureView, timerEndTimeInMillis, R.id.we_notification_big_timer_expanded);
                setChronometerColor(bigPictureView, R.id.we_notification_big_timer_expanded);
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                    if (timerEndTimeInMillis - System.currentTimeMillis() > 0)
                        mBuilder.setTimeoutAfter((timerEndTimeInMillis - System.currentTimeMillis()));
                }
            }

            if (bigPictureView != null) {
                if (!areButtonsPresent()) {
                    bigPictureView.setViewVisibility(R.id.push_action_margin_view, View.VISIBLE);
                }
                customBigView.removeAllViews(R.id.custom_base_container);
                customBigView.addView(R.id.custom_base_container, bigPictureView);
            }

            notificationConfigurator.addIntents(customBigView, mBuilder, pushNotificationData, applicationContext);
        }
    }

    @Override
    public boolean onRender(Context context, PushNotificationData pushNotificationData) {
        init(pushNotificationData, null);
        if (timerEndTimeInMillis - System.currentTimeMillis() <= 0) {
            logPushFailureEvent(context, WebEngageConstant.FAILURE_REASONS.TIMER_DATE_EXPIRED);
            return false;
        }
        return super.onRender(context, pushNotificationData);
    }

    //used for initializing notificationConfigurator, push data, when, and epoch time
    private void init(PushNotificationData pushNotificationData, Bundle extras) {
        if (notificationConfigurator == null)
            notificationConfigurator = new NotificationConfigurator();
        this.pushNotificationData = pushNotificationData;
        if (extras != null) {
            when = extras.getLong(WebEngageConstant.WHEN, System.currentTimeMillis());
        }
        if (when == null) {
            when = System.currentTimeMillis();
        }
        calculateEndTimeInMillisecond();

    }

    //logging push failure event
    private void logPushFailureEvent(Context context, WebEngageConstant.FAILURE_REASONS failureReason) {
        Map<String, Object> eventData = new HashMap<>();
        eventData.put(WebEngageConstant.WE_ERROR_CODE, failureReason.getErrorCode());
        eventData.put(WebEngageConstant.WE_FAILURE_REASON, failureReason.toString());
        eventData.put(WebEngageConstant.WE_FAILURE_MESSAGE, failureReason.getErrorMessage());
        eventData.put(WebEngageConstant.AMPLIFIED, pushNotificationData.isAmplified());
        Map<String, Object> systemData = new HashMap<>();
        systemData.put(WebEngageConstant.EXPERIMENT_ID, pushNotificationData.getExperimentId());
        systemData.put(WebEngageConstant.NOTIFICATION_ID, pushNotificationData.getVariationId());
        Intent intent = IntentFactory.newIntent(Topic.EVENT, EventFactory.newSystemEvent(EventName.PUSH_NOTIFICATION_FAILED, systemData, eventData, null, context), context);
        WebEngage.startService(intent, context);
        Logger.e(WebEngageConstant.TAG, "Timer Push failed to render with reason : " + failureReason.getErrorMessage());
    }

    //for calculating end time
    private void calculateEndTimeInMillisecond() {
        PushNotificationData.TimerStyle timerStyleData = pushNotificationData.getTimerStyleData();
        //if future time is available in payload(yyyy-MM-dd HH:mm:ss format)
        if (timerStyleData.getFutureTime() != null && !timerStyleData.getFutureTime().isEmpty()) {
            String dateString = timerStyleData.getFutureTime();
            SimpleDateFormat sdf = new SimpleDateFormat(WebEngageConstant.FUTURE_TIME_DATE_FORMAT, Locale.US);
            try {
                sdf.setTimeZone(Objects.requireNonNull(PushUtils.getTimeZone(pushNotificationData)));
                Date date = sdf.parse(dateString);
                timerEndTimeInMillis = date.getTime();
            } catch (Exception e) {
                logPushFailureEvent(applicationContext, WebEngageConstant.FAILURE_REASONS.UNKNOWN_SDK_FAILURE);
                Logger.e(WebEngageConstant.TAG, "Could not parse futureTime format, push with experimentId = " + pushNotificationData.getExperimentId() + " will not render");
            }
        }
        //if duration is available in payload(hh:mm format)
        else if (timerStyleData.getDuration() != null && !timerStyleData.getDuration().isEmpty()) {
            int timeInMinutes;
            String[] parts = timerStyleData.getDuration().split(":");
            if (parts.length == 2) {
                try {
                    int hours = Integer.parseInt(parts[0]);
                    int minutes = Integer.parseInt(parts[1]);
                    timeInMinutes = hours * 60 + minutes;
                    timerEndTimeInMillis = when + (timeInMinutes * WebEngageConstant.ONE_MINUTE);
                } catch (NumberFormatException e) {
                    logPushFailureEvent(applicationContext, WebEngageConstant.FAILURE_REASONS.UNKNOWN_SDK_FAILURE);
                    Logger.e(WebEngageConstant.TAG, "Could not parse duration format, push with experimentId = " + pushNotificationData.getExperimentId() + " will not render");
                }
            } else {
                logPushFailureEvent(applicationContext, WebEngageConstant.FAILURE_REASONS.UNKNOWN_SDK_FAILURE);
                Logger.e(WebEngageConstant.TAG, "Could not parse duration format, push with experimentId = " + pushNotificationData.getExperimentId() + " will not render");
            }
        }
        //if both future time and duration not present
        else {
            logPushFailureEvent(applicationContext, WebEngageConstant.FAILURE_REASONS.UNKNOWN_SDK_FAILURE);
            Logger.e(WebEngageConstant.TAG, "FutureTime & Duration not found, push with experimentId = " + pushNotificationData.getExperimentId() + " will not render");
        }
    }

    private RemoteViews constructExpandedPushBase() {
        RemoteViews pushBase = notificationConfigurator.getCustomPushBase(applicationContext, pushNotificationData, when, R.layout.timer_push_base);
        notificationConfigurator.setTitleMaxLines(pushBase, 1);
        notificationConfigurator.setDescriptionMaxLines(pushBase, 2);
        pushBase.setViewVisibility(R.id.custom_base_container, View.VISIBLE);
        PushNotificationData.TimerStyle timerStyle = pushNotificationData.getTimerStyleData();
        if (WebEngageConstant.STYLE.TIMER.equals(pushNotificationData.getStyle()) || WebEngageConstant.STYLE.PROGRESS_BAR.equals(pushNotificationData.getStyle())) {
            pushBase.setViewVisibility(R.id.large_icon, View.GONE);
            pushBase.setViewVisibility(R.id.we_notification_timer, View.VISIBLE);
            setTimer(timerStyle, pushBase);
        }
        return pushBase;
    }

    /**
     * For setting timer and progress bar
     */
    private void setTimer(PushNotificationData.TimerStyle timerStyle, RemoteViews pushBase) {
        long currentTime = System.currentTimeMillis();
        timeDiff = timerEndTimeInMillis - currentTime;

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            mBuilder.setTimeoutAfter(timeDiff);
        }
        long chronometerTime = timeDiff + SystemClock.elapsedRealtime();
        int chronometerId = R.id.we_notification_timer;
        if (WebEngageConstant.STYLE.PROGRESS_BAR.equals(pushNotificationData.getStyle())) {
            chronometerId = R.id.we_progress_bar_timer;
            pushBase.setViewVisibility(R.id.we_push_right_layout, View.GONE);
            pushBase.setViewVisibility(R.id.we_progress_bar_timer, View.VISIBLE);
            pushBase.setChronometer(
                    R.id.we_progress_bar_timer,
                    chronometerTime,
                    WebEngageConstant.TIMER_FORMAT,
                    true
            );
            pushBase.setViewVisibility(R.id.we_notification_progressBar, View.VISIBLE);
            notificationConfigurator.setProgressBarForegroundColor(
                    pushBase,
                    timerStyle.getProgressBarColor()
            );

            notificationConfigurator.setProgressBarBackgroundColor(
                    pushBase,
                    timerStyle.getProgressBarBackgroundColor()
            );

            pushBase.setProgressBar(
                    R.id.we_notification_progressBar,
                    (int) (timerEndTimeInMillis - when),
                    (int) (System.currentTimeMillis() - when),
                    false
            );
        } else if (WebEngageConstant.STYLE.TIMER.equals(pushNotificationData.getStyle())) {
            pushBase.setViewVisibility(R.id.large_icon, View.GONE);
            pushBase.setViewVisibility(R.id.we_notification_timer, View.VISIBLE);
            pushBase.setChronometer(
                    R.id.we_notification_timer,
                    chronometerTime,
                    WebEngageConstant.TIMER_FORMAT,
                    true
            );

        }
        setChronometerColor(pushBase, chronometerId);
    }

    //onReRender is triggered only for alarm scheduled messages.
    @Override
    public boolean onRerender(Context context, PushNotificationData pushNotificationData, Bundle extras) {
        init(pushNotificationData, extras);
        if (notificationConfigurator == null)
            notificationConfigurator = new NotificationConfigurator();

        if (timerEndTimeInMillis - System.currentTimeMillis() <= 0) {
            Logger.e(WebEngageConstant.TAG, "Timer end time (" + timerEndTimeInMillis + ") has passed current time (" + System.currentTimeMillis() + ")push with experimentId = " + pushNotificationData.getExperimentId() + " will not render");
            return false;
        }
        if ((System.currentTimeMillis() - when) >= 0) {
            super.onRerender(context, pushNotificationData, null);
            if (!extras.getBoolean(WebEngageConstant.SNOOZE, false) && WebEngageConstant.STYLE.PROGRESS_BAR.equals(pushNotificationData.getStyle()))
                notificationConfigurator.scheduleProgressBarNotification(applicationContext, pushNotificationData, when, timerEndTimeInMillis, false);
        }
        return true;
    }

    @Override
    void buildCollapsedPush() {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            String channelId = notificationConfigurator.getChannelId(pushNotificationData, applicationContext);
            mBuilder = new Notification.Builder(this.applicationContext, channelId);
        } else {
            mBuilder = new Notification.Builder(this.applicationContext);
        }
        mBuilder.setSmallIcon(pushNotificationData.getSmallIcon());
        if (pushNotificationData.getContentSummary() != null) {
            mBuilder.setSubText(new WEHtmlParserInterface().fromHtml(pushNotificationData.getContentSummary()));
        }
        RemoteViews pushBase = notificationConfigurator.getCustomPushBase(applicationContext, pushNotificationData, when, R.layout.timer_push_base);
        PushNotificationData.TimerStyle timerStyle = pushNotificationData.getTimerStyleData();
        notificationConfigurator.setTitleMaxLines(pushBase, 1);
        notificationConfigurator.setDescriptionMaxLines(pushBase, 1);
        if (timerStyle != null) {
            if (WebEngageConstant.STYLE.TIMER.equals(pushNotificationData.getStyle()) || WebEngageConstant.STYLE.PROGRESS_BAR.equals(pushNotificationData.getStyle())) {
                if (WebEngageConstant.STYLE.PROGRESS_BAR.equals(pushNotificationData.getStyle())) {
                    pushBase.setViewVisibility(R.id.custom_message, View.GONE);
                    pushBase.setViewVisibility(R.id.custom_message_native, View.GONE);
                }
                setTimer(timerStyle, pushBase);
            } else {
                pushBase.setViewVisibility(R.id.large_icon, View.GONE);
                pushBase.setViewVisibility(R.id.we_notification_big_timer_collapsed, View.VISIBLE);
                setChronometerColor(pushBase, R.id.we_notification_big_timer_collapsed);
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                    if (timerEndTimeInMillis - System.currentTimeMillis() > 0)
                        mBuilder.setTimeoutAfter((timerEndTimeInMillis - System.currentTimeMillis()));
                }
                notificationConfigurator.setTimer(pushBase, timerEndTimeInMillis, R.id.we_notification_big_timer_collapsed);
            }
        }
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            mBuilder.setCustomContentView(pushBase);
        } else {
            mBuilder.setContent(pushBase);
        }

        mBuilder.setContentTitle(new WEHtmlParserInterface().fromHtml(pushNotificationData.getTitle()))
                .setContentText(new WEHtmlParserInterface().fromHtml(pushNotificationData.getContentText()));

        if (pushNotificationData.isSticky())
            mBuilder.setOngoing(true);

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            mBuilder.setVisibility(Notification.VISIBILITY_PRIVATE);
        }

    }

    @Override
    void show() {
        if (pushNotificationData.getTimerStyleData() != null
                && (WebEngageConstant.STYLE.PROGRESS_BAR.equals(pushNotificationData.getStyle())
                || WebEngageConstant.STYLE.BIG_TIMER.equals(pushNotificationData.getStyle()))) {

            int sdkInt = Build.VERSION.SDK_INT;
            int targetSdkVersion = applicationContext.getApplicationInfo().targetSdkVersion;
            boolean hasForegroundServicePermission = ManifestUtils.checkPermission(ManifestUtils.FOREGROUND_SERVICE, applicationContext);
            //Below Android 9 there is no restriction in starting foreground service
            boolean checksForAndroid8 = sdkInt < Build.VERSION_CODES.P;
            //In Android 9 and above FOREGROUND_SERVICE permission is mandatory
            // for starting foreground services
            boolean checksForAndroid9to13 = sdkInt < Build.VERSION_CODES.UPSIDE_DOWN_CAKE && hasForegroundServicePermission;
            //In Android 14 and above FOREGROUND_SERVICE permission and Type of Foreground service
            // is mandatory for starting foreground services
            boolean checksForAndroid14 = targetSdkVersion >= Build.VERSION_CODES.UPSIDE_DOWN_CAKE &&
                    hasForegroundServicePermission &&
                    PushUtils.isForegroundServiceTypeAdded(applicationContext);

            if (checksForAndroid8 || checksForAndroid9to13 || checksForAndroid14) {
                startTimerService();
                return;
            } else {
                if (WebEngageConstant.STYLE.PROGRESS_BAR.equals(pushNotificationData.getStyle())) {
                    notificationConfigurator.scheduleProgressBarNotification(applicationContext, pushNotificationData, when, timerEndTimeInMillis, false);
                }
            }
        }
        super.show();
    }

    /**
     * for starting timer service
     */
    private void startTimerService() {
        Intent intent = new Intent(applicationContext, TimerService.class);
        intent.setAction(WebEngageConstant.PROGRESS_BAR_ACTION);
        intent.putExtra(WebEngageConstant.DATA, pushNotificationData.getPushPayloadJSON().toString());
        intent.putExtra(WebEngageConstant.WHEN, when);
        intent.putExtra(WebEngageConstant.TIMER_END_TIME_IN_MILLIS, timerEndTimeInMillis);
        try {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                applicationContext.startForegroundService(intent);
            } else {
                applicationContext.startService(intent);
            }
        } catch (Exception exception) {
            notificationConfigurator.scheduleProgressBarNotification(applicationContext, pushNotificationData, when, timerEndTimeInMillis, true);
        }
    }

    private void setChronometerColor(RemoteViews remoteViews, int chronometerId) {
        notificationConfigurator.setChronometerViewColor(remoteViews, pushNotificationData.getTimerStyleData().getTimerColor(), chronometerId);
    }

}
