package com.webengage.sdk.android.actions.render;

import android.app.Notification;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.app.Service;
import android.content.Context;
import android.content.Intent;
import android.graphics.Bitmap;
import android.graphics.Color;
import android.os.Build;
import android.os.CountDownTimer;
import android.os.IBinder;
import android.view.View;
import android.widget.RemoteViews;

import androidx.annotation.Nullable;

import com.webengage.sdk.android.Logger;
import com.webengage.sdk.android.PendingIntentFactory;
import com.webengage.sdk.android.R;
import com.webengage.sdk.android.utils.WebEngageConstant;

import org.json.JSONException;
import org.json.JSONObject;

public class TimerService extends Service {

    private long timerEndTimeInMillis = 0;
    private CountDownTimer countDownTimer = null;
    private PushNotificationData pushNotificationData;
    private long when;
    private Notification.Builder mBuilder = null;
    private Context context = null;
    private NotificationConfigurator notificationConfigurator = null;
    boolean hasBackgroundColor = false;
    Notification notification = null;

    @Override
    public int onStartCommand(Intent intent, int flags, int startId) {
        if (intent != null && intent.getAction().equals(WebEngageConstant.PROGRESS_BAR_ACTION)) {
            //only the latest notification will be visible
            if (countDownTimer != null && pushNotificationData != null) {
                stopForeground(true);
                if (pushNotificationData != null) {
                    NotificationManager notificationManager = (NotificationManager) this.getSystemService(Context.NOTIFICATION_SERVICE);
                    notificationManager.cancel(pushNotificationData.getVariationId().hashCode());
                }
                countDownTimer.cancel();
            }
            this.context = getApplicationContext();
            try {
                pushNotificationData = new PushNotificationData(new JSONObject(intent.getExtras().getString(WebEngageConstant.DATA)), context);
            } catch (JSONException e) {
                Logger.e(WebEngageConstant.TAG, "TimerService : Failed to parse pushNotificationData json");
                return START_STICKY;
            }
            if (notificationConfigurator == null)
                notificationConfigurator = new NotificationConfigurator();
            String channelId = notificationConfigurator.getChannelId(pushNotificationData, context);
            if (Build.VERSION.SDK_INT >= 26) {
                mBuilder = new Notification.Builder(context, channelId);
            } else {
                mBuilder = new Notification.Builder(context);
            }
            when = intent.getExtras().getLong(WebEngageConstant.WHEN);
            timerEndTimeInMillis = intent.getLongExtra(WebEngageConstant.TIMER_END_TIME_IN_MILLIS, 0);
            hasBackgroundColor = pushNotificationData.getBackgroundColor() != Color.parseColor(WebEngageConstant.WE_TRANSPARENT_COLOR);
            if (timerEndTimeInMillis - System.currentTimeMillis() > 0) {
                initializeNotification();
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.S) {
                    mBuilder.setForegroundServiceBehavior(Notification.FOREGROUND_SERVICE_IMMEDIATE);
                }
                mBuilder.setAutoCancel(true);
                mBuilder.setOnlyAlertOnce(true);
                try {
                    startForeground(
                            pushNotificationData.getVariationId().hashCode(),
                            mBuilder.build()
                    );
                    startCounter();
                } catch (IllegalStateException illegalStateException) {
                    new NotificationConfigurator().scheduleProgressBarNotification(context, pushNotificationData, when, timerEndTimeInMillis, true);
                }
            }
        }
        return START_STICKY;
    }

    void initializeNotification() {
        constructNotification();
    }

    /**
     * Start a CountDownTimer to update the notification periodically at a delay of updateFrequency.
     * The Timer will run till the System time becomes greater than the provided future time OR
     * if the service has been stopped.
     */
    private void startCounter() {
        long currentTime = System.currentTimeMillis();
        long timeDiff = timerEndTimeInMillis - currentTime;
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            mBuilder.setTimeoutAfter(timeDiff);
        }
        PendingIntent dismissIntent = PendingIntentFactory.constructPushDeletePendingIntent(context, pushNotificationData);
        //interval is 1 second in progress bar re render
        long interval = WebEngageConstant.ONE_SECOND;
        //interval is 10 second in big timer re render
        if (WebEngageConstant.STYLE.BIG_TIMER.equals(pushNotificationData.getStyle()))
            interval = 10 * WebEngageConstant.ONE_SECOND;

        countDownTimer = new CountDownTimer(
                timeDiff,
                interval) {
            @Override
            public void onTick(long l) {
                if ((System.currentTimeMillis() - when) > 0) {
                    constructNotification();
                    notificationConfigurator.showNotification(notification, context, timerEndTimeInMillis, pushNotificationData.getVariationId().hashCode());
                } else {
                    try {
                        dismissIntent.send();
                    } catch (PendingIntent.CanceledException e) {
                    }
                    stopSelf();
                }
            }

            @Override
            public void onFinish() {
                try {
                    dismissIntent.send();
                } catch (PendingIntent.CanceledException e) {
                }
                stopSelf();
            }

        };
        countDownTimer.start();

    }

    void constructNotification() {
        //notification creation logic
        mBuilder.setProgress(
                (int) (timerEndTimeInMillis - when),
                (int) (System.currentTimeMillis() - when),
                false
        );
        notificationConfigurator.buildNotificationBuilder(mBuilder, pushNotificationData, context);
        mBuilder.setOngoing(true);
        mBuilder.setWhen(when);
        //attaching collapsed remote view to builder
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            mBuilder.setCustomContentView(constructCollapsedTimerPushBase());
        } else {
            mBuilder.setContent(constructCollapsedTimerPushBase());
        }
        notification = notificationConfigurator.buildNotification(mBuilder);
        //attaching expanded remote view to notification
        if (Build.VERSION.SDK_INT <= 23) {
            notification.bigContentView = constructExpandedTimerPushBase();
        } else {
            notification = mBuilder.setCustomBigContentView(constructExpandedTimerPushBase()).build();
        }
    }

    //building expanded timer base
    private RemoteViews constructExpandedTimerPushBase() {
        RemoteViews customBigView = notificationConfigurator.getCustomPushBase(context, pushNotificationData, when, R.layout.timer_push_base);
        customBigView.setOnClickPendingIntent(R.id.push_base_container, PendingIntentFactory.constructPushClickPendingIntent(context, pushNotificationData, pushNotificationData.getPrimeCallToAction(), true));
        notificationConfigurator.setTitleMaxLines(customBigView, 1);
        Bitmap bitmap = notificationConfigurator.loadImage(pushNotificationData.getTimerStyleData().getImageUrl(), context);
        RemoteViews bigPictureView = new RemoteViews(context.getPackageName(), R.layout.timer_layout);
        if (WebEngageConstant.STYLE.PROGRESS_BAR.equals(pushNotificationData.getStyle())) {
            notificationConfigurator.setDescriptionMaxLines(customBigView, 1);
            customBigView.setViewVisibility(R.id.custom_base_container, View.VISIBLE);
            if (bitmap != null) {
                bigPictureView.setViewVisibility(R.id.big_picture_image, View.VISIBLE);
                notificationConfigurator.setBitmapToRemoteView(bitmap, bigPictureView, R.id.big_picture_image);
            }
        } else if (WebEngageConstant.STYLE.BIG_TIMER.equals(pushNotificationData.getStyle())) {
            notificationConfigurator.setDescriptionMaxLines(customBigView, 2);
            customBigView.setViewVisibility(R.id.custom_base_container, View.VISIBLE);
            bigPictureView = new RemoteViews(context.getPackageName(), R.layout.big_timer);
            //setting image to big icon
            if (bitmap != null) {
                notificationConfigurator.setBitmapToRemoteView(bitmap, customBigView, R.id.large_icon);
                if (context.getApplicationInfo().targetSdkVersion >= Build.VERSION_CODES.S && Build.VERSION.SDK_INT >= Build.VERSION_CODES.S
                        && pushNotificationData.getTimerStyleData().getProgressBarColor() != Color.parseColor(WebEngageConstant.WE_TRANSPARENT_COLOR)) {
                    int padding = context.getResources().getDimensionPixelSize(R.dimen.we_push_image_margin_colorbg);
                    customBigView.setViewPadding(R.id.large_icon, 0, padding, 0, 0);
                }
            }
            //the variance between nearest 9 at timer end time seconds' digit place
            int variance = notificationConfigurator.calculateVariance(timerEndTimeInMillis);
            //adjusting timer end time according to variance
            timerEndTimeInMillis += ((long) variance * WebEngageConstant.ONE_SECOND);
            notificationConfigurator.setBigTimer(bigPictureView, false, timerEndTimeInMillis, context, pushNotificationData);
            notificationConfigurator.setBigTimerColor(bigPictureView, pushNotificationData.getTimerStyleData().getTimerColor());
        }
        if (!notificationConfigurator.areButtonsPresent(pushNotificationData)) {
            bigPictureView.setViewVisibility(R.id.push_action_margin_view, View.VISIBLE);
        }
        customBigView.removeAllViews(R.id.custom_base_container);
        customBigView.addView(R.id.custom_base_container, bigPictureView);
        notificationConfigurator.addIntents(bigPictureView, mBuilder, pushNotificationData, context);
        if (WebEngageConstant.STYLE.PROGRESS_BAR.equals(pushNotificationData.getStyle())) {
            notificationConfigurator.setProgressBar(customBigView, timerEndTimeInMillis, when);
            notificationConfigurator.setProgressBarForegroundColor(customBigView, pushNotificationData.getTimerStyleData().getProgressBarColor());
            notificationConfigurator.setProgressBarBackgroundColor(customBigView, pushNotificationData.getTimerStyleData().getProgressBarBackgroundColor());
            notificationConfigurator.setChronometer(customBigView, timerEndTimeInMillis, pushNotificationData);
            notificationConfigurator.setChronometerViewColor(customBigView, pushNotificationData.getTimerStyleData().getTimerColor(), R.id.we_progress_bar_timer);
        }
        return customBigView;
    }

    //building collapsed timer base
    private RemoteViews constructCollapsedTimerPushBase() {
        RemoteViews remoteView = notificationConfigurator.getCustomPushBase(context, pushNotificationData, when, R.layout.timer_push_base);
        remoteView.setOnClickPendingIntent(R.id.push_base_container, PendingIntentFactory.constructPushClickPendingIntent(context, pushNotificationData, pushNotificationData.getPrimeCallToAction(), true));
        if (WebEngageConstant.STYLE.PROGRESS_BAR.equals(pushNotificationData.getStyle())) {
            //hiding message in case of collapsed push
            remoteView.setViewVisibility(R.id.custom_message, View.GONE);
            remoteView.setViewVisibility(R.id.custom_message_native, View.GONE);
            notificationConfigurator.setProgressBar(remoteView, timerEndTimeInMillis, when);
            notificationConfigurator.setChronometer(remoteView, timerEndTimeInMillis, pushNotificationData);
            notificationConfigurator.setChronometerViewColor(remoteView, pushNotificationData.getTimerStyleData().getTimerColor(), R.id.we_progress_bar_timer);
            notificationConfigurator.setProgressBarForegroundColor(remoteView, pushNotificationData.getTimerStyleData().getProgressBarColor());
            notificationConfigurator.setProgressBarBackgroundColor(remoteView, pushNotificationData.getTimerStyleData().getProgressBarBackgroundColor());
        } else if (WebEngageConstant.STYLE.BIG_TIMER.equals(pushNotificationData.getStyle())) {
            remoteView.setViewVisibility(R.id.we_big_timer_collapsed, View.VISIBLE);
            remoteView.setViewVisibility(R.id.large_icon, View.GONE);
            //the variance between nearest 9 at timer end time seconds' digit place
            int variance = notificationConfigurator.calculateVariance(timerEndTimeInMillis);
            //adjusting timer end time according to variance
            timerEndTimeInMillis += ((long) variance * WebEngageConstant.ONE_SECOND);
            notificationConfigurator.setBigTimer(remoteView, true, timerEndTimeInMillis, context, pushNotificationData);
            notificationConfigurator.setBigTimerColor(remoteView, pushNotificationData.getTimerStyleData().getTimerColor());
        }
        notificationConfigurator.setTitleMaxLines(remoteView, 1);
        notificationConfigurator.setDescriptionMaxLines(remoteView, 1);
        remoteView.setViewVisibility(R.id.push_base_margin_view, View.VISIBLE);
        return remoteView;
    }

    @Override
    public void onDestroy() {
        stopForeground(true);
        if (pushNotificationData != null) {
            NotificationManager notificationManager = (NotificationManager) this.getSystemService(Context.NOTIFICATION_SERVICE);
            notificationManager.cancel(pushNotificationData.getVariationId().hashCode());
        }
        if (countDownTimer != null)
            countDownTimer.cancel();
        super.onDestroy();
    }

    @Nullable
    @Override
    public IBinder onBind(Intent intent) {
        return null;
    }

}
