package com.webengage.sdk.android.utils.http;

import android.content.Context;
import android.text.TextUtils;

import androidx.annotation.RestrictTo;

import com.webengage.sdk.android.CallbackDispatcher;
import com.webengage.sdk.android.Logger;
import com.webengage.sdk.android.utils.WebEngageConstant;

import java.util.Map;

public class RequestObject {
    public static final int FLAG_PERSIST_AFTER_CONFIG_REFRESH = 1;

    private String proxyUrl;
    private final String url;
    private final RequestMethod method;
    private Map<String, String> headers;
    private final Object params;
    private final String tag;
    private int flags;
    private final int cachePolicy;
    private Context context;

    private RequestObject(Builder builder) {
        this.proxyUrl = builder.proxyUrl;
        this.url = builder.url;
        this.method = builder.method;
        this.headers = builder.headers;
        this.params = builder.params;
        this.tag = builder.tag;
        this.flags = builder.flags;
        this.cachePolicy = builder.cachePolicy;
        this.context = builder.context;
    }

    public static class Builder {
        private String proxyUrl;
        private final String url;
        private final RequestMethod method;
        private Map<String, String> headers = null;
        private Object params = null;
        private String tag = null;
        private Context context = null;
        private int flags = 0;
        private int cachePolicy = CachePolicy.GET_DATA_FROM_NETWORK_ONLY_NO_CACHING;

        public Builder(String url, RequestMethod method, Context context) {
            this.url = url;
            this.method = method;
            this.context = context.getApplicationContext();
        }

        public Builder setHeaders(Map<String, String> headers) {
            this.headers = headers;
            return this;
        }

        public Builder setParams(Object params) {
            this.params = params;
            return this;
        }

        @RestrictTo(RestrictTo.Scope.LIBRARY)
        public Builder setTag(String tag) {
            this.tag = tag;
            return this;
        }

        @RestrictTo(RestrictTo.Scope.LIBRARY)
        public Builder setFlags(int flags) {
            this.flags |= flags;
            return this;
        }

        public Builder setCachePolicy(int policy) {
            this.cachePolicy = policy;
            return this;
        }

        protected Builder setProxyURL(String proxyUrl) {
            this.proxyUrl = proxyUrl;
            return this;
        }

        public RequestObject build() {
            return new RequestObject(this);
        }
    }


    protected String getURL() {
        return this.url;
    }

    public RequestMethod getRequestMethod() {
        return this.method;
    }

    public Map<String, String> getHeaders() {
        return this.headers;
    }

    public Object getParams() {
        return this.params;
    }

    protected String getTag() {
        return this.tag;
    }

    protected int getFlags() {
        return this.flags;
    }

    public int getCachePolicy() {
        return this.cachePolicy;
    }

    protected String getProxyURL() {
        return this.proxyUrl;
    }

    public String getRequestURL(){
        if (!TextUtils.isEmpty(this.proxyUrl))
            return this.proxyUrl;
        else
            return this.url;
    }

    public Builder getCurrentState() {
        Builder builder = new Builder(this.url, this.method, this.context)
                .setTag(this.tag)
                .setFlags(this.flags)
                .setCachePolicy(this.cachePolicy)
                .setHeaders(this.headers)
                .setParams(params)
                .setProxyURL(this.proxyUrl);
        return builder;
    }

    protected void setProxyUrl(String newUrl) {
        this.proxyUrl = newUrl;
    }

    protected Response execute() {
        boolean flag = true;
        synchronized (RequestExecutor.interceptors) {
            for (Interceptor interceptor : RequestExecutor.interceptors) {
                flag &= interceptor.onRequest(this, this.context);
            }
        }

        CallbackDispatcher callbackDispatcher = CallbackDispatcher.init(context);
        Response response = null;
        if (flag) {
            /**
             * Calling CustomInterceptors method only if the flag is true
             * i.e. request is not intercepted previously using {@link Interceptor}
             **/
            RequestObject modifiedRequestObject = callbackDispatcher.onRequest(context, this);
            RequestExecutor requestExecutor;
            if (modifiedRequestObject != null) {
                requestExecutor = new RequestExecutor(this.context, modifiedRequestObject);
                response = requestExecutor.applyCachePolicy();
            } else {
                Logger.e(WebEngageConstant.TAG, "Error while processing network request as request object is null for url "+this.getRequestURL());
                //If modified Request is a null object, do not execute it
            }
        }
        if (response == null) {
            response = new Response.Builder().setRequestUrl(this.getRequestURL()).build();
        }
        return response;
    }
}
