package com.webengage.sdk.android.utils.http;

import android.content.Context;
import android.content.Intent;
import android.text.TextUtils;

import com.webengage.sdk.android.CallbackDispatcher;
import com.webengage.sdk.android.IntentFactory;
import com.webengage.sdk.android.Logger;
import com.webengage.sdk.android.Topic;
import com.webengage.sdk.android.WebEngage;
import com.webengage.sdk.android.WebEngageConfig;
import com.webengage.sdk.android.utils.NetworkUtils;
import com.webengage.sdk.android.utils.WebEngageConstant;

import java.io.InputStream;
import java.net.HttpURLConnection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class WENetworkUtil {
    private static final String errorMsgHttpUnavailable = "503 service Temporarily Unavailable";

    public static Map<String, Object> makeRequest(Context context, RequestObject requestObject, boolean isSecured, boolean isTriggerSecurityException) {
        String cuid = WebEngage.get().getCUID();
        return makeRequest(context, requestObject, isSecured, isTriggerSecurityException, cuid);
    }

    public static Map<String, Object> makeRequest(Context context, RequestObject requestObject, boolean isSecured, boolean isTriggerSecurityException, String cuid) {
        Map<String, String> headers = requestObject.getHeaders();
        if (headers == null) {
            headers = new HashMap<>();
        }
        WebEngageConfig webEngageConfig = WebEngage.get().getWebEngageConfig();
        if (shouldAddAuthHeaders(requestObject.getURL(), webEngageConfig.getEnvironment())) {
            headers.put("lc", webEngageConfig.getWebEngageKey());
            if (isSecured && !TextUtils.isEmpty(cuid)) {
                headers.put("cuid", cuid);
                headers.put("Authorization", "Bearer " + WebEngage.get().getSecurityToken(cuid));
            }
        }
        String proxyURL = getProxyURL(webEngageConfig.getWebEngageKey());
        if (shouldWrapProxyURL(requestObject.getURL(), proxyURL)) {
            String newUrl = WebEngageConstant.Urls.getUrlModifiedWithProxyURL(requestObject.getURL(), proxyURL);
            requestObject.setProxyUrl(newUrl);
        }
        return makeRequest(context, requestObject, isTriggerSecurityException);
    }

    private static Map<String, Object> makeRequest(Context context, RequestObject requestObject, boolean isTriggerSecurityException) {
        Map<String, Object> result = new HashMap<>();
        Response response = requestObject.execute();
        int statusCode = response.getResponseCode();
        result.put("status", response.getResponseCode());
        String contentType = "";
        Map<String, List<String>> headers = response.getResponseHeaders();
        if (headers != null && headers.containsKey("content-type")) {
            List<String> contentTypeHeader = headers.get("content-type");
            if (contentTypeHeader != null && !contentTypeHeader.isEmpty()) {
                contentType = contentTypeHeader.get(0);
            }
        }

        try {
            if (response.isReadable()) {
                if (statusCode >= HttpURLConnection.HTTP_OK && statusCode < HttpURLConnection.HTTP_BAD_REQUEST) {
                    result.put("data", response.getInputStream());
                    result.put("content-type", contentType);
                } else {
                    handleError(response, result, statusCode, context, isTriggerSecurityException);
                    //System.out.println(result);
                }
            } else {
                handleError(response, result, statusCode, context, isTriggerSecurityException);
                response.closeErrorStream();
                //System.out.println(result);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return result;
    }

    private static void handleError(Response response, Map<String, Object> result, int statusCode, Context context, boolean isTriggerSecurityException) {
        Logger.e(WebEngageConstant.TAG, "Error " + response.getResponseCode() + " while fetching response from url " + response.getRequestURL());
        try {
            if (statusCode == HttpURLConnection.HTTP_UNAVAILABLE) {
                result.put("error", new HashMap<>().put("message", errorMsgHttpUnavailable));
            } else {
                if (response != null) {
                    if (response.getErrorStream() == null) {
                        result.put("error", null);
                    } else {
                        try {
                            result.put("error", getError(response.getErrorStream()));
                        } catch (Exception e) {
                            result.put("error", "");
                        }
                    }

                    if (response.getException() != null) {
                        result.put("exception", response.getException());
                        // statusCode is -1 incase of FileNotFound, UnknownHostException and No internet Connection
                        if (statusCode != -1) {
                            String exc = "Exception: " + response.getException() + "\nURL: " + response.getRequestURL() + "\nResponseCode: " + response.getResponseCode() + "\nIsInputStreamNull: " + (response.getInputStream() == null);
                            dispatchException(new Exception("Exception: " + response.getException() + "\nURL: " + response.getRequestURL() + "\nResponseCode: " + response.getResponseCode() + "\nIsInputStreamNull: " + (response.getInputStream() == null)), context);
                        }
                    }

                    if (statusCode == HttpURLConnection.HTTP_UNAUTHORIZED && isTriggerSecurityException)
                        CallbackDispatcher.init(context).onSecurityException(result);
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private static Map<String, Object> getError(InputStream stream) throws Exception {
        return NetworkUtils.getAsMap(stream, false);
    }

    public static String readEntireStream(InputStream inputStream) {
        return NetworkUtils.readEntireStream(inputStream);
    }

    private static void dispatchException(Object dataException, Context context) {
        Intent intent = IntentFactory.newIntent(Topic.EXCEPTION, dataException, context);
        WebEngage.startService(intent, context);
    }

    private static boolean shouldAddAuthHeaders(String url, String environment) {
        if (url.startsWith(WebEngageConstant.Urls.URL_BASE.toString())
                && !url.startsWith(WebEngageConstant.Urls.EXCEPTION_END_POINT.toString())
        )
            return true;
        else
            return false;
    }

    private static boolean shouldWrapProxyURL(String url, String proxyURL) {
        if (!(TextUtils.isEmpty(proxyURL) || url.startsWith(proxyURL)))
            return true;
        return false;
    }

    private static String getProxyURL(String webEngageKey) {
        //Modify this to get LC specific proxy in MultiLC
        return WebEngage.get().getWebEngageConfig().getProxyURL();
    }

    public static String getAcceptHeadersForPushImages(String url){
        if(url.endsWith(".gif"))
            return "image/gif";
        else
            return "image/gif, image/webp";
    }
}
