/*
 * This file is part of WebLookAndFeel library.
 *
 * WebLookAndFeel library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * WebLookAndFeel library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with WebLookAndFeel library.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.alee.managers.language.data;

import com.alee.api.merge.Mergeable;
import com.alee.managers.language.LanguageUtils;
import com.thoughtworks.xstream.annotations.XStreamAlias;
import com.thoughtworks.xstream.annotations.XStreamAsAttribute;
import com.thoughtworks.xstream.annotations.XStreamConverter;

import java.io.Serializable;
import java.util.Locale;

/**
 * {@link TranslationInformation} contains some general information about single available translation.
 * Instances of {@link TranslationInformation} might be automatically generated by {@link Dictionary} when necessary.
 *
 * @author Mikle Garin
 * @see <a href="https://github.com/mgarin/weblaf/wiki/How-to-use-LanguageManager">How to use LanguageManager</a>
 * @see com.alee.managers.language.LanguageManager
 * @see Dictionary
 */
@XStreamAlias ( "Translation" )
@XStreamConverter ( TranslationInformationConverter.class )
public final class TranslationInformation implements Mergeable, Cloneable, Serializable
{
    /**
     * Translation {@link Locale}.
     */
    @XStreamAsAttribute
    @XStreamAlias ( "lang" )
    private Locale locale;

    /**
     * Translation language title.
     */
    @XStreamAsAttribute
    private String title;

    /**
     * Translation author.
     */
    @XStreamAsAttribute
    private String author;

    /**
     * Constructs new empty {@link TranslationInformation}.
     */
    public TranslationInformation ()
    {
        super ();
    }

    /**
     * Constructs new {@link TranslationInformation}.
     *
     * @param locale translation {@link Locale}
     * @param title  translation language title
     * @param author translation author
     */
    public TranslationInformation ( final Locale locale, final String title, final String author )
    {
        super ();
        this.locale = locale;
        this.title = title;
        this.author = author;
    }

    /**
     * Returns translation {@link Locale}.
     *
     * @return translation {@link Locale}
     */
    public Locale getLocale ()
    {
        return locale;
    }

    /**
     * Sets translation {@link Locale}.
     *
     * @param locale new translation {@link Locale}
     */
    public void setLocale ( final Locale locale )
    {
        this.locale = locale;
    }

    /**
     * Returns translation language title.
     *
     * @return translation language title
     */
    public String getTitle ()
    {
        return title;
    }

    /**
     * Sets translation language title.
     *
     * @param title new translation language title
     */
    public void setTitle ( final String title )
    {
        this.title = title;
    }

    /**
     * Returns translation author.
     *
     * @return translation author
     */
    public String getAuthor ()
    {
        return author;
    }

    /**
     * Sets translation author.
     *
     * @param author new translation author
     */
    public void setAuthor ( final String author )
    {
        this.author = author;
    }

    @Override
    public String toString ()
    {
        return "Translation[" + LanguageUtils.toString ( getLocale () ) + ", " + getAuthor () + ", " + getTitle () + "]";
    }
}