/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.gateway;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.LongSupplier;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.metadata.IndexMetaData;
import org.elasticsearch.cluster.metadata.Manifest;
import org.elasticsearch.cluster.metadata.MetaData;
import org.elasticsearch.cluster.routing.RoutingNode;
import org.elasticsearch.cluster.routing.ShardRouting;
import org.elasticsearch.common.settings.ClusterSettings;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.gateway.MetaStateService;
import org.elasticsearch.gateway.WriteStateException;
import org.elasticsearch.index.Index;

public class IncrementalClusterStateWriter {
    private static final Logger logger = LogManager.getLogger(IncrementalClusterStateWriter.class);
    public static final Setting<TimeValue> SLOW_WRITE_LOGGING_THRESHOLD = Setting.timeSetting("gateway.slow_write_logging_threshold", TimeValue.timeValueSeconds(10L), TimeValue.ZERO, Setting.Property.NodeScope, Setting.Property.Dynamic);
    private final MetaStateService metaStateService;
    private Manifest previousManifest;
    private ClusterState previousClusterState;
    private final LongSupplier relativeTimeMillisSupplier;
    private boolean incrementalWrite;
    private volatile TimeValue slowWriteLoggingThreshold;

    IncrementalClusterStateWriter(Settings settings, ClusterSettings clusterSettings, MetaStateService metaStateService, Manifest manifest, ClusterState clusterState, LongSupplier relativeTimeMillisSupplier) {
        this.metaStateService = metaStateService;
        this.previousManifest = manifest;
        this.previousClusterState = clusterState;
        this.relativeTimeMillisSupplier = relativeTimeMillisSupplier;
        this.incrementalWrite = false;
        this.slowWriteLoggingThreshold = SLOW_WRITE_LOGGING_THRESHOLD.get(settings);
        clusterSettings.addSettingsUpdateConsumer(SLOW_WRITE_LOGGING_THRESHOLD, this::setSlowWriteLoggingThreshold);
    }

    private void setSlowWriteLoggingThreshold(TimeValue slowWriteLoggingThreshold) {
        this.slowWriteLoggingThreshold = slowWriteLoggingThreshold;
    }

    void setCurrentTerm(long currentTerm) throws WriteStateException {
        Manifest manifest = new Manifest(currentTerm, this.previousManifest.getClusterStateVersion(), this.previousManifest.getGlobalGeneration(), new HashMap<Index, Long>(this.previousManifest.getIndexGenerations()));
        this.metaStateService.writeManifestAndCleanup("current term changed", manifest);
        this.previousManifest = manifest;
    }

    Manifest getPreviousManifest() {
        return this.previousManifest;
    }

    ClusterState getPreviousClusterState() {
        return this.previousClusterState;
    }

    void setIncrementalWrite(boolean incrementalWrite) {
        this.incrementalWrite = incrementalWrite;
    }

    void updateClusterState(ClusterState newState) throws WriteStateException {
        MetaData newMetaData = newState.metaData();
        long startTimeMillis = this.relativeTimeMillisSupplier.getAsLong();
        AtomicClusterStateWriter writer = new AtomicClusterStateWriter(this.metaStateService, this.previousManifest);
        long globalStateGeneration = this.writeGlobalState(writer, newMetaData);
        Map<Index, Long> indexGenerations = this.writeIndicesMetadata(writer, newState);
        Manifest manifest = new Manifest(this.previousManifest.getCurrentTerm(), newState.version(), globalStateGeneration, indexGenerations);
        this.writeManifest(writer, manifest);
        this.previousManifest = manifest;
        this.previousClusterState = newState;
        long durationMillis = this.relativeTimeMillisSupplier.getAsLong() - startTimeMillis;
        TimeValue finalSlowWriteLoggingThreshold = this.slowWriteLoggingThreshold;
        if (durationMillis >= finalSlowWriteLoggingThreshold.getMillis()) {
            logger.warn("writing cluster state took [{}ms] which is above the warn threshold of [{}]; wrote metadata for [{}] indices and skipped [{}] unchanged indices", (Object)durationMillis, (Object)finalSlowWriteLoggingThreshold, (Object)writer.getIndicesWritten(), (Object)writer.getIndicesSkipped());
        } else {
            logger.debug("writing cluster state took [{}ms]; wrote metadata for [{}] indices and skipped [{}] unchanged indices", (Object)durationMillis, (Object)writer.getIndicesWritten(), (Object)writer.getIndicesSkipped());
        }
    }

    private void writeManifest(AtomicClusterStateWriter writer, Manifest manifest) throws WriteStateException {
        if (!manifest.equals(this.previousManifest)) {
            writer.writeManifestAndCleanup("changed", manifest);
        }
    }

    private Map<Index, Long> writeIndicesMetadata(AtomicClusterStateWriter writer, ClusterState newState) throws WriteStateException {
        Map<Index, Long> previouslyWrittenIndices = this.previousManifest.getIndexGenerations();
        Set<Index> relevantIndices = IncrementalClusterStateWriter.getRelevantIndices(newState);
        HashMap<Index, Long> newIndices = new HashMap<Index, Long>();
        MetaData previousMetaData = this.incrementalWrite ? this.previousClusterState.metaData() : null;
        List<IndexMetaDataAction> actions = IncrementalClusterStateWriter.resolveIndexMetaDataActions(previouslyWrittenIndices, relevantIndices, previousMetaData, newState.metaData());
        for (IndexMetaDataAction action : actions) {
            long generation = action.execute(writer);
            newIndices.put(action.getIndex(), generation);
        }
        return newIndices;
    }

    private long writeGlobalState(AtomicClusterStateWriter writer, MetaData newMetaData) throws WriteStateException {
        if (!this.incrementalWrite || !MetaData.isGlobalStateEquals(this.previousClusterState.metaData(), newMetaData)) {
            return writer.writeGlobalState("changed", newMetaData);
        }
        return this.previousManifest.getGlobalGeneration();
    }

    static List<IndexMetaDataAction> resolveIndexMetaDataActions(Map<Index, Long> previouslyWrittenIndices, Set<Index> relevantIndices, MetaData previousMetaData, MetaData newMetaData) {
        ArrayList<IndexMetaDataAction> actions = new ArrayList<IndexMetaDataAction>();
        for (Index index : relevantIndices) {
            IndexMetaData previousIndexMetaData;
            IndexMetaData newIndexMetaData = newMetaData.getIndexSafe(index);
            IndexMetaData indexMetaData = previousIndexMetaData = previousMetaData == null ? null : previousMetaData.index(index);
            if (!previouslyWrittenIndices.containsKey(index) || previousIndexMetaData == null) {
                actions.add(new WriteNewIndexMetaData(newIndexMetaData));
                continue;
            }
            if (previousIndexMetaData.getVersion() != newIndexMetaData.getVersion()) {
                actions.add(new WriteChangedIndexMetaData(previousIndexMetaData, newIndexMetaData));
                continue;
            }
            actions.add(new KeepPreviousGeneration(index, previouslyWrittenIndices.get(index)));
        }
        return actions;
    }

    private static Set<Index> getRelevantIndicesOnDataOnlyNode(ClusterState state) {
        RoutingNode newRoutingNode = state.getRoutingNodes().node(state.nodes().getLocalNodeId());
        if (newRoutingNode == null) {
            throw new IllegalStateException("cluster state does not contain this node - cannot write index meta state");
        }
        HashSet<Index> indices = new HashSet<Index>();
        for (ShardRouting routing : newRoutingNode) {
            indices.add(routing.index());
        }
        return indices;
    }

    private static Set<Index> getRelevantIndicesForMasterEligibleNode(ClusterState state) {
        HashSet<Index> relevantIndices = new HashSet<Index>();
        for (IndexMetaData indexMetaData : state.metaData()) {
            relevantIndices.add(indexMetaData.getIndex());
        }
        return relevantIndices;
    }

    static Set<Index> getRelevantIndices(ClusterState state) {
        if (state.nodes().getLocalNode().isMasterNode()) {
            return IncrementalClusterStateWriter.getRelevantIndicesForMasterEligibleNode(state);
        }
        if (state.nodes().getLocalNode().isDataNode()) {
            return IncrementalClusterStateWriter.getRelevantIndicesOnDataOnlyNode(state);
        }
        return Collections.emptySet();
    }

    static class WriteChangedIndexMetaData
    implements IndexMetaDataAction {
        private final IndexMetaData newIndexMetaData;
        private final IndexMetaData oldIndexMetaData;

        WriteChangedIndexMetaData(IndexMetaData oldIndexMetaData, IndexMetaData newIndexMetaData) {
            this.oldIndexMetaData = oldIndexMetaData;
            this.newIndexMetaData = newIndexMetaData;
        }

        @Override
        public Index getIndex() {
            return this.newIndexMetaData.getIndex();
        }

        @Override
        public long execute(AtomicClusterStateWriter writer) throws WriteStateException {
            writer.incrementIndicesWritten();
            return writer.writeIndex("version changed from [" + this.oldIndexMetaData.getVersion() + "] to [" + this.newIndexMetaData.getVersion() + "]", this.newIndexMetaData);
        }
    }

    static class WriteNewIndexMetaData
    implements IndexMetaDataAction {
        private final IndexMetaData indexMetaData;

        WriteNewIndexMetaData(IndexMetaData indexMetaData) {
            this.indexMetaData = indexMetaData;
        }

        @Override
        public Index getIndex() {
            return this.indexMetaData.getIndex();
        }

        @Override
        public long execute(AtomicClusterStateWriter writer) throws WriteStateException {
            writer.incrementIndicesWritten();
            return writer.writeIndex("freshly created", this.indexMetaData);
        }
    }

    static class KeepPreviousGeneration
    implements IndexMetaDataAction {
        private final Index index;
        private final long generation;

        KeepPreviousGeneration(Index index, long generation) {
            this.index = index;
            this.generation = generation;
        }

        @Override
        public Index getIndex() {
            return this.index;
        }

        @Override
        public long execute(AtomicClusterStateWriter writer) {
            writer.incrementIndicesSkipped();
            return this.generation;
        }
    }

    static class AtomicClusterStateWriter {
        private static final String FINISHED_MSG = "AtomicClusterStateWriter is finished";
        private final List<Runnable> commitCleanupActions;
        private final List<Runnable> rollbackCleanupActions;
        private final Manifest previousManifest;
        private final MetaStateService metaStateService;
        private boolean finished;
        private int indicesWritten;
        private int indicesSkipped;

        AtomicClusterStateWriter(MetaStateService metaStateService, Manifest previousManifest) {
            this.metaStateService = metaStateService;
            assert (previousManifest != null);
            this.previousManifest = previousManifest;
            this.commitCleanupActions = new ArrayList<Runnable>();
            this.rollbackCleanupActions = new ArrayList<Runnable>();
            this.finished = false;
        }

        long writeGlobalState(String reason, MetaData metaData) throws WriteStateException {
            assert (!this.finished) : "AtomicClusterStateWriter is finished";
            try {
                this.rollbackCleanupActions.add(() -> this.metaStateService.cleanupGlobalState(this.previousManifest.getGlobalGeneration()));
                long generation = this.metaStateService.writeGlobalState(reason, metaData);
                this.commitCleanupActions.add(() -> this.metaStateService.cleanupGlobalState(generation));
                return generation;
            }
            catch (WriteStateException e) {
                this.rollback();
                throw e;
            }
        }

        long writeIndex(String reason, IndexMetaData metaData) throws WriteStateException {
            assert (!this.finished) : "AtomicClusterStateWriter is finished";
            try {
                Index index = metaData.getIndex();
                Long previousGeneration = this.previousManifest.getIndexGenerations().get(index);
                if (previousGeneration != null) {
                    this.rollbackCleanupActions.add(() -> this.metaStateService.cleanupIndex(index, previousGeneration));
                }
                long generation = this.metaStateService.writeIndex(reason, metaData);
                this.commitCleanupActions.add(() -> this.metaStateService.cleanupIndex(index, generation));
                return generation;
            }
            catch (WriteStateException e) {
                this.rollback();
                throw e;
            }
        }

        void writeManifestAndCleanup(String reason, Manifest manifest) throws WriteStateException {
            assert (!this.finished) : "AtomicClusterStateWriter is finished";
            try {
                this.metaStateService.writeManifestAndCleanup(reason, manifest);
                this.commitCleanupActions.forEach(Runnable::run);
                this.finished = true;
            }
            catch (WriteStateException e) {
                if (!e.isDirty()) {
                    this.rollback();
                }
                throw e;
            }
        }

        void rollback() {
            this.rollbackCleanupActions.forEach(Runnable::run);
            this.finished = true;
        }

        void incrementIndicesWritten() {
            ++this.indicesWritten;
        }

        void incrementIndicesSkipped() {
            ++this.indicesSkipped;
        }

        int getIndicesWritten() {
            return this.indicesWritten;
        }

        int getIndicesSkipped() {
            return this.indicesSkipped;
        }
    }

    static interface IndexMetaDataAction {
        public Index getIndex();

        public long execute(AtomicClusterStateWriter var1) throws WriteStateException;
    }
}

