package assertk.assertions

import assertk.Assert
import assertk.PlatformName
import assertk.assertAll
import assertk.assertions.support.ListDiffer
import assertk.assertions.support.expected
import assertk.assertions.support.show
import assertk.assertions.support.fail

/**
 * Returns an assert on the ByteArray's size.
 */
@PlatformName("byteArraySize")
fun Assert<ByteArray>.size() = prop("size", ByteArray::size)

/**
 * Asserts the ByteArray contents are equal to the expected one, using [contentDeepEquals].
 * @see isNotEqualTo
 */
fun Assert<ByteArray>.isEqualTo(expected: ByteArray) {
    if (actual.contentEquals(expected)) return
    fail(expected, actual)
}

/**
 * Asserts the ByteArray contents are not equal to the expected one, using [contentDeepEquals].
 * @see isEqualTo
 */
fun Assert<ByteArray>.isNotEqualTo(expected: ByteArray) {
    if (!(actual.contentEquals(expected))) return
    val showExpected = show(expected)
    val showActual = show(actual)
    // if they display the same, only show one.
    if (showExpected == showActual) {
        expected("to not be equal to:$showActual")
    } else {
        expected(":$showExpected not to be equal to:$showActual")
    }
}

/**
 * Asserts the ByteArray is empty.
 * @see [isNotEmpty]
 * @see [isNullOrEmpty]
 */
@PlatformName("byteArrayIsEmpty")
fun Assert<ByteArray>.isEmpty() {
    if (actual.isEmpty()) return
    expected("to be empty but was:${show(actual)}")
}

/**
 * Asserts the ByteArray is not empty.
 * @see [isEmpty]
 */
@PlatformName("byteArrayIsNotEmpty")
fun Assert<ByteArray>.isNotEmpty() {
    if (actual.isNotEmpty()) return
    expected("to not be empty")
}

/**
 * Asserts the ByteArray is null or empty.
 * @see [isEmpty]
 */
@PlatformName("byteArrayIsNullOrEmpty")
fun Assert<ByteArray?>.isNullOrEmpty() {
    if (actual == null || actual.isEmpty()) return
    expected("to be null or empty but was:${show(actual)}")
}

/**
 * Asserts the ByteArray has the expected size.
 */
@PlatformName("byteArrayHasSize")
fun Assert<ByteArray>.hasSize(size: Int) {
    assert(actual.size, "size").isEqualTo(size)
}

/**
 * Asserts the ByteArray has the same size as the expected array.
 */
@PlatformName("byteArrayHasSameSizeAs")
fun Assert<ByteArray>.hasSameSizeAs(other: ByteArray) {
    val actualSize = actual.size
    val otherSize = other.size
    if (actualSize == otherSize) return
    expected("to have same size as:${show(other)} ($otherSize) but was size:($actualSize)")
}

/**
 * Asserts the ByteArray contains the expected element, using `in`.
 * @see [doesNotContain]
 */
@PlatformName("byteArrayContains")
fun Assert<ByteArray>.contains(element: Byte) {
    if (element in actual) return
    expected("to contain:${show(element)} but was:${show(actual)}")
}

/**
 * Asserts the ByteArray does not contain the expected element, using `!in`.
 * @see [contains]
 */
@PlatformName("byteArrayDoesNotContain")
fun Assert<ByteArray>.doesNotContain(element: Byte) {
    if (element !in actual) return
    expected("to not contain:${show(element)} but was:${show(actual)}")
}

/**
 * Asserts the ByteArray does not contain any of the expected elements.
 * @see [containsAll]
 */
fun Assert<ByteArray>.containsNone(vararg elements: Byte) {
    if (elements.none { it in actual }) {
        return
    }

    val notExpected = elements.filter { it in actual }
    expected("to contain none of:${show(elements)} some elements were not expected:${show(notExpected)}")
}

/**
 * Asserts the ByteArray contains all the expected elements, in any order. The array may also contain
 * additional elements.
 * @see [containsExactly]
 */
@PlatformName("byteArrayContainsAll")
fun Assert<ByteArray>.containsAll(vararg elements: Byte) {
    if (elements.all { actual.contains(it) }) return
    val notFound = elements.filterNot { it in actual }
    expected("to contain all:${show(elements)} some elements were not found:${show(notFound)}")
}

/**
 * Returns an assert that assertion on the value at the given index in the ByteArray.
 *
 * ```
 * assert(byteArrayOf(0, 1, 2)).index(1) { it.isPositive() }
 * ```
 */
@PlatformName("byteArrayIndex")
fun Assert<ByteArray>.index(index: Int, f: (Assert<Byte>) -> Unit) {
    if (index in 0 until actual.size) {
        f(assert(actual[index], "${name ?: ""}${show(index, "[]")}"))
    } else {
        expected("index to be in range:[0-${actual.size}) but was:${show(index)}")
    }
}

/**
 * Asserts the ByteArray contains exactly the expected elements. They must be in the same order and
 * there must not be any extra elements.
 * @see [containsAll]
 */
@PlatformName("byteArrayContainsExactly")
fun Assert<ByteArray>.containsExactly(vararg elements: Byte) {
    if (actual.contentEquals(elements)) return

    val diff = ListDiffer.diff(elements.asList(), actual.asList())
        .filterNot { it is ListDiffer.Edit.Eq }

    expected(diff.joinToString(prefix = "to contain exactly:\n", separator = "\n") { edit ->
        when (edit) {
            is ListDiffer.Edit.Del -> " at index:${edit.oldIndex} expected:${show(edit.oldValue)}"
            is ListDiffer.Edit.Ins -> " at index:${edit.newIndex} unexpected:${show(edit.newValue)}"
            else -> throw IllegalStateException()
        }
    })
}

/**
 * Asserts on each item in the ByteArray. The given lambda will be run for each item.
 *
 * ```
 * assert(byteArrayOf("one", "two")).each {
 *   it.hasLength(3)
 * }
 * ```
 */
@PlatformName("byteArrayEach")
fun Assert<ByteArray>.each(f: (Assert<Byte>) -> Unit) {
    assertAll {
        actual.forEachIndexed { index, item ->
            f(assert(item, "${name ?: ""}${show(index, "[]")}"))
        }
    }
}

/**
 * Returns an assert on the IntArray's size.
 */
@PlatformName("intArraySize")
fun Assert<IntArray>.size() = prop("size", IntArray::size)

/**
 * Asserts the IntArray contents are equal to the expected one, using [contentDeepEquals].
 * @see isNotEqualTo
 */
fun Assert<IntArray>.isEqualTo(expected: IntArray) {
    if (actual.contentEquals(expected)) return
    fail(expected, actual)
}

/**
 * Asserts the IntArray contents are not equal to the expected one, using [contentDeepEquals].
 * @see isEqualTo
 */
fun Assert<IntArray>.isNotEqualTo(expected: IntArray) {
    if (!(actual.contentEquals(expected))) return
    val showExpected = show(expected)
    val showActual = show(actual)
    // if they display the same, only show one.
    if (showExpected == showActual) {
        expected("to not be equal to:$showActual")
    } else {
        expected(":$showExpected not to be equal to:$showActual")
    }
}

/**
 * Asserts the IntArray is empty.
 * @see [isNotEmpty]
 * @see [isNullOrEmpty]
 */
@PlatformName("intArrayIsEmpty")
fun Assert<IntArray>.isEmpty() {
    if (actual.isEmpty()) return
    expected("to be empty but was:${show(actual)}")
}

/**
 * Asserts the IntArray is not empty.
 * @see [isEmpty]
 */
@PlatformName("intArrayIsNotEmpty")
fun Assert<IntArray>.isNotEmpty() {
    if (actual.isNotEmpty()) return
    expected("to not be empty")
}

/**
 * Asserts the IntArray is null or empty.
 * @see [isEmpty]
 */
@PlatformName("intArrayIsNullOrEmpty")
fun Assert<IntArray?>.isNullOrEmpty() {
    if (actual == null || actual.isEmpty()) return
    expected("to be null or empty but was:${show(actual)}")
}

/**
 * Asserts the IntArray has the expected size.
 */
@PlatformName("intArrayHasSize")
fun Assert<IntArray>.hasSize(size: Int) {
    assert(actual.size, "size").isEqualTo(size)
}

/**
 * Asserts the IntArray has the same size as the expected array.
 */
@PlatformName("intArrayHasSameSizeAs")
fun Assert<IntArray>.hasSameSizeAs(other: IntArray) {
    val actualSize = actual.size
    val otherSize = other.size
    if (actualSize == otherSize) return
    expected("to have same size as:${show(other)} ($otherSize) but was size:($actualSize)")
}

/**
 * Asserts the IntArray contains the expected element, using `in`.
 * @see [doesNotContain]
 */
@PlatformName("intArrayContains")
fun Assert<IntArray>.contains(element: Int) {
    if (element in actual) return
    expected("to contain:${show(element)} but was:${show(actual)}")
}

/**
 * Asserts the IntArray does not contain the expected element, using `!in`.
 * @see [contains]
 */
@PlatformName("intArrayDoesNotContain")
fun Assert<IntArray>.doesNotContain(element: Int) {
    if (element !in actual) return
    expected("to not contain:${show(element)} but was:${show(actual)}")
}

/**
 * Asserts the IntArray does not contain any of the expected elements.
 * @see [containsAll]
 */
fun Assert<IntArray>.containsNone(vararg elements: Int) {
    if (elements.none { it in actual }) {
        return
    }

    val notExpected = elements.filter { it in actual }
    expected("to contain none of:${show(elements)} some elements were not expected:${show(notExpected)}")
}

/**
 * Asserts the IntArray contains all the expected elements, in any order. The array may also contain
 * additional elements.
 * @see [containsExactly]
 */
@PlatformName("intArrayContainsAll")
fun Assert<IntArray>.containsAll(vararg elements: Int) {
    if (elements.all { actual.contains(it) }) return
    val notFound = elements.filterNot { it in actual }
    expected("to contain all:${show(elements)} some elements were not found:${show(notFound)}")
}

/**
 * Returns an assert that assertion on the value at the given index in the IntArray.
 *
 * ```
 * assert(intArrayOf(0, 1, 2)).index(1) { it.isPositive() }
 * ```
 */
@PlatformName("intArrayIndex")
fun Assert<IntArray>.index(index: Int, f: (Assert<Int>) -> Unit) {
    if (index in 0 until actual.size) {
        f(assert(actual[index], "${name ?: ""}${show(index, "[]")}"))
    } else {
        expected("index to be in range:[0-${actual.size}) but was:${show(index)}")
    }
}

/**
 * Asserts the IntArray contains exactly the expected elements. They must be in the same order and
 * there must not be any extra elements.
 * @see [containsAll]
 */
@PlatformName("intArrayContainsExactly")
fun Assert<IntArray>.containsExactly(vararg elements: Int) {
    if (actual.contentEquals(elements)) return

    val diff = ListDiffer.diff(elements.asList(), actual.asList())
        .filterNot { it is ListDiffer.Edit.Eq }

    expected(diff.joinToString(prefix = "to contain exactly:\n", separator = "\n") { edit ->
        when (edit) {
            is ListDiffer.Edit.Del -> " at index:${edit.oldIndex} expected:${show(edit.oldValue)}"
            is ListDiffer.Edit.Ins -> " at index:${edit.newIndex} unexpected:${show(edit.newValue)}"
            else -> throw IllegalStateException()
        }
    })
}

/**
 * Asserts on each item in the IntArray. The given lambda will be run for each item.
 *
 * ```
 * assert(intArrayOf("one", "two")).each {
 *   it.hasLength(3)
 * }
 * ```
 */
@PlatformName("intArrayEach")
fun Assert<IntArray>.each(f: (Assert<Int>) -> Unit) {
    assertAll {
        actual.forEachIndexed { index, item ->
            f(assert(item, "${name ?: ""}${show(index, "[]")}"))
        }
    }
}

/**
 * Returns an assert on the ShortArray's size.
 */
@PlatformName("shortArraySize")
fun Assert<ShortArray>.size() = prop("size", ShortArray::size)

/**
 * Asserts the ShortArray contents are equal to the expected one, using [contentDeepEquals].
 * @see isNotEqualTo
 */
fun Assert<ShortArray>.isEqualTo(expected: ShortArray) {
    if (actual.contentEquals(expected)) return
    fail(expected, actual)
}

/**
 * Asserts the ShortArray contents are not equal to the expected one, using [contentDeepEquals].
 * @see isEqualTo
 */
fun Assert<ShortArray>.isNotEqualTo(expected: ShortArray) {
    if (!(actual.contentEquals(expected))) return
    val showExpected = show(expected)
    val showActual = show(actual)
    // if they display the same, only show one.
    if (showExpected == showActual) {
        expected("to not be equal to:$showActual")
    } else {
        expected(":$showExpected not to be equal to:$showActual")
    }
}

/**
 * Asserts the ShortArray is empty.
 * @see [isNotEmpty]
 * @see [isNullOrEmpty]
 */
@PlatformName("shortArrayIsEmpty")
fun Assert<ShortArray>.isEmpty() {
    if (actual.isEmpty()) return
    expected("to be empty but was:${show(actual)}")
}

/**
 * Asserts the ShortArray is not empty.
 * @see [isEmpty]
 */
@PlatformName("shortArrayIsNotEmpty")
fun Assert<ShortArray>.isNotEmpty() {
    if (actual.isNotEmpty()) return
    expected("to not be empty")
}

/**
 * Asserts the ShortArray is null or empty.
 * @see [isEmpty]
 */
@PlatformName("shortArrayIsNullOrEmpty")
fun Assert<ShortArray?>.isNullOrEmpty() {
    if (actual == null || actual.isEmpty()) return
    expected("to be null or empty but was:${show(actual)}")
}

/**
 * Asserts the ShortArray has the expected size.
 */
@PlatformName("shortArrayHasSize")
fun Assert<ShortArray>.hasSize(size: Int) {
    assert(actual.size, "size").isEqualTo(size)
}

/**
 * Asserts the ShortArray has the same size as the expected array.
 */
@PlatformName("shortArrayHasSameSizeAs")
fun Assert<ShortArray>.hasSameSizeAs(other: ShortArray) {
    val actualSize = actual.size
    val otherSize = other.size
    if (actualSize == otherSize) return
    expected("to have same size as:${show(other)} ($otherSize) but was size:($actualSize)")
}

/**
 * Asserts the ShortArray contains the expected element, using `in`.
 * @see [doesNotContain]
 */
@PlatformName("shortArrayContains")
fun Assert<ShortArray>.contains(element: Short) {
    if (element in actual) return
    expected("to contain:${show(element)} but was:${show(actual)}")
}

/**
 * Asserts the ShortArray does not contain the expected element, using `!in`.
 * @see [contains]
 */
@PlatformName("shortArrayDoesNotContain")
fun Assert<ShortArray>.doesNotContain(element: Short) {
    if (element !in actual) return
    expected("to not contain:${show(element)} but was:${show(actual)}")
}

/**
 * Asserts the ShortArray does not contain any of the expected elements.
 * @see [containsAll]
 */
fun Assert<ShortArray>.containsNone(vararg elements: Short) {
    if (elements.none { it in actual }) {
        return
    }

    val notExpected = elements.filter { it in actual }
    expected("to contain none of:${show(elements)} some elements were not expected:${show(notExpected)}")
}

/**
 * Asserts the ShortArray contains all the expected elements, in any order. The array may also contain
 * additional elements.
 * @see [containsExactly]
 */
@PlatformName("shortArrayContainsAll")
fun Assert<ShortArray>.containsAll(vararg elements: Short) {
    if (elements.all { actual.contains(it) }) return
    val notFound = elements.filterNot { it in actual }
    expected("to contain all:${show(elements)} some elements were not found:${show(notFound)}")
}

/**
 * Returns an assert that assertion on the value at the given index in the ShortArray.
 *
 * ```
 * assert(shortArrayOf(0, 1, 2)).index(1) { it.isPositive() }
 * ```
 */
@PlatformName("shortArrayIndex")
fun Assert<ShortArray>.index(index: Int, f: (Assert<Short>) -> Unit) {
    if (index in 0 until actual.size) {
        f(assert(actual[index], "${name ?: ""}${show(index, "[]")}"))
    } else {
        expected("index to be in range:[0-${actual.size}) but was:${show(index)}")
    }
}

/**
 * Asserts the ShortArray contains exactly the expected elements. They must be in the same order and
 * there must not be any extra elements.
 * @see [containsAll]
 */
@PlatformName("shortArrayContainsExactly")
fun Assert<ShortArray>.containsExactly(vararg elements: Short) {
    if (actual.contentEquals(elements)) return

    val diff = ListDiffer.diff(elements.asList(), actual.asList())
        .filterNot { it is ListDiffer.Edit.Eq }

    expected(diff.joinToString(prefix = "to contain exactly:\n", separator = "\n") { edit ->
        when (edit) {
            is ListDiffer.Edit.Del -> " at index:${edit.oldIndex} expected:${show(edit.oldValue)}"
            is ListDiffer.Edit.Ins -> " at index:${edit.newIndex} unexpected:${show(edit.newValue)}"
            else -> throw IllegalStateException()
        }
    })
}

/**
 * Asserts on each item in the ShortArray. The given lambda will be run for each item.
 *
 * ```
 * assert(shortArrayOf("one", "two")).each {
 *   it.hasLength(3)
 * }
 * ```
 */
@PlatformName("shortArrayEach")
fun Assert<ShortArray>.each(f: (Assert<Short>) -> Unit) {
    assertAll {
        actual.forEachIndexed { index, item ->
            f(assert(item, "${name ?: ""}${show(index, "[]")}"))
        }
    }
}

/**
 * Returns an assert on the LongArray's size.
 */
@PlatformName("longArraySize")
fun Assert<LongArray>.size() = prop("size", LongArray::size)

/**
 * Asserts the LongArray contents are equal to the expected one, using [contentDeepEquals].
 * @see isNotEqualTo
 */
fun Assert<LongArray>.isEqualTo(expected: LongArray) {
    if (actual.contentEquals(expected)) return
    fail(expected, actual)
}

/**
 * Asserts the LongArray contents are not equal to the expected one, using [contentDeepEquals].
 * @see isEqualTo
 */
fun Assert<LongArray>.isNotEqualTo(expected: LongArray) {
    if (!(actual.contentEquals(expected))) return
    val showExpected = show(expected)
    val showActual = show(actual)
    // if they display the same, only show one.
    if (showExpected == showActual) {
        expected("to not be equal to:$showActual")
    } else {
        expected(":$showExpected not to be equal to:$showActual")
    }
}

/**
 * Asserts the LongArray is empty.
 * @see [isNotEmpty]
 * @see [isNullOrEmpty]
 */
@PlatformName("longArrayIsEmpty")
fun Assert<LongArray>.isEmpty() {
    if (actual.isEmpty()) return
    expected("to be empty but was:${show(actual)}")
}

/**
 * Asserts the LongArray is not empty.
 * @see [isEmpty]
 */
@PlatformName("longArrayIsNotEmpty")
fun Assert<LongArray>.isNotEmpty() {
    if (actual.isNotEmpty()) return
    expected("to not be empty")
}

/**
 * Asserts the LongArray is null or empty.
 * @see [isEmpty]
 */
@PlatformName("longArrayIsNullOrEmpty")
fun Assert<LongArray?>.isNullOrEmpty() {
    if (actual == null || actual.isEmpty()) return
    expected("to be null or empty but was:${show(actual)}")
}

/**
 * Asserts the LongArray has the expected size.
 */
@PlatformName("longArrayHasSize")
fun Assert<LongArray>.hasSize(size: Int) {
    assert(actual.size, "size").isEqualTo(size)
}

/**
 * Asserts the LongArray has the same size as the expected array.
 */
@PlatformName("longArrayHasSameSizeAs")
fun Assert<LongArray>.hasSameSizeAs(other: LongArray) {
    val actualSize = actual.size
    val otherSize = other.size
    if (actualSize == otherSize) return
    expected("to have same size as:${show(other)} ($otherSize) but was size:($actualSize)")
}

/**
 * Asserts the LongArray contains the expected element, using `in`.
 * @see [doesNotContain]
 */
@PlatformName("longArrayContains")
fun Assert<LongArray>.contains(element: Long) {
    if (element in actual) return
    expected("to contain:${show(element)} but was:${show(actual)}")
}

/**
 * Asserts the LongArray does not contain the expected element, using `!in`.
 * @see [contains]
 */
@PlatformName("longArrayDoesNotContain")
fun Assert<LongArray>.doesNotContain(element: Long) {
    if (element !in actual) return
    expected("to not contain:${show(element)} but was:${show(actual)}")
}

/**
 * Asserts the LongArray does not contain any of the expected elements.
 * @see [containsAll]
 */
fun Assert<LongArray>.containsNone(vararg elements: Long) {
    if (elements.none { it in actual }) {
        return
    }

    val notExpected = elements.filter { it in actual }
    expected("to contain none of:${show(elements)} some elements were not expected:${show(notExpected)}")
}

/**
 * Asserts the LongArray contains all the expected elements, in any order. The array may also contain
 * additional elements.
 * @see [containsExactly]
 */
@PlatformName("longArrayContainsAll")
fun Assert<LongArray>.containsAll(vararg elements: Long) {
    if (elements.all { actual.contains(it) }) return
    val notFound = elements.filterNot { it in actual }
    expected("to contain all:${show(elements)} some elements were not found:${show(notFound)}")
}

/**
 * Returns an assert that assertion on the value at the given index in the LongArray.
 *
 * ```
 * assert(longArrayOf(0, 1, 2)).index(1) { it.isPositive() }
 * ```
 */
@PlatformName("longArrayIndex")
fun Assert<LongArray>.index(index: Int, f: (Assert<Long>) -> Unit) {
    if (index in 0 until actual.size) {
        f(assert(actual[index], "${name ?: ""}${show(index, "[]")}"))
    } else {
        expected("index to be in range:[0-${actual.size}) but was:${show(index)}")
    }
}

/**
 * Asserts the LongArray contains exactly the expected elements. They must be in the same order and
 * there must not be any extra elements.
 * @see [containsAll]
 */
@PlatformName("longArrayContainsExactly")
fun Assert<LongArray>.containsExactly(vararg elements: Long) {
    if (actual.contentEquals(elements)) return

    val diff = ListDiffer.diff(elements.asList(), actual.asList())
        .filterNot { it is ListDiffer.Edit.Eq }

    expected(diff.joinToString(prefix = "to contain exactly:\n", separator = "\n") { edit ->
        when (edit) {
            is ListDiffer.Edit.Del -> " at index:${edit.oldIndex} expected:${show(edit.oldValue)}"
            is ListDiffer.Edit.Ins -> " at index:${edit.newIndex} unexpected:${show(edit.newValue)}"
            else -> throw IllegalStateException()
        }
    })
}

/**
 * Asserts on each item in the LongArray. The given lambda will be run for each item.
 *
 * ```
 * assert(longArrayOf("one", "two")).each {
 *   it.hasLength(3)
 * }
 * ```
 */
@PlatformName("longArrayEach")
fun Assert<LongArray>.each(f: (Assert<Long>) -> Unit) {
    assertAll {
        actual.forEachIndexed { index, item ->
            f(assert(item, "${name ?: ""}${show(index, "[]")}"))
        }
    }
}

/**
 * Returns an assert on the FloatArray's size.
 */
@PlatformName("floatArraySize")
fun Assert<FloatArray>.size() = prop("size", FloatArray::size)

/**
 * Asserts the FloatArray contents are equal to the expected one, using [contentDeepEquals].
 * @see isNotEqualTo
 */
fun Assert<FloatArray>.isEqualTo(expected: FloatArray) {
    if (actual.contentEquals(expected)) return
    fail(expected, actual)
}

/**
 * Asserts the FloatArray contents are not equal to the expected one, using [contentDeepEquals].
 * @see isEqualTo
 */
fun Assert<FloatArray>.isNotEqualTo(expected: FloatArray) {
    if (!(actual.contentEquals(expected))) return
    val showExpected = show(expected)
    val showActual = show(actual)
    // if they display the same, only show one.
    if (showExpected == showActual) {
        expected("to not be equal to:$showActual")
    } else {
        expected(":$showExpected not to be equal to:$showActual")
    }
}

/**
 * Asserts the FloatArray is empty.
 * @see [isNotEmpty]
 * @see [isNullOrEmpty]
 */
@PlatformName("floatArrayIsEmpty")
fun Assert<FloatArray>.isEmpty() {
    if (actual.isEmpty()) return
    expected("to be empty but was:${show(actual)}")
}

/**
 * Asserts the FloatArray is not empty.
 * @see [isEmpty]
 */
@PlatformName("floatArrayIsNotEmpty")
fun Assert<FloatArray>.isNotEmpty() {
    if (actual.isNotEmpty()) return
    expected("to not be empty")
}

/**
 * Asserts the FloatArray is null or empty.
 * @see [isEmpty]
 */
@PlatformName("floatArrayIsNullOrEmpty")
fun Assert<FloatArray?>.isNullOrEmpty() {
    if (actual == null || actual.isEmpty()) return
    expected("to be null or empty but was:${show(actual)}")
}

/**
 * Asserts the FloatArray has the expected size.
 */
@PlatformName("floatArrayHasSize")
fun Assert<FloatArray>.hasSize(size: Int) {
    assert(actual.size, "size").isEqualTo(size)
}

/**
 * Asserts the FloatArray has the same size as the expected array.
 */
@PlatformName("floatArrayHasSameSizeAs")
fun Assert<FloatArray>.hasSameSizeAs(other: FloatArray) {
    val actualSize = actual.size
    val otherSize = other.size
    if (actualSize == otherSize) return
    expected("to have same size as:${show(other)} ($otherSize) but was size:($actualSize)")
}

/**
 * Asserts the FloatArray contains the expected element, using `in`.
 * @see [doesNotContain]
 */
@PlatformName("floatArrayContains")
fun Assert<FloatArray>.contains(element: Float) {
    if (element in actual) return
    expected("to contain:${show(element)} but was:${show(actual)}")
}

/**
 * Asserts the FloatArray does not contain the expected element, using `!in`.
 * @see [contains]
 */
@PlatformName("floatArrayDoesNotContain")
fun Assert<FloatArray>.doesNotContain(element: Float) {
    if (element !in actual) return
    expected("to not contain:${show(element)} but was:${show(actual)}")
}

/**
 * Asserts the FloatArray does not contain any of the expected elements.
 * @see [containsAll]
 */
fun Assert<FloatArray>.containsNone(vararg elements: Float) {
    if (elements.none { it in actual }) {
        return
    }

    val notExpected = elements.filter { it in actual }
    expected("to contain none of:${show(elements)} some elements were not expected:${show(notExpected)}")
}

/**
 * Asserts the FloatArray contains all the expected elements, in any order. The array may also contain
 * additional elements.
 * @see [containsExactly]
 */
@PlatformName("floatArrayContainsAll")
fun Assert<FloatArray>.containsAll(vararg elements: Float) {
    if (elements.all { actual.contains(it) }) return
    val notFound = elements.filterNot { it in actual }
    expected("to contain all:${show(elements)} some elements were not found:${show(notFound)}")
}

/**
 * Returns an assert that assertion on the value at the given index in the FloatArray.
 *
 * ```
 * assert(floatArrayOf(0, 1, 2)).index(1) { it.isPositive() }
 * ```
 */
@PlatformName("floatArrayIndex")
fun Assert<FloatArray>.index(index: Int, f: (Assert<Float>) -> Unit) {
    if (index in 0 until actual.size) {
        f(assert(actual[index], "${name ?: ""}${show(index, "[]")}"))
    } else {
        expected("index to be in range:[0-${actual.size}) but was:${show(index)}")
    }
}

/**
 * Asserts the FloatArray contains exactly the expected elements. They must be in the same order and
 * there must not be any extra elements.
 * @see [containsAll]
 */
@PlatformName("floatArrayContainsExactly")
fun Assert<FloatArray>.containsExactly(vararg elements: Float) {
    if (actual.contentEquals(elements)) return

    val diff = ListDiffer.diff(elements.asList(), actual.asList())
        .filterNot { it is ListDiffer.Edit.Eq }

    expected(diff.joinToString(prefix = "to contain exactly:\n", separator = "\n") { edit ->
        when (edit) {
            is ListDiffer.Edit.Del -> " at index:${edit.oldIndex} expected:${show(edit.oldValue)}"
            is ListDiffer.Edit.Ins -> " at index:${edit.newIndex} unexpected:${show(edit.newValue)}"
            else -> throw IllegalStateException()
        }
    })
}

/**
 * Asserts on each item in the FloatArray. The given lambda will be run for each item.
 *
 * ```
 * assert(floatArrayOf("one", "two")).each {
 *   it.hasLength(3)
 * }
 * ```
 */
@PlatformName("floatArrayEach")
fun Assert<FloatArray>.each(f: (Assert<Float>) -> Unit) {
    assertAll {
        actual.forEachIndexed { index, item ->
            f(assert(item, "${name ?: ""}${show(index, "[]")}"))
        }
    }
}

/**
 * Returns an assert on the DoubleArray's size.
 */
@PlatformName("doubleArraySize")
fun Assert<DoubleArray>.size() = prop("size", DoubleArray::size)

/**
 * Asserts the DoubleArray contents are equal to the expected one, using [contentDeepEquals].
 * @see isNotEqualTo
 */
fun Assert<DoubleArray>.isEqualTo(expected: DoubleArray) {
    if (actual.contentEquals(expected)) return
    fail(expected, actual)
}

/**
 * Asserts the DoubleArray contents are not equal to the expected one, using [contentDeepEquals].
 * @see isEqualTo
 */
fun Assert<DoubleArray>.isNotEqualTo(expected: DoubleArray) {
    if (!(actual.contentEquals(expected))) return
    val showExpected = show(expected)
    val showActual = show(actual)
    // if they display the same, only show one.
    if (showExpected == showActual) {
        expected("to not be equal to:$showActual")
    } else {
        expected(":$showExpected not to be equal to:$showActual")
    }
}

/**
 * Asserts the DoubleArray is empty.
 * @see [isNotEmpty]
 * @see [isNullOrEmpty]
 */
@PlatformName("doubleArrayIsEmpty")
fun Assert<DoubleArray>.isEmpty() {
    if (actual.isEmpty()) return
    expected("to be empty but was:${show(actual)}")
}

/**
 * Asserts the DoubleArray is not empty.
 * @see [isEmpty]
 */
@PlatformName("doubleArrayIsNotEmpty")
fun Assert<DoubleArray>.isNotEmpty() {
    if (actual.isNotEmpty()) return
    expected("to not be empty")
}

/**
 * Asserts the DoubleArray is null or empty.
 * @see [isEmpty]
 */
@PlatformName("doubleArrayIsNullOrEmpty")
fun Assert<DoubleArray?>.isNullOrEmpty() {
    if (actual == null || actual.isEmpty()) return
    expected("to be null or empty but was:${show(actual)}")
}

/**
 * Asserts the DoubleArray has the expected size.
 */
@PlatformName("doubleArrayHasSize")
fun Assert<DoubleArray>.hasSize(size: Int) {
    assert(actual.size, "size").isEqualTo(size)
}

/**
 * Asserts the DoubleArray has the same size as the expected array.
 */
@PlatformName("doubleArrayHasSameSizeAs")
fun Assert<DoubleArray>.hasSameSizeAs(other: DoubleArray) {
    val actualSize = actual.size
    val otherSize = other.size
    if (actualSize == otherSize) return
    expected("to have same size as:${show(other)} ($otherSize) but was size:($actualSize)")
}

/**
 * Asserts the DoubleArray contains the expected element, using `in`.
 * @see [doesNotContain]
 */
@PlatformName("doubleArrayContains")
fun Assert<DoubleArray>.contains(element: Double) {
    if (element in actual) return
    expected("to contain:${show(element)} but was:${show(actual)}")
}

/**
 * Asserts the DoubleArray does not contain the expected element, using `!in`.
 * @see [contains]
 */
@PlatformName("doubleArrayDoesNotContain")
fun Assert<DoubleArray>.doesNotContain(element: Double) {
    if (element !in actual) return
    expected("to not contain:${show(element)} but was:${show(actual)}")
}

/**
 * Asserts the DoubleArray does not contain any of the expected elements.
 * @see [containsAll]
 */
fun Assert<DoubleArray>.containsNone(vararg elements: Double) {
    if (elements.none { it in actual }) {
        return
    }

    val notExpected = elements.filter { it in actual }
    expected("to contain none of:${show(elements)} some elements were not expected:${show(notExpected)}")
}

/**
 * Asserts the DoubleArray contains all the expected elements, in any order. The array may also contain
 * additional elements.
 * @see [containsExactly]
 */
@PlatformName("doubleArrayContainsAll")
fun Assert<DoubleArray>.containsAll(vararg elements: Double) {
    if (elements.all { actual.contains(it) }) return
    val notFound = elements.filterNot { it in actual }
    expected("to contain all:${show(elements)} some elements were not found:${show(notFound)}")
}

/**
 * Returns an assert that assertion on the value at the given index in the DoubleArray.
 *
 * ```
 * assert(doubleArrayOf(0, 1, 2)).index(1) { it.isPositive() }
 * ```
 */
@PlatformName("doubleArrayIndex")
fun Assert<DoubleArray>.index(index: Int, f: (Assert<Double>) -> Unit) {
    if (index in 0 until actual.size) {
        f(assert(actual[index], "${name ?: ""}${show(index, "[]")}"))
    } else {
        expected("index to be in range:[0-${actual.size}) but was:${show(index)}")
    }
}

/**
 * Asserts the DoubleArray contains exactly the expected elements. They must be in the same order and
 * there must not be any extra elements.
 * @see [containsAll]
 */
@PlatformName("doubleArrayContainsExactly")
fun Assert<DoubleArray>.containsExactly(vararg elements: Double) {
    if (actual.contentEquals(elements)) return

    val diff = ListDiffer.diff(elements.asList(), actual.asList())
        .filterNot { it is ListDiffer.Edit.Eq }

    expected(diff.joinToString(prefix = "to contain exactly:\n", separator = "\n") { edit ->
        when (edit) {
            is ListDiffer.Edit.Del -> " at index:${edit.oldIndex} expected:${show(edit.oldValue)}"
            is ListDiffer.Edit.Ins -> " at index:${edit.newIndex} unexpected:${show(edit.newValue)}"
            else -> throw IllegalStateException()
        }
    })
}

/**
 * Asserts on each item in the DoubleArray. The given lambda will be run for each item.
 *
 * ```
 * assert(doubleArrayOf("one", "two")).each {
 *   it.hasLength(3)
 * }
 * ```
 */
@PlatformName("doubleArrayEach")
fun Assert<DoubleArray>.each(f: (Assert<Double>) -> Unit) {
    assertAll {
        actual.forEachIndexed { index, item ->
            f(assert(item, "${name ?: ""}${show(index, "[]")}"))
        }
    }
}

/**
 * Returns an assert on the CharArray's size.
 */
@PlatformName("charArraySize")
fun Assert<CharArray>.size() = prop("size", CharArray::size)

/**
 * Asserts the CharArray contents are equal to the expected one, using [contentDeepEquals].
 * @see isNotEqualTo
 */
fun Assert<CharArray>.isEqualTo(expected: CharArray) {
    if (actual.contentEquals(expected)) return
    fail(expected, actual)
}

/**
 * Asserts the CharArray contents are not equal to the expected one, using [contentDeepEquals].
 * @see isEqualTo
 */
fun Assert<CharArray>.isNotEqualTo(expected: CharArray) {
    if (!(actual.contentEquals(expected))) return
    val showExpected = show(expected)
    val showActual = show(actual)
    // if they display the same, only show one.
    if (showExpected == showActual) {
        expected("to not be equal to:$showActual")
    } else {
        expected(":$showExpected not to be equal to:$showActual")
    }
}

/**
 * Asserts the CharArray is empty.
 * @see [isNotEmpty]
 * @see [isNullOrEmpty]
 */
@PlatformName("charArrayIsEmpty")
fun Assert<CharArray>.isEmpty() {
    if (actual.isEmpty()) return
    expected("to be empty but was:${show(actual)}")
}

/**
 * Asserts the CharArray is not empty.
 * @see [isEmpty]
 */
@PlatformName("charArrayIsNotEmpty")
fun Assert<CharArray>.isNotEmpty() {
    if (actual.isNotEmpty()) return
    expected("to not be empty")
}

/**
 * Asserts the CharArray is null or empty.
 * @see [isEmpty]
 */
@PlatformName("charArrayIsNullOrEmpty")
fun Assert<CharArray?>.isNullOrEmpty() {
    if (actual == null || actual.isEmpty()) return
    expected("to be null or empty but was:${show(actual)}")
}

/**
 * Asserts the CharArray has the expected size.
 */
@PlatformName("charArrayHasSize")
fun Assert<CharArray>.hasSize(size: Int) {
    assert(actual.size, "size").isEqualTo(size)
}

/**
 * Asserts the CharArray has the same size as the expected array.
 */
@PlatformName("charArrayHasSameSizeAs")
fun Assert<CharArray>.hasSameSizeAs(other: CharArray) {
    val actualSize = actual.size
    val otherSize = other.size
    if (actualSize == otherSize) return
    expected("to have same size as:${show(other)} ($otherSize) but was size:($actualSize)")
}

/**
 * Asserts the CharArray contains the expected element, using `in`.
 * @see [doesNotContain]
 */
@PlatformName("charArrayContains")
fun Assert<CharArray>.contains(element: Char) {
    if (element in actual) return
    expected("to contain:${show(element)} but was:${show(actual)}")
}

/**
 * Asserts the CharArray does not contain the expected element, using `!in`.
 * @see [contains]
 */
@PlatformName("charArrayDoesNotContain")
fun Assert<CharArray>.doesNotContain(element: Char) {
    if (element !in actual) return
    expected("to not contain:${show(element)} but was:${show(actual)}")
}

/**
 * Asserts the CharArray does not contain any of the expected elements.
 * @see [containsAll]
 */
fun Assert<CharArray>.containsNone(vararg elements: Char) {
    if (elements.none { it in actual }) {
        return
    }

    val notExpected = elements.filter { it in actual }
    expected("to contain none of:${show(elements)} some elements were not expected:${show(notExpected)}")
}

/**
 * Asserts the CharArray contains all the expected elements, in any order. The array may also contain
 * additional elements.
 * @see [containsExactly]
 */
@PlatformName("charArrayContainsAll")
fun Assert<CharArray>.containsAll(vararg elements: Char) {
    if (elements.all { actual.contains(it) }) return
    val notFound = elements.filterNot { it in actual }
    expected("to contain all:${show(elements)} some elements were not found:${show(notFound)}")
}

/**
 * Returns an assert that assertion on the value at the given index in the CharArray.
 *
 * ```
 * assert(charArrayOf(0, 1, 2)).index(1) { it.isPositive() }
 * ```
 */
@PlatformName("charArrayIndex")
fun Assert<CharArray>.index(index: Int, f: (Assert<Char>) -> Unit) {
    if (index in 0 until actual.size) {
        f(assert(actual[index], "${name ?: ""}${show(index, "[]")}"))
    } else {
        expected("index to be in range:[0-${actual.size}) but was:${show(index)}")
    }
}

/**
 * Asserts the CharArray contains exactly the expected elements. They must be in the same order and
 * there must not be any extra elements.
 * @see [containsAll]
 */
@PlatformName("charArrayContainsExactly")
fun Assert<CharArray>.containsExactly(vararg elements: Char) {
    if (actual.contentEquals(elements)) return

    val diff = ListDiffer.diff(elements.asList(), actual.asList())
        .filterNot { it is ListDiffer.Edit.Eq }

    expected(diff.joinToString(prefix = "to contain exactly:\n", separator = "\n") { edit ->
        when (edit) {
            is ListDiffer.Edit.Del -> " at index:${edit.oldIndex} expected:${show(edit.oldValue)}"
            is ListDiffer.Edit.Ins -> " at index:${edit.newIndex} unexpected:${show(edit.newValue)}"
            else -> throw IllegalStateException()
        }
    })
}

/**
 * Asserts on each item in the CharArray. The given lambda will be run for each item.
 *
 * ```
 * assert(charArrayOf("one", "two")).each {
 *   it.hasLength(3)
 * }
 * ```
 */
@PlatformName("charArrayEach")
fun Assert<CharArray>.each(f: (Assert<Char>) -> Unit) {
    assertAll {
        actual.forEachIndexed { index, item ->
            f(assert(item, "${name ?: ""}${show(index, "[]")}"))
        }
    }
}

