package assertk.assertions

import assertk.Assert
import assertk.PlatformName
import assertk.all
import assertk.assertions.support.expected
import assertk.assertions.support.show
import assertk.assertions.support.fail

/**
 * Returns an assert on the ByteArray's size.
 */
@PlatformName("byteArraySize")
fun Assert<ByteArray>.size() = prop("size") { it.size }

/**
 * Asserts the ByteArray contents are equal to the expected one, using [contentDeepEquals].
 * @see isNotEqualTo
 */
fun Assert<ByteArray>.isEqualTo(expected: ByteArray) = given { actual ->
    if (actual.contentEquals(expected)) return
    fail(expected, actual)
}

/**
 * Asserts the ByteArray contents are not equal to the expected one, using [contentDeepEquals].
 * @see isEqualTo
 */
fun Assert<ByteArray>.isNotEqualTo(expected: ByteArray) = given { actual ->
    if (!(actual.contentEquals(expected))) return
    val showExpected = show(expected)
    val showActual = show(actual)
    // if they display the same, only show one.
    if (showExpected == showActual) {
        expected("to not be equal to:$showActual")
    } else {
        expected(":$showExpected not to be equal to:$showActual")
    }
}

/**
 * Asserts the ByteArray is empty.
 * @see [isNotEmpty]
 * @see [isNullOrEmpty]
 */
@PlatformName("byteArrayIsEmpty")
fun Assert<ByteArray>.isEmpty() = given { actual ->
    if (actual.isEmpty()) return
    expected("to be empty but was:${show(actual)}")
}

/**
 * Asserts the ByteArray is not empty.
 * @see [isEmpty]
 */
@PlatformName("byteArrayIsNotEmpty")
fun Assert<ByteArray>.isNotEmpty() = given { actual ->
    if (actual.isNotEmpty()) return
    expected("to not be empty")
}

/**
 * Asserts the ByteArray is null or empty.
 * @see [isEmpty]
 */
@PlatformName("byteArrayIsNullOrEmpty")
fun Assert<ByteArray?>.isNullOrEmpty() = given { actual ->
    if (actual == null || actual.isEmpty()) return
    expected("to be null or empty but was:${show(actual)}")
}

/**
 * Asserts the ByteArray has the expected size.
 */
@PlatformName("byteArrayHasSize")
fun Assert<ByteArray>.hasSize(size: Int) {
    size().isEqualTo(size)
}

/**
 * Asserts the ByteArray has the same size as the expected array.
 */
@PlatformName("byteArrayHasSameSizeAs")
fun Assert<ByteArray>.hasSameSizeAs(other: ByteArray) = given { actual ->
    val actualSize = actual.size
    val otherSize = other.size
    if (actualSize == otherSize) return
    expected("to have same size as:${show(other)} ($otherSize) but was size:($actualSize)")
}

/**
 * Asserts the ByteArray contains the expected element, using `in`.
 * @see [doesNotContain]
 */
@PlatformName("byteArrayContains")
fun Assert<ByteArray>.contains(element: Byte) = given { actual ->
    if (element in actual) return
    expected("to contain:${show(element)} but was:${show(actual)}")
}

/**
 * Asserts the ByteArray does not contain the expected element, using `!in`.
 * @see [contains]
 */
@PlatformName("byteArrayDoesNotContain")
fun Assert<ByteArray>.doesNotContain(element: Byte) = given { actual ->
    if (element !in actual) return
    expected("to not contain:${show(element)} but was:${show(actual)}")
}

/**
 * Asserts the ByteArray does not contain any of the expected elements.
 * @see [containsAll]
 */
fun Assert<ByteArray>.containsNone(vararg elements: Byte) = given { actual ->
    if (elements.none { it in actual }) {
        return
    }

    val notExpected = elements.filter { it in actual }
    expected("to contain none of:${show(elements)} some elements were not expected:${show(notExpected)}")
}

/**
 * Asserts the ByteArray contains all the expected elements, in any order. The array may also contain
 * additional elements.
 * @see [containsExactly]
 */
@PlatformName("byteArrayContainsAll")
fun Assert<ByteArray>.containsAll(vararg elements: Byte) = given { actual ->
    if (elements.all { actual.contains(it) }) return
    val notFound = elements.filterNot { it in actual }
    expected("to contain all:${show(elements)} some elements were not found:${show(notFound)}")
}

/**
 * Returns an assert that assertion on the value at the given index in the array.
 *
 * ```
 * assertThat(byteArrayOf(0, 1, 2)).index(1) { it.isPositive() }
 * ```
 */
@PlatformName("byteArrayIndexOld")
@Deprecated(message = "Use index(index) instead.", replaceWith = ReplaceWith("index(index).let(f)"))
fun Assert<ByteArray>.index(index: Int, f: (Assert<Byte>) -> Unit) {
    index(index).let(f)
}

/**
 * Returns an assert that assertion on the value at the given index in the array.
 *
 * ```
 * assertThat(byteArrayOf(0, 1, 2)).index(1).isPositive()
 * ```
 */
@PlatformName("byteArrayIndex")
fun Assert<ByteArray>.index(index: Int): Assert<Byte> =
    transform("${name ?: ""}${show(index, "[]")}") { actual ->
        if (index in 0 until actual.size) {
            actual[index]
        } else {
            expected("index to be in range:[0-${actual.size}) but was:${show(index)}")
        }
    }

/**
 * Asserts the ByteArray contains exactly the expected elements. They must be in the same order and
 * there must not be any extra elements.
 * @see [containsAll]
 */
@PlatformName("byteArrayContainsExactly")
fun Assert<ByteArray>.containsExactly(vararg elements: Byte) = given { actual ->
    if (actual.contentEquals(elements)) return

    expected(listDifferExpected(elements.toList(), actual.toList()))
}

/**
 * Asserts on each item in the ByteArray. The given lambda will be run for each item.
 *
 * ```
 * assertThat(byteArrayOf("one", "two")).each {
 *   it.hasLength(3)
 * }
 * ```
 */
@PlatformName("byteArrayEach")
fun Assert<ByteArray>.each(f: (Assert<Byte>) -> Unit) = given { actual ->
    all {
        actual.forEachIndexed { index, item ->
            f(assertThat(item, name = "${name ?: ""}${show(index, "[]")}"))
        }
    }
}

/**
 * Returns an assert on the IntArray's size.
 */
@PlatformName("intArraySize")
fun Assert<IntArray>.size() = prop("size") { it.size }

/**
 * Asserts the IntArray contents are equal to the expected one, using [contentDeepEquals].
 * @see isNotEqualTo
 */
fun Assert<IntArray>.isEqualTo(expected: IntArray) = given { actual ->
    if (actual.contentEquals(expected)) return
    fail(expected, actual)
}

/**
 * Asserts the IntArray contents are not equal to the expected one, using [contentDeepEquals].
 * @see isEqualTo
 */
fun Assert<IntArray>.isNotEqualTo(expected: IntArray) = given { actual ->
    if (!(actual.contentEquals(expected))) return
    val showExpected = show(expected)
    val showActual = show(actual)
    // if they display the same, only show one.
    if (showExpected == showActual) {
        expected("to not be equal to:$showActual")
    } else {
        expected(":$showExpected not to be equal to:$showActual")
    }
}

/**
 * Asserts the IntArray is empty.
 * @see [isNotEmpty]
 * @see [isNullOrEmpty]
 */
@PlatformName("intArrayIsEmpty")
fun Assert<IntArray>.isEmpty() = given { actual ->
    if (actual.isEmpty()) return
    expected("to be empty but was:${show(actual)}")
}

/**
 * Asserts the IntArray is not empty.
 * @see [isEmpty]
 */
@PlatformName("intArrayIsNotEmpty")
fun Assert<IntArray>.isNotEmpty() = given { actual ->
    if (actual.isNotEmpty()) return
    expected("to not be empty")
}

/**
 * Asserts the IntArray is null or empty.
 * @see [isEmpty]
 */
@PlatformName("intArrayIsNullOrEmpty")
fun Assert<IntArray?>.isNullOrEmpty() = given { actual ->
    if (actual == null || actual.isEmpty()) return
    expected("to be null or empty but was:${show(actual)}")
}

/**
 * Asserts the IntArray has the expected size.
 */
@PlatformName("intArrayHasSize")
fun Assert<IntArray>.hasSize(size: Int) {
    size().isEqualTo(size)
}

/**
 * Asserts the IntArray has the same size as the expected array.
 */
@PlatformName("intArrayHasSameSizeAs")
fun Assert<IntArray>.hasSameSizeAs(other: IntArray) = given { actual ->
    val actualSize = actual.size
    val otherSize = other.size
    if (actualSize == otherSize) return
    expected("to have same size as:${show(other)} ($otherSize) but was size:($actualSize)")
}

/**
 * Asserts the IntArray contains the expected element, using `in`.
 * @see [doesNotContain]
 */
@PlatformName("intArrayContains")
fun Assert<IntArray>.contains(element: Int) = given { actual ->
    if (element in actual) return
    expected("to contain:${show(element)} but was:${show(actual)}")
}

/**
 * Asserts the IntArray does not contain the expected element, using `!in`.
 * @see [contains]
 */
@PlatformName("intArrayDoesNotContain")
fun Assert<IntArray>.doesNotContain(element: Int) = given { actual ->
    if (element !in actual) return
    expected("to not contain:${show(element)} but was:${show(actual)}")
}

/**
 * Asserts the IntArray does not contain any of the expected elements.
 * @see [containsAll]
 */
fun Assert<IntArray>.containsNone(vararg elements: Int) = given { actual ->
    if (elements.none { it in actual }) {
        return
    }

    val notExpected = elements.filter { it in actual }
    expected("to contain none of:${show(elements)} some elements were not expected:${show(notExpected)}")
}

/**
 * Asserts the IntArray contains all the expected elements, in any order. The array may also contain
 * additional elements.
 * @see [containsExactly]
 */
@PlatformName("intArrayContainsAll")
fun Assert<IntArray>.containsAll(vararg elements: Int) = given { actual ->
    if (elements.all { actual.contains(it) }) return
    val notFound = elements.filterNot { it in actual }
    expected("to contain all:${show(elements)} some elements were not found:${show(notFound)}")
}

/**
 * Returns an assert that assertion on the value at the given index in the array.
 *
 * ```
 * assertThat(intArrayOf(0, 1, 2)).index(1) { it.isPositive() }
 * ```
 */
@PlatformName("intArrayIndexOld")
@Deprecated(message = "Use index(index) instead.", replaceWith = ReplaceWith("index(index).let(f)"))
fun Assert<IntArray>.index(index: Int, f: (Assert<Int>) -> Unit) {
    index(index).let(f)
}

/**
 * Returns an assert that assertion on the value at the given index in the array.
 *
 * ```
 * assertThat(intArrayOf(0, 1, 2)).index(1).isPositive()
 * ```
 */
@PlatformName("intArrayIndex")
fun Assert<IntArray>.index(index: Int): Assert<Int> =
    transform("${name ?: ""}${show(index, "[]")}") { actual ->
        if (index in 0 until actual.size) {
            actual[index]
        } else {
            expected("index to be in range:[0-${actual.size}) but was:${show(index)}")
        }
    }

/**
 * Asserts the IntArray contains exactly the expected elements. They must be in the same order and
 * there must not be any extra elements.
 * @see [containsAll]
 */
@PlatformName("intArrayContainsExactly")
fun Assert<IntArray>.containsExactly(vararg elements: Int) = given { actual ->
    if (actual.contentEquals(elements)) return

    expected(listDifferExpected(elements.toList(), actual.toList()))
}

/**
 * Asserts on each item in the IntArray. The given lambda will be run for each item.
 *
 * ```
 * assertThat(intArrayOf("one", "two")).each {
 *   it.hasLength(3)
 * }
 * ```
 */
@PlatformName("intArrayEach")
fun Assert<IntArray>.each(f: (Assert<Int>) -> Unit) = given { actual ->
    all {
        actual.forEachIndexed { index, item ->
            f(assertThat(item, name = "${name ?: ""}${show(index, "[]")}"))
        }
    }
}

/**
 * Returns an assert on the ShortArray's size.
 */
@PlatformName("shortArraySize")
fun Assert<ShortArray>.size() = prop("size") { it.size }

/**
 * Asserts the ShortArray contents are equal to the expected one, using [contentDeepEquals].
 * @see isNotEqualTo
 */
fun Assert<ShortArray>.isEqualTo(expected: ShortArray) = given { actual ->
    if (actual.contentEquals(expected)) return
    fail(expected, actual)
}

/**
 * Asserts the ShortArray contents are not equal to the expected one, using [contentDeepEquals].
 * @see isEqualTo
 */
fun Assert<ShortArray>.isNotEqualTo(expected: ShortArray) = given { actual ->
    if (!(actual.contentEquals(expected))) return
    val showExpected = show(expected)
    val showActual = show(actual)
    // if they display the same, only show one.
    if (showExpected == showActual) {
        expected("to not be equal to:$showActual")
    } else {
        expected(":$showExpected not to be equal to:$showActual")
    }
}

/**
 * Asserts the ShortArray is empty.
 * @see [isNotEmpty]
 * @see [isNullOrEmpty]
 */
@PlatformName("shortArrayIsEmpty")
fun Assert<ShortArray>.isEmpty() = given { actual ->
    if (actual.isEmpty()) return
    expected("to be empty but was:${show(actual)}")
}

/**
 * Asserts the ShortArray is not empty.
 * @see [isEmpty]
 */
@PlatformName("shortArrayIsNotEmpty")
fun Assert<ShortArray>.isNotEmpty() = given { actual ->
    if (actual.isNotEmpty()) return
    expected("to not be empty")
}

/**
 * Asserts the ShortArray is null or empty.
 * @see [isEmpty]
 */
@PlatformName("shortArrayIsNullOrEmpty")
fun Assert<ShortArray?>.isNullOrEmpty() = given { actual ->
    if (actual == null || actual.isEmpty()) return
    expected("to be null or empty but was:${show(actual)}")
}

/**
 * Asserts the ShortArray has the expected size.
 */
@PlatformName("shortArrayHasSize")
fun Assert<ShortArray>.hasSize(size: Int) {
    size().isEqualTo(size)
}

/**
 * Asserts the ShortArray has the same size as the expected array.
 */
@PlatformName("shortArrayHasSameSizeAs")
fun Assert<ShortArray>.hasSameSizeAs(other: ShortArray) = given { actual ->
    val actualSize = actual.size
    val otherSize = other.size
    if (actualSize == otherSize) return
    expected("to have same size as:${show(other)} ($otherSize) but was size:($actualSize)")
}

/**
 * Asserts the ShortArray contains the expected element, using `in`.
 * @see [doesNotContain]
 */
@PlatformName("shortArrayContains")
fun Assert<ShortArray>.contains(element: Short) = given { actual ->
    if (element in actual) return
    expected("to contain:${show(element)} but was:${show(actual)}")
}

/**
 * Asserts the ShortArray does not contain the expected element, using `!in`.
 * @see [contains]
 */
@PlatformName("shortArrayDoesNotContain")
fun Assert<ShortArray>.doesNotContain(element: Short) = given { actual ->
    if (element !in actual) return
    expected("to not contain:${show(element)} but was:${show(actual)}")
}

/**
 * Asserts the ShortArray does not contain any of the expected elements.
 * @see [containsAll]
 */
fun Assert<ShortArray>.containsNone(vararg elements: Short) = given { actual ->
    if (elements.none { it in actual }) {
        return
    }

    val notExpected = elements.filter { it in actual }
    expected("to contain none of:${show(elements)} some elements were not expected:${show(notExpected)}")
}

/**
 * Asserts the ShortArray contains all the expected elements, in any order. The array may also contain
 * additional elements.
 * @see [containsExactly]
 */
@PlatformName("shortArrayContainsAll")
fun Assert<ShortArray>.containsAll(vararg elements: Short) = given { actual ->
    if (elements.all { actual.contains(it) }) return
    val notFound = elements.filterNot { it in actual }
    expected("to contain all:${show(elements)} some elements were not found:${show(notFound)}")
}

/**
 * Returns an assert that assertion on the value at the given index in the array.
 *
 * ```
 * assertThat(shortArrayOf(0, 1, 2)).index(1) { it.isPositive() }
 * ```
 */
@PlatformName("shortArrayIndexOld")
@Deprecated(message = "Use index(index) instead.", replaceWith = ReplaceWith("index(index).let(f)"))
fun Assert<ShortArray>.index(index: Int, f: (Assert<Short>) -> Unit) {
    index(index).let(f)
}

/**
 * Returns an assert that assertion on the value at the given index in the array.
 *
 * ```
 * assertThat(shortArrayOf(0, 1, 2)).index(1).isPositive()
 * ```
 */
@PlatformName("shortArrayIndex")
fun Assert<ShortArray>.index(index: Int): Assert<Short> =
    transform("${name ?: ""}${show(index, "[]")}") { actual ->
        if (index in 0 until actual.size) {
            actual[index]
        } else {
            expected("index to be in range:[0-${actual.size}) but was:${show(index)}")
        }
    }

/**
 * Asserts the ShortArray contains exactly the expected elements. They must be in the same order and
 * there must not be any extra elements.
 * @see [containsAll]
 */
@PlatformName("shortArrayContainsExactly")
fun Assert<ShortArray>.containsExactly(vararg elements: Short) = given { actual ->
    if (actual.contentEquals(elements)) return

    expected(listDifferExpected(elements.toList(), actual.toList()))
}

/**
 * Asserts on each item in the ShortArray. The given lambda will be run for each item.
 *
 * ```
 * assertThat(shortArrayOf("one", "two")).each {
 *   it.hasLength(3)
 * }
 * ```
 */
@PlatformName("shortArrayEach")
fun Assert<ShortArray>.each(f: (Assert<Short>) -> Unit) = given { actual ->
    all {
        actual.forEachIndexed { index, item ->
            f(assertThat(item, name = "${name ?: ""}${show(index, "[]")}"))
        }
    }
}

/**
 * Returns an assert on the LongArray's size.
 */
@PlatformName("longArraySize")
fun Assert<LongArray>.size() = prop("size") { it.size }

/**
 * Asserts the LongArray contents are equal to the expected one, using [contentDeepEquals].
 * @see isNotEqualTo
 */
fun Assert<LongArray>.isEqualTo(expected: LongArray) = given { actual ->
    if (actual.contentEquals(expected)) return
    fail(expected, actual)
}

/**
 * Asserts the LongArray contents are not equal to the expected one, using [contentDeepEquals].
 * @see isEqualTo
 */
fun Assert<LongArray>.isNotEqualTo(expected: LongArray) = given { actual ->
    if (!(actual.contentEquals(expected))) return
    val showExpected = show(expected)
    val showActual = show(actual)
    // if they display the same, only show one.
    if (showExpected == showActual) {
        expected("to not be equal to:$showActual")
    } else {
        expected(":$showExpected not to be equal to:$showActual")
    }
}

/**
 * Asserts the LongArray is empty.
 * @see [isNotEmpty]
 * @see [isNullOrEmpty]
 */
@PlatformName("longArrayIsEmpty")
fun Assert<LongArray>.isEmpty() = given { actual ->
    if (actual.isEmpty()) return
    expected("to be empty but was:${show(actual)}")
}

/**
 * Asserts the LongArray is not empty.
 * @see [isEmpty]
 */
@PlatformName("longArrayIsNotEmpty")
fun Assert<LongArray>.isNotEmpty() = given { actual ->
    if (actual.isNotEmpty()) return
    expected("to not be empty")
}

/**
 * Asserts the LongArray is null or empty.
 * @see [isEmpty]
 */
@PlatformName("longArrayIsNullOrEmpty")
fun Assert<LongArray?>.isNullOrEmpty() = given { actual ->
    if (actual == null || actual.isEmpty()) return
    expected("to be null or empty but was:${show(actual)}")
}

/**
 * Asserts the LongArray has the expected size.
 */
@PlatformName("longArrayHasSize")
fun Assert<LongArray>.hasSize(size: Int) {
    size().isEqualTo(size)
}

/**
 * Asserts the LongArray has the same size as the expected array.
 */
@PlatformName("longArrayHasSameSizeAs")
fun Assert<LongArray>.hasSameSizeAs(other: LongArray) = given { actual ->
    val actualSize = actual.size
    val otherSize = other.size
    if (actualSize == otherSize) return
    expected("to have same size as:${show(other)} ($otherSize) but was size:($actualSize)")
}

/**
 * Asserts the LongArray contains the expected element, using `in`.
 * @see [doesNotContain]
 */
@PlatformName("longArrayContains")
fun Assert<LongArray>.contains(element: Long) = given { actual ->
    if (element in actual) return
    expected("to contain:${show(element)} but was:${show(actual)}")
}

/**
 * Asserts the LongArray does not contain the expected element, using `!in`.
 * @see [contains]
 */
@PlatformName("longArrayDoesNotContain")
fun Assert<LongArray>.doesNotContain(element: Long) = given { actual ->
    if (element !in actual) return
    expected("to not contain:${show(element)} but was:${show(actual)}")
}

/**
 * Asserts the LongArray does not contain any of the expected elements.
 * @see [containsAll]
 */
fun Assert<LongArray>.containsNone(vararg elements: Long) = given { actual ->
    if (elements.none { it in actual }) {
        return
    }

    val notExpected = elements.filter { it in actual }
    expected("to contain none of:${show(elements)} some elements were not expected:${show(notExpected)}")
}

/**
 * Asserts the LongArray contains all the expected elements, in any order. The array may also contain
 * additional elements.
 * @see [containsExactly]
 */
@PlatformName("longArrayContainsAll")
fun Assert<LongArray>.containsAll(vararg elements: Long) = given { actual ->
    if (elements.all { actual.contains(it) }) return
    val notFound = elements.filterNot { it in actual }
    expected("to contain all:${show(elements)} some elements were not found:${show(notFound)}")
}

/**
 * Returns an assert that assertion on the value at the given index in the array.
 *
 * ```
 * assertThat(longArrayOf(0, 1, 2)).index(1) { it.isPositive() }
 * ```
 */
@PlatformName("longArrayIndexOld")
@Deprecated(message = "Use index(index) instead.", replaceWith = ReplaceWith("index(index).let(f)"))
fun Assert<LongArray>.index(index: Int, f: (Assert<Long>) -> Unit) {
    index(index).let(f)
}

/**
 * Returns an assert that assertion on the value at the given index in the array.
 *
 * ```
 * assertThat(longArrayOf(0, 1, 2)).index(1).isPositive()
 * ```
 */
@PlatformName("longArrayIndex")
fun Assert<LongArray>.index(index: Int): Assert<Long> =
    transform("${name ?: ""}${show(index, "[]")}") { actual ->
        if (index in 0 until actual.size) {
            actual[index]
        } else {
            expected("index to be in range:[0-${actual.size}) but was:${show(index)}")
        }
    }

/**
 * Asserts the LongArray contains exactly the expected elements. They must be in the same order and
 * there must not be any extra elements.
 * @see [containsAll]
 */
@PlatformName("longArrayContainsExactly")
fun Assert<LongArray>.containsExactly(vararg elements: Long) = given { actual ->
    if (actual.contentEquals(elements)) return

    expected(listDifferExpected(elements.toList(), actual.toList()))
}

/**
 * Asserts on each item in the LongArray. The given lambda will be run for each item.
 *
 * ```
 * assertThat(longArrayOf("one", "two")).each {
 *   it.hasLength(3)
 * }
 * ```
 */
@PlatformName("longArrayEach")
fun Assert<LongArray>.each(f: (Assert<Long>) -> Unit) = given { actual ->
    all {
        actual.forEachIndexed { index, item ->
            f(assertThat(item, name = "${name ?: ""}${show(index, "[]")}"))
        }
    }
}

/**
 * Returns an assert on the FloatArray's size.
 */
@PlatformName("floatArraySize")
fun Assert<FloatArray>.size() = prop("size") { it.size }

/**
 * Asserts the FloatArray contents are equal to the expected one, using [contentDeepEquals].
 * @see isNotEqualTo
 */
fun Assert<FloatArray>.isEqualTo(expected: FloatArray) = given { actual ->
    if (actual.contentEquals(expected)) return
    fail(expected, actual)
}

/**
 * Asserts the FloatArray contents are not equal to the expected one, using [contentDeepEquals].
 * @see isEqualTo
 */
fun Assert<FloatArray>.isNotEqualTo(expected: FloatArray) = given { actual ->
    if (!(actual.contentEquals(expected))) return
    val showExpected = show(expected)
    val showActual = show(actual)
    // if they display the same, only show one.
    if (showExpected == showActual) {
        expected("to not be equal to:$showActual")
    } else {
        expected(":$showExpected not to be equal to:$showActual")
    }
}

/**
 * Asserts the FloatArray is empty.
 * @see [isNotEmpty]
 * @see [isNullOrEmpty]
 */
@PlatformName("floatArrayIsEmpty")
fun Assert<FloatArray>.isEmpty() = given { actual ->
    if (actual.isEmpty()) return
    expected("to be empty but was:${show(actual)}")
}

/**
 * Asserts the FloatArray is not empty.
 * @see [isEmpty]
 */
@PlatformName("floatArrayIsNotEmpty")
fun Assert<FloatArray>.isNotEmpty() = given { actual ->
    if (actual.isNotEmpty()) return
    expected("to not be empty")
}

/**
 * Asserts the FloatArray is null or empty.
 * @see [isEmpty]
 */
@PlatformName("floatArrayIsNullOrEmpty")
fun Assert<FloatArray?>.isNullOrEmpty() = given { actual ->
    if (actual == null || actual.isEmpty()) return
    expected("to be null or empty but was:${show(actual)}")
}

/**
 * Asserts the FloatArray has the expected size.
 */
@PlatformName("floatArrayHasSize")
fun Assert<FloatArray>.hasSize(size: Int) {
    size().isEqualTo(size)
}

/**
 * Asserts the FloatArray has the same size as the expected array.
 */
@PlatformName("floatArrayHasSameSizeAs")
fun Assert<FloatArray>.hasSameSizeAs(other: FloatArray) = given { actual ->
    val actualSize = actual.size
    val otherSize = other.size
    if (actualSize == otherSize) return
    expected("to have same size as:${show(other)} ($otherSize) but was size:($actualSize)")
}

/**
 * Asserts the FloatArray contains the expected element, using `in`.
 * @see [doesNotContain]
 */
@PlatformName("floatArrayContains")
fun Assert<FloatArray>.contains(element: Float) = given { actual ->
    if (element in actual) return
    expected("to contain:${show(element)} but was:${show(actual)}")
}

/**
 * Asserts the FloatArray does not contain the expected element, using `!in`.
 * @see [contains]
 */
@PlatformName("floatArrayDoesNotContain")
fun Assert<FloatArray>.doesNotContain(element: Float) = given { actual ->
    if (element !in actual) return
    expected("to not contain:${show(element)} but was:${show(actual)}")
}

/**
 * Asserts the FloatArray does not contain any of the expected elements.
 * @see [containsAll]
 */
fun Assert<FloatArray>.containsNone(vararg elements: Float) = given { actual ->
    if (elements.none { it in actual }) {
        return
    }

    val notExpected = elements.filter { it in actual }
    expected("to contain none of:${show(elements)} some elements were not expected:${show(notExpected)}")
}

/**
 * Asserts the FloatArray contains all the expected elements, in any order. The array may also contain
 * additional elements.
 * @see [containsExactly]
 */
@PlatformName("floatArrayContainsAll")
fun Assert<FloatArray>.containsAll(vararg elements: Float) = given { actual ->
    if (elements.all { actual.contains(it) }) return
    val notFound = elements.filterNot { it in actual }
    expected("to contain all:${show(elements)} some elements were not found:${show(notFound)}")
}

/**
 * Returns an assert that assertion on the value at the given index in the array.
 *
 * ```
 * assertThat(floatArrayOf(0, 1, 2)).index(1) { it.isPositive() }
 * ```
 */
@PlatformName("floatArrayIndexOld")
@Deprecated(message = "Use index(index) instead.", replaceWith = ReplaceWith("index(index).let(f)"))
fun Assert<FloatArray>.index(index: Int, f: (Assert<Float>) -> Unit) {
    index(index).let(f)
}

/**
 * Returns an assert that assertion on the value at the given index in the array.
 *
 * ```
 * assertThat(floatArrayOf(0, 1, 2)).index(1).isPositive()
 * ```
 */
@PlatformName("floatArrayIndex")
fun Assert<FloatArray>.index(index: Int): Assert<Float> =
    transform("${name ?: ""}${show(index, "[]")}") { actual ->
        if (index in 0 until actual.size) {
            actual[index]
        } else {
            expected("index to be in range:[0-${actual.size}) but was:${show(index)}")
        }
    }

/**
 * Asserts the FloatArray contains exactly the expected elements. They must be in the same order and
 * there must not be any extra elements.
 * @see [containsAll]
 */
@PlatformName("floatArrayContainsExactly")
fun Assert<FloatArray>.containsExactly(vararg elements: Float) = given { actual ->
    if (actual.contentEquals(elements)) return

    expected(listDifferExpected(elements.toList(), actual.toList()))
}

/**
 * Asserts on each item in the FloatArray. The given lambda will be run for each item.
 *
 * ```
 * assertThat(floatArrayOf("one", "two")).each {
 *   it.hasLength(3)
 * }
 * ```
 */
@PlatformName("floatArrayEach")
fun Assert<FloatArray>.each(f: (Assert<Float>) -> Unit) = given { actual ->
    all {
        actual.forEachIndexed { index, item ->
            f(assertThat(item, name = "${name ?: ""}${show(index, "[]")}"))
        }
    }
}

/**
 * Returns an assert on the DoubleArray's size.
 */
@PlatformName("doubleArraySize")
fun Assert<DoubleArray>.size() = prop("size") { it.size }

/**
 * Asserts the DoubleArray contents are equal to the expected one, using [contentDeepEquals].
 * @see isNotEqualTo
 */
fun Assert<DoubleArray>.isEqualTo(expected: DoubleArray) = given { actual ->
    if (actual.contentEquals(expected)) return
    fail(expected, actual)
}

/**
 * Asserts the DoubleArray contents are not equal to the expected one, using [contentDeepEquals].
 * @see isEqualTo
 */
fun Assert<DoubleArray>.isNotEqualTo(expected: DoubleArray) = given { actual ->
    if (!(actual.contentEquals(expected))) return
    val showExpected = show(expected)
    val showActual = show(actual)
    // if they display the same, only show one.
    if (showExpected == showActual) {
        expected("to not be equal to:$showActual")
    } else {
        expected(":$showExpected not to be equal to:$showActual")
    }
}

/**
 * Asserts the DoubleArray is empty.
 * @see [isNotEmpty]
 * @see [isNullOrEmpty]
 */
@PlatformName("doubleArrayIsEmpty")
fun Assert<DoubleArray>.isEmpty() = given { actual ->
    if (actual.isEmpty()) return
    expected("to be empty but was:${show(actual)}")
}

/**
 * Asserts the DoubleArray is not empty.
 * @see [isEmpty]
 */
@PlatformName("doubleArrayIsNotEmpty")
fun Assert<DoubleArray>.isNotEmpty() = given { actual ->
    if (actual.isNotEmpty()) return
    expected("to not be empty")
}

/**
 * Asserts the DoubleArray is null or empty.
 * @see [isEmpty]
 */
@PlatformName("doubleArrayIsNullOrEmpty")
fun Assert<DoubleArray?>.isNullOrEmpty() = given { actual ->
    if (actual == null || actual.isEmpty()) return
    expected("to be null or empty but was:${show(actual)}")
}

/**
 * Asserts the DoubleArray has the expected size.
 */
@PlatformName("doubleArrayHasSize")
fun Assert<DoubleArray>.hasSize(size: Int) {
    size().isEqualTo(size)
}

/**
 * Asserts the DoubleArray has the same size as the expected array.
 */
@PlatformName("doubleArrayHasSameSizeAs")
fun Assert<DoubleArray>.hasSameSizeAs(other: DoubleArray) = given { actual ->
    val actualSize = actual.size
    val otherSize = other.size
    if (actualSize == otherSize) return
    expected("to have same size as:${show(other)} ($otherSize) but was size:($actualSize)")
}

/**
 * Asserts the DoubleArray contains the expected element, using `in`.
 * @see [doesNotContain]
 */
@PlatformName("doubleArrayContains")
fun Assert<DoubleArray>.contains(element: Double) = given { actual ->
    if (element in actual) return
    expected("to contain:${show(element)} but was:${show(actual)}")
}

/**
 * Asserts the DoubleArray does not contain the expected element, using `!in`.
 * @see [contains]
 */
@PlatformName("doubleArrayDoesNotContain")
fun Assert<DoubleArray>.doesNotContain(element: Double) = given { actual ->
    if (element !in actual) return
    expected("to not contain:${show(element)} but was:${show(actual)}")
}

/**
 * Asserts the DoubleArray does not contain any of the expected elements.
 * @see [containsAll]
 */
fun Assert<DoubleArray>.containsNone(vararg elements: Double) = given { actual ->
    if (elements.none { it in actual }) {
        return
    }

    val notExpected = elements.filter { it in actual }
    expected("to contain none of:${show(elements)} some elements were not expected:${show(notExpected)}")
}

/**
 * Asserts the DoubleArray contains all the expected elements, in any order. The array may also contain
 * additional elements.
 * @see [containsExactly]
 */
@PlatformName("doubleArrayContainsAll")
fun Assert<DoubleArray>.containsAll(vararg elements: Double) = given { actual ->
    if (elements.all { actual.contains(it) }) return
    val notFound = elements.filterNot { it in actual }
    expected("to contain all:${show(elements)} some elements were not found:${show(notFound)}")
}

/**
 * Returns an assert that assertion on the value at the given index in the array.
 *
 * ```
 * assertThat(doubleArrayOf(0, 1, 2)).index(1) { it.isPositive() }
 * ```
 */
@PlatformName("doubleArrayIndexOld")
@Deprecated(message = "Use index(index) instead.", replaceWith = ReplaceWith("index(index).let(f)"))
fun Assert<DoubleArray>.index(index: Int, f: (Assert<Double>) -> Unit) {
    index(index).let(f)
}

/**
 * Returns an assert that assertion on the value at the given index in the array.
 *
 * ```
 * assertThat(doubleArrayOf(0, 1, 2)).index(1).isPositive()
 * ```
 */
@PlatformName("doubleArrayIndex")
fun Assert<DoubleArray>.index(index: Int): Assert<Double> =
    transform("${name ?: ""}${show(index, "[]")}") { actual ->
        if (index in 0 until actual.size) {
            actual[index]
        } else {
            expected("index to be in range:[0-${actual.size}) but was:${show(index)}")
        }
    }

/**
 * Asserts the DoubleArray contains exactly the expected elements. They must be in the same order and
 * there must not be any extra elements.
 * @see [containsAll]
 */
@PlatformName("doubleArrayContainsExactly")
fun Assert<DoubleArray>.containsExactly(vararg elements: Double) = given { actual ->
    if (actual.contentEquals(elements)) return

    expected(listDifferExpected(elements.toList(), actual.toList()))
}

/**
 * Asserts on each item in the DoubleArray. The given lambda will be run for each item.
 *
 * ```
 * assertThat(doubleArrayOf("one", "two")).each {
 *   it.hasLength(3)
 * }
 * ```
 */
@PlatformName("doubleArrayEach")
fun Assert<DoubleArray>.each(f: (Assert<Double>) -> Unit) = given { actual ->
    all {
        actual.forEachIndexed { index, item ->
            f(assertThat(item, name = "${name ?: ""}${show(index, "[]")}"))
        }
    }
}

/**
 * Returns an assert on the CharArray's size.
 */
@PlatformName("charArraySize")
fun Assert<CharArray>.size() = prop("size") { it.size }

/**
 * Asserts the CharArray contents are equal to the expected one, using [contentDeepEquals].
 * @see isNotEqualTo
 */
fun Assert<CharArray>.isEqualTo(expected: CharArray) = given { actual ->
    if (actual.contentEquals(expected)) return
    fail(expected, actual)
}

/**
 * Asserts the CharArray contents are not equal to the expected one, using [contentDeepEquals].
 * @see isEqualTo
 */
fun Assert<CharArray>.isNotEqualTo(expected: CharArray) = given { actual ->
    if (!(actual.contentEquals(expected))) return
    val showExpected = show(expected)
    val showActual = show(actual)
    // if they display the same, only show one.
    if (showExpected == showActual) {
        expected("to not be equal to:$showActual")
    } else {
        expected(":$showExpected not to be equal to:$showActual")
    }
}

/**
 * Asserts the CharArray is empty.
 * @see [isNotEmpty]
 * @see [isNullOrEmpty]
 */
@PlatformName("charArrayIsEmpty")
fun Assert<CharArray>.isEmpty() = given { actual ->
    if (actual.isEmpty()) return
    expected("to be empty but was:${show(actual)}")
}

/**
 * Asserts the CharArray is not empty.
 * @see [isEmpty]
 */
@PlatformName("charArrayIsNotEmpty")
fun Assert<CharArray>.isNotEmpty() = given { actual ->
    if (actual.isNotEmpty()) return
    expected("to not be empty")
}

/**
 * Asserts the CharArray is null or empty.
 * @see [isEmpty]
 */
@PlatformName("charArrayIsNullOrEmpty")
fun Assert<CharArray?>.isNullOrEmpty() = given { actual ->
    if (actual == null || actual.isEmpty()) return
    expected("to be null or empty but was:${show(actual)}")
}

/**
 * Asserts the CharArray has the expected size.
 */
@PlatformName("charArrayHasSize")
fun Assert<CharArray>.hasSize(size: Int) {
    size().isEqualTo(size)
}

/**
 * Asserts the CharArray has the same size as the expected array.
 */
@PlatformName("charArrayHasSameSizeAs")
fun Assert<CharArray>.hasSameSizeAs(other: CharArray) = given { actual ->
    val actualSize = actual.size
    val otherSize = other.size
    if (actualSize == otherSize) return
    expected("to have same size as:${show(other)} ($otherSize) but was size:($actualSize)")
}

/**
 * Asserts the CharArray contains the expected element, using `in`.
 * @see [doesNotContain]
 */
@PlatformName("charArrayContains")
fun Assert<CharArray>.contains(element: Char) = given { actual ->
    if (element in actual) return
    expected("to contain:${show(element)} but was:${show(actual)}")
}

/**
 * Asserts the CharArray does not contain the expected element, using `!in`.
 * @see [contains]
 */
@PlatformName("charArrayDoesNotContain")
fun Assert<CharArray>.doesNotContain(element: Char) = given { actual ->
    if (element !in actual) return
    expected("to not contain:${show(element)} but was:${show(actual)}")
}

/**
 * Asserts the CharArray does not contain any of the expected elements.
 * @see [containsAll]
 */
fun Assert<CharArray>.containsNone(vararg elements: Char) = given { actual ->
    if (elements.none { it in actual }) {
        return
    }

    val notExpected = elements.filter { it in actual }
    expected("to contain none of:${show(elements)} some elements were not expected:${show(notExpected)}")
}

/**
 * Asserts the CharArray contains all the expected elements, in any order. The array may also contain
 * additional elements.
 * @see [containsExactly]
 */
@PlatformName("charArrayContainsAll")
fun Assert<CharArray>.containsAll(vararg elements: Char) = given { actual ->
    if (elements.all { actual.contains(it) }) return
    val notFound = elements.filterNot { it in actual }
    expected("to contain all:${show(elements)} some elements were not found:${show(notFound)}")
}

/**
 * Returns an assert that assertion on the value at the given index in the array.
 *
 * ```
 * assertThat(charArrayOf(0, 1, 2)).index(1) { it.isPositive() }
 * ```
 */
@PlatformName("charArrayIndexOld")
@Deprecated(message = "Use index(index) instead.", replaceWith = ReplaceWith("index(index).let(f)"))
fun Assert<CharArray>.index(index: Int, f: (Assert<Char>) -> Unit) {
    index(index).let(f)
}

/**
 * Returns an assert that assertion on the value at the given index in the array.
 *
 * ```
 * assertThat(charArrayOf(0, 1, 2)).index(1).isPositive()
 * ```
 */
@PlatformName("charArrayIndex")
fun Assert<CharArray>.index(index: Int): Assert<Char> =
    transform("${name ?: ""}${show(index, "[]")}") { actual ->
        if (index in 0 until actual.size) {
            actual[index]
        } else {
            expected("index to be in range:[0-${actual.size}) but was:${show(index)}")
        }
    }

/**
 * Asserts the CharArray contains exactly the expected elements. They must be in the same order and
 * there must not be any extra elements.
 * @see [containsAll]
 */
@PlatformName("charArrayContainsExactly")
fun Assert<CharArray>.containsExactly(vararg elements: Char) = given { actual ->
    if (actual.contentEquals(elements)) return

    expected(listDifferExpected(elements.toList(), actual.toList()))
}

/**
 * Asserts on each item in the CharArray. The given lambda will be run for each item.
 *
 * ```
 * assertThat(charArrayOf("one", "two")).each {
 *   it.hasLength(3)
 * }
 * ```
 */
@PlatformName("charArrayEach")
fun Assert<CharArray>.each(f: (Assert<Char>) -> Unit) = given { actual ->
    all {
        actual.forEachIndexed { index, item ->
            f(assertThat(item, name = "${name ?: ""}${show(index, "[]")}"))
        }
    }
}

