package com.spring.boxes.dollar;

import java.util.Date;

import lombok.extern.slf4j.Slf4j;

/**
 * 流量打散工具类
 */
@Slf4j
public class DisperseUtils {

    // 弹窗最大等待时间
    public static final Long maxWaitTime = 5000L;

    // 弹窗生命周期时间
    public static final Long durationTime = 2 * 60 * 1000L;

    /**
     * 基于事件时间计算递减周期
     * 初始弹窗50000ms，在2分钟的时间内追减递减直至消亡。
     *
     * @param eventTime 热点事件时间
     * @return 打散时间
     */
    public static long timeDismiss(Date eventTime) {
        return timeDismiss(eventTime, maxWaitTime, durationTime);
    }

    /**
     * 基于事件时间计算递减周期
     * 初始弹窗50000ms，在2分钟的时间内追减递减直至消亡。
     *
     * @param eventTime    热点事件时间
     * @param initialTime  初始值弹窗时间 50000ms
     * @param durationTime 初始弹窗消亡持续总时间长度。示例中的2分钟
     * @return 打散值
     */
    public static long timeDismiss(Date eventTime, long initialTime, long durationTime) {
        // 环绕削峰，前后各取一半
        long half = durationTime / 2;
        // 计算时间距离热点时间的偏移
        long offset = eventTime.getTime() - System.currentTimeMillis();
        // 超出约束区间，直接打散结果为0
        if (Math.abs(offset) > half) {
            return 0;
        }
        // 进入约束区间，进行打散时间计算
        return timeDismiss(initialTime, half, Math.abs(offset));
    }

    /**
     * 初始弹窗50000ms，在2分钟的时间内追减递减直至消亡。
     *
     * @param initialTime  初始值弹窗时间 50000ms
     * @param durationTime 初始弹窗消亡持续总时间长度。示例中的2分钟
     * @param count        消亡次数
     * @return 消亡剩余次数
     */
    public static long timeDismiss(long initialTime, long durationTime, long count) {
        return initialTime - (initialTime * count / durationTime);
    }

}
