package com.spring.boxes.dollar;

import static com.fasterxml.jackson.core.JsonParser.Feature.ALLOW_SINGLE_QUOTES;
import static com.fasterxml.jackson.core.JsonParser.Feature.ALLOW_UNQUOTED_FIELD_NAMES;
import static com.fasterxml.jackson.databind.type.TypeFactory.defaultInstance;
import static com.fasterxml.jackson.module.kotlin.ExtensionsKt.jacksonObjectMapper;

import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.format.DateTimeFormatter;
import java.util.Collection;
import java.util.Map;
import java.util.Objects;
import java.util.TimeZone;

import javax.annotation.Nullable;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.module.kotlin.KotlinModule;
import com.jayway.jsonpath.*;
import com.jayway.jsonpath.spi.json.JacksonJsonProvider;
import com.jayway.jsonpath.spi.mapper.JacksonMappingProvider;
import org.apache.commons.lang3.StringUtils;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.datatype.guava.GuavaModule;
import com.fasterxml.jackson.datatype.jdk8.Jdk8Module;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.fasterxml.jackson.datatype.jsr310.deser.LocalDateDeserializer;
import com.fasterxml.jackson.datatype.jsr310.deser.LocalDateTimeDeserializer;
import com.fasterxml.jackson.datatype.jsr310.deser.LocalTimeDeserializer;
import com.fasterxml.jackson.datatype.jsr310.ser.LocalDateSerializer;
import com.fasterxml.jackson.datatype.jsr310.ser.LocalDateTimeSerializer;
import com.fasterxml.jackson.datatype.jsr310.ser.LocalTimeSerializer;
import com.fasterxml.jackson.module.paramnames.ParameterNamesModule;
import com.google.common.util.concurrent.UncheckedExecutionException;

public final class JSONUtils {

    // Object处理

    public static String toPrettyJSON(@Nullable Object obj) {
        if (Objects.isNull(obj)) {
            return null;
        }
        try {
            return OBJECT_MAPPER.writerWithDefaultPrettyPrinter().writeValueAsString(obj);
        } catch (JsonProcessingException e) {
            throw JSONException(e);
        }
    }

    public static String toJSON(@Nullable Object obj) {
        if (Objects.isNull(obj)) {
            return null;
        }
        try {
            return OBJECT_MAPPER.writeValueAsString(obj);
        } catch (JsonProcessingException e) {
            throw JSONException(e);
        }
    }
    public static <T> T fromJSON(@Nullable String json, Class<T> valueType) {
        if (StringUtils.isBlank(json)) {
            return null;
        }
        try {
            return OBJECT_MAPPER.readValue(json, valueType);
        } catch (IOException e) {
            throw JSONException(e);
        }
    }

    public static <T> T fromJSON(@Nullable String json, JavaType javaType) {
        if (StringUtils.isBlank(json)) {
            return null;
        }
        try {
            return OBJECT_MAPPER.readValue(json, javaType);
        } catch (IOException e) {
            throw JSONException(e);
        }
    }

    // Collection处理

    public static <E, T extends Collection> T fromJSON(String json,
                                                       Class<? extends Collection> collectionType, Class<E> valueType) {
        if (StringUtils.isEmpty(json)) {
            json = EMPTY_ARRAY_JSON;
        }
        try {
            return OBJECT_MAPPER.readValue(json,
                    defaultInstance().constructCollectionType(collectionType, valueType));
        } catch (IOException e) {
            throw JSONException(e);
        }
    }

    public static <E, T extends Collection> T fromJSON(byte[] bytes,
                                                       Class<? extends Collection> collectionType, Class<E> valueType) {
        try {
            return OBJECT_MAPPER.readValue(bytes, defaultInstance().constructCollectionType(collectionType, valueType));
        } catch (IOException e) {
            throw JSONException(e);
        }
    }

    public static <E, T extends Collection> T fromJSON(InputStream inputStream,
                                                       Class<? extends Collection> collectionType, Class<E> valueType) {
        try {
            return OBJECT_MAPPER.readValue(inputStream, defaultInstance().constructCollectionType(collectionType, valueType));
        } catch (IOException e) {
            throw JSONException(e);
        }
    }

    // Map处理

    public static Map<String, Object> fromJSON(String json) {
        return fromJSON(json, Map.class, String.class, Object.class);
    }

    public static Map<String, Object> fromJSON(InputStream is) {
        return fromJSON(is, Map.class, String.class, Object.class);
    }

    public static Map<String, Object> fromJSON(byte[] bytes) {
        return fromJSON(bytes, Map.class, String.class, Object.class);
    }

    public static <K, V, T extends Map<K, V>> T fromJSON(String json, Class<? extends Map> mapType,
                                                         Class<K> keyType, Class<V> valueType) {
        if (StringUtils.isEmpty(json)) {
            json = EMPTY_JSON;
        }
        try {
            return OBJECT_MAPPER.readValue(json,
                    defaultInstance().constructMapType(mapType, keyType, valueType));
        } catch (IOException e) {
            throw JSONException(e);
        }
    }

    public static <K, V, T extends Map<K, V>> T fromJSON(byte[] bytes, Class<? extends Map> mapType, Class<K> keyType, Class<V> valueType) {
        try {
            return OBJECT_MAPPER.readValue(bytes, defaultInstance().constructMapType(mapType, keyType, valueType));
        } catch (IOException e) {
            throw JSONException(e);
        }
    }

    public static <K, V, T extends Map<K, V>> T fromJSON(InputStream inputStream, Class<? extends Map> mapType, Class<K> keyType, Class<V> valueType) {
        try {
            return OBJECT_MAPPER.readValue(inputStream, defaultInstance().constructMapType(mapType, keyType, valueType));
        } catch (IOException e) {
            throw JSONException(e);
        }
    }

    public static <T> T extractJsonPathAsObject(Object argument, String jsonPath) {
        String json = toJSON(argument);
        return extractJsonPathAsObject(json, jsonPath);
    }

    public static <T> T extractJsonPathAsObject(String json, String jsonPath) {
        return JsonPath.read(json, jsonPath);
    }

    public static <T> T extractJsonPathAsObject(Object argument, String jsonPath, TypeRef<T> typeRef) {
        String json = toJSON(argument);
        return extractJsonPathAsObject(json, jsonPath, typeRef);
    }

    public static <T> T extractJsonPathAsObject(String json, String jsonPath, TypeRef<T> typeRef) {
        return PARSE_CONTEXT.parse(json).read(jsonPath, typeRef);
    }

    public static <T> T extractJsonPathAsObject(DocumentContext ctx, String jsonPath, TypeRef<T> typeRef) {
        return ctx.read(jsonPath, typeRef);
    }

    public static <T> T extractJsonPathAsObject(Object argument, String jsonPath, Class<T> type) {
        String json = toJSON(argument);
        return extractJsonPathAsObject(json, jsonPath, type);
    }

    public static <T> T extractJsonPathAsObject(String json, String jsonPath, Class<T> type) {
        return PARSE_CONTEXT.parse(json).read(jsonPath, type);
    }

    public static <T> T extractJsonPathAsObject(DocumentContext ctx, String jsonPath, Class<T> type) {
        return ctx.read(jsonPath, type);
    }

    private static RuntimeException JSONException(IOException e) {
        if (e instanceof JsonProcessingException) {
            return new UncheckedExecutionException(e);
        } else {
            return new UncheckedIOException(e);
        }
    }

    private static JavaTimeModule javaTimeModule() {
        JavaTimeModule javaTimeModule = new JavaTimeModule();
        javaTimeModule.addSerializer(LocalDateTime.class, new LocalDateTimeSerializer(DateTimeFormatter.ofPattern(DEFAULT_DATE_TIME_PATTERN)));
        javaTimeModule.addSerializer(LocalDate.class, new LocalDateSerializer(DateTimeFormatter.ofPattern(DEFAULT_DATE_PATTERN)));
        javaTimeModule.addSerializer(LocalTime.class, new LocalTimeSerializer(DateTimeFormatter.ofPattern(DEFAULT_TIME_PATTERN)));
        javaTimeModule.addDeserializer(LocalDateTime.class, new LocalDateTimeDeserializer(DateTimeFormatter.ofPattern(DEFAULT_DATE_TIME_PATTERN)));
        javaTimeModule.addDeserializer(LocalDate.class, new LocalDateDeserializer(DateTimeFormatter.ofPattern(DEFAULT_DATE_PATTERN)));
        javaTimeModule.addDeserializer(LocalTime.class, new LocalTimeDeserializer(DateTimeFormatter.ofPattern(DEFAULT_TIME_PATTERN)));
        return javaTimeModule;
    }

    public static final String EMPTY_JSON = "{}";
    public static final String EMPTY_ARRAY_JSON = "[]";
    public static final String DEFAULT_DATE_TIME_PATTERN = "yyyy-MM-dd HH:mm:ss";
    public static final String DEFAULT_DATE_PATTERN = "yyyy-MM-dd";
    public static final String DEFAULT_TIME_PATTERN = "HH:mm:ss";

    private static final ObjectMapper OBJECT_MAPPER;

    static {
        OBJECT_MAPPER = new ObjectMapper();
        // 设置时区
        OBJECT_MAPPER.setTimeZone(TimeZone.getTimeZone("GMT+8"));
        // 时间格式
        OBJECT_MAPPER.setDateFormat(new SimpleDateFormat(DEFAULT_DATE_TIME_PATTERN));
        // 许可单引号
        OBJECT_MAPPER.configure(ALLOW_SINGLE_QUOTES, true).configure(ALLOW_UNQUOTED_FIELD_NAMES, true).configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);;
        // 注册模块
        OBJECT_MAPPER.registerModule(new JavaTimeModule()).registerModules(new GuavaModule(), new ParameterNamesModule(), new KotlinModule(), new Jdk8Module());
    }

    public static ObjectMapper getInstance() {
        return OBJECT_MAPPER;
    }

    public static final ParseContext PARSE_CONTEXT = JsonPath.using(
            Configuration.builder()
                    .jsonProvider(new JacksonJsonProvider(jacksonObjectMapper()))
                    .mappingProvider(new JacksonMappingProvider(getInstance()))
                    .build()
                    .addOptions(Option.DEFAULT_PATH_LEAF_TO_NULL)
    );
}
