package com.spring.boxes.dollar;

import static java.util.stream.Collectors.toMap;

import java.util.Arrays;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.ResourceBundle;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.spring.boxes.dollar.support.Language;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.LocaleUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.web.servlet.LocaleResolver;
import org.springframework.web.servlet.support.RequestContextUtils;
import org.springframework.web.servlet.view.RedirectView;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

@Slf4j
public class LanguageUtils {

    private static final List<Locale> ALL_LOCALES = Arrays.stream(Language.values())
            .map(Language::getLocale)
            .filter(Objects::nonNull)
            .distinct()
            .collect(Collectors.toList());

    private static final Map<String, Locale> LOCALE_MAP = Stream.of(Language.values())
            .filter(Objects::nonNull)
            .collect(
                    toMap(v -> toMapKey(LocaleUtils.toLocale(v.name())), Language::getLocale)
            );


    private static String toMapKey(Locale locale) {
        if (StringUtils.isEmpty(locale.getCountry())) {
            return locale.getLanguage();
        } else {
            return locale.getLanguage() + "#" + locale.getCountry();
        }
    }

    /**
     * 获取多语言 Properties locale后缀信息标准化后缀
     * 例如:
     * locale = LanguageUtils.chooseLocale(locale);
     * ResourceBundle.getBundle("message", locale);
     */
    private static Locale chooseLocale(Locale locale) {
        if (Objects.isNull(locale)) {
            return Locale.CHINESE;
        }
        String language = locale.getLanguage();
        Locale result = LOCALE_MAP.get(toMapKey(locale));
        if (Objects.isNull(result)) {
            result = LOCALE_MAP.get(language);
        }
        return Objects.nonNull(result) ? result : Locale.ENGLISH;
    }


    public static List<Locale> getAllResourceLocales() {
        return ALL_LOCALES;
    }

    public static String getMessage(Locale locale, String key) {
        String message = getMessageMap(locale).get(key);
        if (StringUtils.isEmpty(message)) {
            message = getMessageMap(Locale.US).get(key);
        }
        if (StringUtils.isEmpty(message)) {
            message = getMessageMap(Locale.CHINA).get(key);
        }
        return message;
    }

    public static Map<String, String> getMessageMap(Locale locale) {
        String language = chooseLocale(locale).toString();
        return LazyHolder.languageResourceMap.get(language);
    }

    public static RedirectView settingLocale(String locale) {
        HttpServletRequest request = WebUtils.getRequest();
        HttpServletResponse response = WebUtils.getResponse();
        assert request != null;
        LocaleResolver localeResolver = RequestContextUtils.getLocaleResolver(request);
        assert localeResolver != null;
        if (Locale.US.getLanguage().equals(locale)) {
            localeResolver.setLocale(request, response, Locale.US);
        }
        if (Locale.CHINA.getLanguage().equals(locale)) {
            localeResolver.setLocale(request, response, Locale.CHINA);
        }
        if (StringUtils.isBlank(request.getHeader("Referer"))) {
            return WebUtils.redirect("/");
        }
        return WebUtils.redirect(request.getHeader("Referer"));
    }

    // 单例类加载message配置文件
    private static class LazyHolder {
        private static final Map<String, Map<String, String>> languageResourceMap = new HashMap<>();

        static {
            for (Language language : Language.values()) {
                languageResourceMap.put(language.name(), generateLangMap(language.getLocale()));
            }
        }

        private static Map<String, String> generateLangMap(Locale locale) {
            Map<String, String> map = new HashMap<>();
            ResourceBundle bundle = ResourceBundle.getBundle("message", locale);
            Enumeration<String> enu = bundle.getKeys();
            while (enu.hasMoreElements()) {
                String key = enu.nextElement();
                String value = bundle.getString(key);
                map.put(key, value);
            }
            return map;
        }
    }

}
