package com.spring.boxes.dollar;

import com.spring.boxes.dollar.support.privacy.jackson.PrivacySupplier;
import com.vdurmont.emoji.EmojiManager;
import org.apache.commons.lang3.StringUtils;

public class SensitiveUtils {

    /**
     * 用户昵称脱敏
     *
     * @param source 用户名
     * @return 脱敏结果
     */
    public static String hideNickName(String source) {
        return hideNickName(source, 3);
    }

    /**
     * 进行用户昵称脱敏
     *
     * @param source    用户名
     * @param maxMaxVal 最大脱敏串长度
     * @return 脱敏结果
     */
    public static String hideNickName(String source, int maxMaxVal) {
        source = StringUtils.trimToEmpty(source);
        if (StringUtils.isBlank(source)) {
            return "";
        }
        int length = source.length();
        // emoji表情占2个字符
        String first = "";
        if (length > 1) {
            first = source.substring(0, 2);
            if (!EmojiManager.containsEmoji(first)) {
                first = source.substring(0, 1);
            }
        }
        // emoji表情占2个字符
        String last = "";
        if (length - first.length() > 1) {
            last = source.substring(length - 2);
            if (!EmojiManager.containsEmoji(last)) {
                last = source.substring(length - 1);
            }
        }
        int remain = length - first.length() - last.length();
        return first + StringUtils.repeat(PrivacySupplier.DEFAULT_SYMBOL, $.limit(remain, 0, maxMaxVal)) + last;
    }

    /**
     * 【中文姓名】只显示第一个汉字，其他隐藏为2个星号，比如：李**
     *
     * @param fullName 文本
     * @return 打码结果
     */
    public static String hideChineseName(String fullName) {
        if (org.apache.commons.lang3.StringUtils.isBlank(fullName)) {
            return "";
        }
        String name = StringUtils.left(fullName, 1);
        return StringUtils.rightPad(name, StringUtils.length(fullName), PrivacySupplier.DEFAULT_SYMBOL);
    }

    /**
     * 【身份证号】显示最后四位，其他隐藏。共计18位或者15位，比如：*************1234
     *
     * @param id 文本
     * @return 打码结果
     */
    public static String hideIdCard(String id) {
        if (StringUtils.isBlank(id)) {
            return "";
        }
        String num = StringUtils.right(id, 4);
        return StringUtils.leftPad(num, StringUtils.length(id), PrivacySupplier.DEFAULT_SYMBOL);
    }


    /**
     * 【地址】只显示到地区，不显示详细地址，比如：上海徐汇区漕河泾开发区***
     *
     * @param address       地址
     * @param sensitiveSize 敏感信息长度
     * @return 打码结果
     */
    public static String hideAddress(String address, int sensitiveSize) {
        if (StringUtils.isBlank(address)) {
            return "";
        }
        int length = StringUtils.length(address);
        return StringUtils.rightPad(StringUtils.left(address, length - sensitiveSize), length, PrivacySupplier.DEFAULT_SYMBOL);
    }

    /**
     * 【电子邮箱】 邮箱前缀仅显示第一个字母，前缀其他隐藏，用星号代替，@及后面的地址显示，比如：d**@126.com
     *
     * @param email 文本
     * @return 打码结果
     */
    public static String hideEmail(String email) {
        if (StringUtils.isBlank(email)) {
            return "";
        }
        int index = StringUtils.indexOf(email, "@");
        if (index <= 1) {
            return email;
        }
        return StringUtils.rightPad(StringUtils.left(email, 1), index, PrivacySupplier.DEFAULT_SYMBOL).concat(StringUtils.mid(email, index, StringUtils.length(email)));
    }

    /**
     * 【银行卡号】前六位，后四位，其他用星号隐藏每位1个星号，比如：6222600**********1234
     *
     * @param cardNum 文本
     * @return 打码结果
     */
    public static String hideBankCard(String cardNum) {
        if (StringUtils.isBlank(cardNum)) {
            return "";
        }
        return org.apache.commons.lang3.StringUtils.left(cardNum, 6).concat(StringUtils.removeStart(StringUtils.leftPad(StringUtils.right(cardNum, 4), StringUtils.length(cardNum), PrivacySupplier.DEFAULT_SYMBOL), "******"));
    }

    /**
    public static void main(String[] args) {
        System.out.println(hideChineseName("杨先"));
        System.out.println(hideChineseName("杨先生"));
    }*/
}
