package com.spring.boxes.dollar;

import java.util.*;
import java.util.function.LongUnaryOperator;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;

import com.google.common.collect.*;
import com.google.common.collect.Range;

import com.google.common.collect.RangeSet;
import org.apache.commons.lang3.StringUtils;

import lombok.AccessLevel;
import lombok.NoArgsConstructor;

@NoArgsConstructor(access = AccessLevel.PRIVATE)
public class ShuntUtils {

    public static final String DEFAULT = ";;;";

    public static boolean isOnHalf(long id) {
        return isOnFor("100;0-49;;", id);
    }

    /**
     * http://www.ibloger.net/article/3310.html
     * <p>
     * 实例: "100;0-49;123,456;111,222"
     * 说明: 百分比灰度; 灰度50%用户; 白名单用户为123,456; 黑名单用户为111，222
     * <p>
     * 实例: "1000;0-999;123,456;111,222"
     * 说明: 千分比灰度; 灰度100%用户; 白名单用户为123,456; 黑名单用户为111，222
     *
     * @param config 配置比例
     * @param id     待计算命中值
     * @return 命中与否
     */
    public static boolean isOnFor(String config, long id) {
        return isOnFor(LongUnaryOperator.identity(), config, id);
    }

    public static boolean isOnFor(LongUnaryOperator preHashFunction, String config, long id) {
        if (StringUtils.isBlank(config)) {
            return false;
        }
        String[] segments = config.split(";");
        if (org.apache.commons.lang3.ArrayUtils.isEmpty(segments)) {
            return false;
        }
        if (segments.length < 1) {
            throw new IllegalArgumentException("格式格式无效! 正确格式: \"100;0-49;123,456;111,222\", 当前格式: " + config);
        }
        int mod = StringUtils.isNotBlank(segments[0]) ? Integer.parseInt(segments[0]) : 0;
        // parse hit mod range
        RangeSet<Long> tailNumbers = null;
        if (segments.length >= 2 && StringUtils.isNotBlank(segments[1])) {
            tailNumbers = TreeRangeSet.create();
            String[] ranges = segments[1].split(",");
            for (String range : ranges) {
                if (!range.contains("-")) {
                    long num = Long.parseLong(range);
                    // 这里用 canonical 自动合并相连的range 比如 1,2,3,4,5 会合并成 [1,5]
                    tailNumbers.add(Range.closed(num, num).canonical(DiscreteDomain.longs()));
                } else {
                    String[] nums = range.split("-");
                    long lower = Long.parseLong(nums[0]);
                    long upper = Long.parseLong(nums[1]);
                    if (lower >= upper) {
                        throw new IllegalArgumentException("无效区间: " + range);
                    }
                    tailNumbers.add(Range.closed(lower, upper).canonical(DiscreteDomain.longs()));
                }
            }
        }
        Set<Long> whiteList = parseList(segments, 2);
        Set<Long> blackList = parseList(segments, 3);
        return isOnFor(preHashFunction, mod, tailNumbers, whiteList, blackList, id);
    }

    private static boolean isOnFor(LongUnaryOperator hashFunc, long mod, RangeSet<Long> tailNumbers,
                                   Set<Long> whiteList, Set<Long> blackList, long id) {
        hashFunc = Optional.ofNullable(hashFunc).orElse(LongUnaryOperator.identity());
        whiteList = Optional.ofNullable(whiteList).orElse(Collections.emptySet());
        blackList = Optional.ofNullable(blackList).orElse(Collections.emptySet());
        if (blackList.isEmpty() && isAllTail(mod, tailNumbers)) {
            return true;
        }
        if (blackList.contains(id)) {
            return false;
        }
        long tailNumber = getTail(hashFunc, mod, id);
        if (mod > 0 && tailNumbers.contains(tailNumber)) {
            return true;
        }
        // 白名单使用原始id
        return whiteList.contains(id);
    }

    private static long getTail(LongUnaryOperator preHashFunction, long mod, long id) {
        if (mod > 0) {
            long hashed = preHashFunction.applyAsLong(id);
            return ((hashed % mod) + mod) % mod;
        }
        return 0;
    }

    private static boolean isAllTail(long mod, RangeSet<Long> tailNumbers) {
        return mod > 0 && tailNumbers.encloses(Range.closed(0L, mod - 1L));
    }

    private static Set<Long> parseList(String[] segments, int index) {
        if (index >= segments.length || StringUtils.isBlank(segments[index])) {
            return Collections.emptySet();
        }
        return Arrays.stream(segments[index].split(","))
                .map(x -> Long.parseLong(StringUtils.trim(x)))
                .collect(Collectors.toSet());
    }


    public static void main(String[] args) {
        /*String[] arr = {"100", "0-99", "123,456,789,100,", "111,222,333,444,"};
        System.out.println(parseList(arr, 2));
        System.out.println(parseList(arr, 3));*/

        String config = "100;0-99;123,456;111,222";
        for (int i = 0; i < 10; i++) {
            System.out.println(isOnFor(config, i));
        }
        System.out.println("测试黑白名单");
        System.out.println(isOnFor(config, 123));
        System.out.println(isOnFor(config, 456));
        System.out.println(isOnFor(config, 111));
        System.out.println(isOnFor(config, 222));

        /*RangeSet rangeSet = TreeRangeSet.create();
        rangeSet.add(Range.closed(1, 10));
        System.out.println(rangeSet);   // [[1..10]]

        rangeSet.add(Range.closedOpen(11, 15));
        System.out.println(rangeSet);   // [[1..10], [11..15)]

        rangeSet.add(Range.open(15, 20));
        System.out.println(rangeSet);   // [[1..10], [11..15), (15..20)]

        rangeSet.add(Range.openClosed(0, 0));
        System.out.println(rangeSet);   // [[1..10], [11..15), (15..20)]

        rangeSet.remove(Range.open(5, 10));
        System.out.println(rangeSet);   // [[1..5], [10..10], [11..15), (15..20)]*/
    }
}
