package com.spring.boxes.dollar;

import com.spring.boxes.dollar.term.TimeZone;
import org.apache.commons.lang3.ArrayUtils;
import org.joda.time.DateTime;
import org.joda.time.Period;
import org.joda.time.PeriodType;

import java.text.SimpleDateFormat;
import java.time.*;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.concurrent.TimeUnit;

import io.netty.util.internal.ThreadLocalRandom;

public class TimeUtils {

    public static final String[] ZODIAC_NAMES = {
            "鼠", "牛", "虎", "兔", "龙", "蛇", "马", "羊", "猴", "鸡", "狗", "猪"
    };

    public static final String[] ASTROLOGY_NAMES = {
            "白羊座", "金牛座", "双子座", "巨蟹座", "狮子座", "处女座", "天秤座", "天蝎座", "射手座", "摩羯座", "水瓶座", "双鱼座"
    };

    public static final String[] WEEK_DAYS = { "星期日", "星期一", "星期二", "星期三", "星期四", "星期五", "星期六" };

    public static Date now() {
        return new Date();
    }

    public static long toMilliseconds(Date date) {
        return date.getTime();
    }

    public static long toMilliseconds(LocalDate localDate) {
        return toMilliseconds(localDate.atStartOfDay());
    }

    public static long toMilliseconds(LocalDateTime localDateTime) {
        return localDateTime.atZone(ZoneId.systemDefault()).toInstant().toEpochMilli();
    }

    public static Date toDate(long timestamp) {
        return new Date(timestamp);
    }

    public static LocalDate toLocalDate(long timestamp) {
        return toLocalDateTime(timestamp).toLocalDate();
    }

    public static LocalDateTime toLocalDateTime(long timestamp) {
        return from(toDate(timestamp));
    }

    public static boolean isNowOn(TimeZone zoneTime, long now) {
        return zoneTime.getStartTime() < now && zoneTime.getStopTime() > now;
    }

    public static boolean isNowOn(TimeZone zoneTime) {
        return isNowOn(zoneTime, System.currentTimeMillis());
    }

    public static boolean isNowOn(LocalDateTime startTime, LocalDateTime stopTime, long now) {
        return isNowOn(toMilliseconds(startTime), toMilliseconds(stopTime), now);
    }

    public static boolean isNowOn(LocalDateTime startTime, LocalDateTime stopTime) {
        return isNowOn(toMilliseconds(startTime), toMilliseconds(stopTime), System.currentTimeMillis());
    }

    public static boolean isNowOn(long startTime, long stopTime) {
        return isNowOn(startTime, stopTime, System.currentTimeMillis());
    }

    public static boolean isNowOn(long startTime, long stopTime, long now) {
        return startTime < now && stopTime > now;
    }

    public static boolean anyOneIsNowOn(List<TimeZone> zoneTimes) {
        return anyOneIsNowOn(zoneTimes, System.currentTimeMillis());
    }

    public static boolean anyOneIsNowOn(List<TimeZone> zoneTimes, long now) {
        return zoneTimes.stream() //
                .anyMatch(zoneTime -> isNowOn(zoneTime, now));
    }

    public static long randomDate(TimeZone timeZone) {
        return ThreadLocalRandom.current().nextLong(timeZone.getStartTime(), timeZone.getStopTime());
    }

    public static long randomDate(int year1, int month1, int day1, int year2, int month2, int day2) {
        long startTime = TimeUtils.toMilliseconds(LocalDateTime.of(year1, month1, day1, 0, 0));
        long stopTime = TimeUtils.toMilliseconds(LocalDateTime.of(year2, month2, day2, 0, 0));
        return ThreadLocalRandom.current().nextLong(startTime, stopTime);
    }

    public static String getAstrologyName(LocalDateTime birthday) {
        return getAstrologyName(from(birthday));
    }

    public static String getAstrologyName(LocalDate birthday) {
        return getAstrologyName(from(birthday));
    }

    public static String getMomentTime(Date date) {
        return getMomentTime(toMilliseconds(date));
    }

    public static String getMomentTime(LocalDate localDate) {
        return getMomentTime(toMilliseconds(localDate));
    }

    public static String getMomentTime(LocalDateTime localDateTime) {
        return getMomentTime(toMilliseconds(localDateTime));
    }

    public static String getMomentTime(long timeMillis) {
        long intervalMillis = System.currentTimeMillis() - timeMillis;
        if (intervalMillis < TimeUnit.MINUTES.toMillis(1)) {
            return Math.max(1, intervalMillis / TimeUnit.SECONDS.toMillis(1)) + "秒前";
        } else if (intervalMillis < TimeUnit.HOURS.toMillis(1)) {
            return Math.max(1, intervalMillis / TimeUnit.MINUTES.toMillis(1)) + "分钟前";
        } else if (intervalMillis < TimeUnit.DAYS.toMillis(1)) {
            return Math.max(1, intervalMillis / TimeUnit.HOURS.toMillis(1)) + "小时前";
        } else if (intervalMillis < TimeUnit.DAYS.toMillis(2)) {
            return "昨天";
        } else if (intervalMillis < TimeUnit.DAYS.toMillis(30)) {
            return Math.max(1, intervalMillis / TimeUnit.DAYS.toMillis(1)) + "天前";
        } else if (intervalMillis < TimeUnit.DAYS.toMillis(365)) {
            return Math.max(1, intervalMillis / TimeUnit.DAYS.toMillis(30)) + "月前";
        } else {
            return Math.max(1, intervalMillis / TimeUnit.DAYS.toMillis(365)) + "年前";
        }
    }

    /**
     * 获取星座信息
     *
     * @param date 日期
     * @return 星座描述
     */
    public static String getAstrologyName(Date date) {
        int index = getAstrologyByDate(date);
        if (index < 0 || index > 11) {
            return "未知";
        } else {
            return ASTROLOGY_NAMES[index];
        }
    }

    /**
     * 根据日期返回所属星座的 index
     * 0  - 白羊座
     * 1  - 金牛座
     * 2  - 双子座
     * 3  - 巨蟹座
     * 4  - 狮子座
     * 5  - 处女座
     * 6  - 天秤座
     * 7  - 天蝎座
     * 8  - 射手座
     * 9  - 摩羯座
     * 10 - 水瓶座
     * 11 - 双鱼座
     * -1 - 不知道啥星座（不应当发生）
     *
     * @param date 日期
     * @return 星座描述
     */
    private static int getAstrologyByDate(Date date) {
        DateTime dateTime = new DateTime(date);
        int month = dateTime.getMonthOfYear();
        int day = dateTime.getDayOfMonth();
        if (month == 3 && day >= 21 || month == 4 && day <= 20) {
            return 0;
        } else if (month == 4 || month == 5 && day <= 21) {
            return 1;
        } else if (month == 5 || month == 6 && day <= 21) {
            return 2;
        } else if (month == 6 || month == 7 && day <= 22) {
            return 3;
        } else if (month == 7 || month == 8 && day <= 23) {
            return 4;
        } else if (month == 8 || month == 9 && day <= 23) {
            return 5;
        } else if (month == 9 || month == 10 && day <= 23) {
            return 6;
        } else if (month == 10 || month == 11 && day <= 22) {
            return 7;
        } else if (month == 11 || month == 12 && day <= 21) {
            return 8;
        } else if (month == 12 || month == 1 && day <= 20) {
            return 9;
        } else if (month == 1 || month == 2 && day <= 19) {
            return 10;
        } else if (month == 2 || month == 3) {
            return 11;
        } else {
            return -1;
        }
    }

    /**
     * 获取生肖
     *
     * @param year 年份
     * @return 生效描述
     */
    public static String getChineseZodiac(int year) {
        int index = (year - 2020) % 12;
        if (index >= 0) {
            return ZODIAC_NAMES[index];
        } else {
            return ZODIAC_NAMES[12 - (-index)];
        }
    }

    /**
     * 获取年龄
     *
     * @param birthday 生日
     * @return 年龄大小
     */
    public static int getAge(LocalDateTime birthday) {
        return getAge(from(birthday));
    }

    /**
     * 获取年龄
     *
     * @param birthday 生日
     * @return 年龄大小
     */
    public static int getAge(LocalDate birthday){
        return getAge(from(birthday.atStartOfDay()));
    }

    /**
     * 获取年龄
     *
     * @param birthday 生日
     * @return 年龄大小
     */
    public static int getAge(Date birthday) {
        Calendar now = Calendar.getInstance();
        Calendar born = Calendar.getInstance();
        now.setTime(new Date());
        born.setTime(birthday);
        if (born.after(now)) {
            return 0;
        }
        int age = now.get(Calendar.YEAR) - born.get(Calendar.YEAR);
        if (now.get(Calendar.DAY_OF_YEAR) < born.get(Calendar.DAY_OF_YEAR)) {
            age -= 1;
        }
        return age;
    }

    /**
     * 将 {@link LocalDateTime} 转换成 {@link Date}
     *
     * @param localDateTime {@link LocalDateTime} 待转换的日期
     * @return 转换成Date结果
     */
    public static Date from(LocalDateTime localDateTime) {
        Instant instant = localDateTime.atZone(ZoneId.systemDefault()).toInstant();
        return Date.from(instant);
    }

    /**
     * 将 {@link Date} 转换成 {@link LocalDateTime}
     *
     * @param date {@link Date} 待转换的日期
     * @return 转换成 {@link LocalDateTime} 结果
     */
    public static LocalDateTime from(Date date) {
        return LocalDateTime.ofInstant(date.toInstant(), ZoneId.systemDefault());
    }

    public static Date from(LocalDate date) {
        return from(date.atStartOfDay());
    }

    /**
     * 格式化日期
     *
     * @param date    待格式化的日期
     * @param pattern 格式化正则
     * @return 格式化结果串
     */
    public static String format(Date date, String pattern) {
        return new SimpleDateFormat(pattern).format(date);
    }

    public static String format(Date date) {
        return format(date, PATTERN_DATETIME);
    }

    /**
     * 格式化日期
     *
     * @param timestamp 待格式化的日期
     * @param pattern       格式化正式
     * @return 格式化结果串
     */
    public static String format(long timestamp, String pattern) {
        return format(new Date(timestamp), pattern);
    }

    public static String format(long timestamp) {
        return format(timestamp, PATTERN_DATETIME);
    }

    /**
     * 格式化日期
     *
     * @param localDateTime 待格式化的日期
     * @param pattern       格式化正式
     * @return 格式化结果串
     */
    public static String format(LocalDateTime localDateTime, String pattern) {
        return format(localDateTime, DateTimeFormatter.ofPattern(pattern));
    }

    public static String format(LocalDateTime localDateTime) {
        return format(localDateTime, PATTERN_DATETIME);
    }

    /**
     * 格式化日期
     *
     * @param localDateTime     待格式化的日期
     * @param dateTimeFormatter 格式化形式
     * @return 格式化结果串
     */
    public static String format(LocalDateTime localDateTime, DateTimeFormatter dateTimeFormatter) {
        return localDateTime.format(dateTimeFormatter);
    }

    /**
     * 格式化日期
     *
     * @param localDate 待格式化的日期
     * @param pattern   格式化正则, 这里使用的类型 {@link LocalDate}, 所以正则只能设定到天
     * @return 格式化结果串
     */
    public static String format(LocalDate localDate, String pattern) {
        return format(localDate, DateTimeFormatter.ofPattern(pattern));
    }

    /**
     * 格式化日期
     *
     * @param localDate         待格式化的日期
     * @param dateTimeFormatter 格式化正则, 这里使用的类型 {@link LocalDate}, 所以正则只能设定到天
     * @return 格式化结果串
     */
    public static String format(LocalDate localDate, DateTimeFormatter dateTimeFormatter) {
        return localDate.format(dateTimeFormatter);
    }

    /**
     * 当前时间格式化
     *
     * @param formatter 格式化形式
     * @return 格式化后的字符串
     */
    public static String nowFormat(DateTimeFormatter formatter) {
        return format(LocalDateTime.now(), formatter);
    }

    /**
     * 当前时间格式化
     *
     * @param pattern 格式化形式
     * @return 格式化后的字符串
     */
    public static String nowFormat(String pattern) {
        return nowFormat(DateTimeFormatter.ofPattern(pattern, Locale.getDefault()));
    }

    /**
     * 获取时间区间内的天数间隔
     * @param start 开始时间
     * @param end 结束时间
     * @return 间隔天数
     */
    public static int dayGap(Date start, Date end) {
        return new Period(start.getTime(), end.getTime(), PeriodType.days()).getDays();
    }

    /**
     * 获取时间区间内的天数间隔
     * @param start 开始时间
     * @param end 结束时间
     * @return 间隔天数
     */
    public static int dayGap(DateTime start, DateTime end) {
        return new Period(start, end, PeriodType.days()).getDays();
    }

    /**
     * 获取时间区间内的天数间隔
     * @param startTimestamp 开始时间
     * @param endTimestamp 结束时间
     * @return 间隔天数
     */
    public static int dayGap(long startTimestamp, long endTimestamp) {
        return (new Period(startTimestamp, endTimestamp, PeriodType.days())).getDays();
    }

    /**
     * 日期相加
     * @param days 天数
     * @return 相加后的日期
     */
    public static Date plusDays(int days) {
        return DateTime.now().plusDays(days).toDate();
    }

    /**
     * 获取本月最后一刻时间戳
     *
     * @return 时间戳
     */
    public static long thisMonthLastTimeMilli() {
        return toMonthLastTimeMilli(System.currentTimeMillis());
    }

    /**
     * 获取本周第一天开始一刻时间戳
     *
     * @return 时间戳
     */
    public static long thisWeekFirstTimeMilli() {
        return toWeekFirstTimeMilli(System.currentTimeMillis());
    }

    /**
     * 获取本周最后一刻时间戳
     *
     * @return 时间戳
     */
    public static long thisWeekLastTimeMilli() {
        return toWeekLastTimeMilli(System.currentTimeMillis());
    }

    /**
     * 获取本日开始一刻时间戳
     *
     * @return 时间戳
     */
    public static long thisDailyFirstTimeMilli() {
        return toDailyFirstTimeMilli(System.currentTimeMillis());
    }

    /**
     * 获取当日最后一刻时间戳
     *
     * @return 时间戳
     */
    public static long thisDailyLastTimeMilli() {
        return toDailyLastTimeMilli(System.currentTimeMillis());
    }

    /**
     * 获取本月第一天开始一刻时间戳
     *
     * @return 时间戳
     */
    public static long thisMonthFirstTimeMilli() {
        return toMonthFirstTimeMilli(System.currentTimeMillis());
    }

    /**
     * 获取当日开始一刻时间戳
     * @param timestamp 指定时间戳
     * @return 时间戳
     */
    public static long toDailyFirstTimeMilli(long timestamp) {
        return toDailyFirstTimeMilli(timestamp, ZoneId.systemDefault());
    }

    /**
     * 获取当日开始一刻时间戳
     * @param timestamp 指定时间戳
     * @param zoneId 时区
     * @return 时间戳
     */
    public static long toDailyFirstTimeMilli(long timestamp, ZoneId zoneId) {
        return LocalDate.now(Clock.fixed(Instant.ofEpochMilli(timestamp), zoneId))
                .atStartOfDay(zoneId)
                .toInstant()
                .toEpochMilli();
    }

    /**
     * 获取当日最后一秒时间戳
     * @param timestamp 指定时间戳
     * @return 时间戳
     */
    public static long toDailyLastTimeMilli(long timestamp) {
        return toDailyLastTimeMilli(timestamp, ZoneId.systemDefault());
    }

    /**
     * 获取当日最后一秒时间戳
     * @param timestamp 指定时间戳
     * @param zoneId 时区
     * @return 时间戳
     */
    public static long toDailyLastTimeMilli(long timestamp, ZoneId zoneId) {
        return LocalDate.now(Clock.fixed(Instant.ofEpochMilli(timestamp), zoneId))
                .atStartOfDay(zoneId)
                .plusDays(1)
                .toInstant()
                .toEpochMilli() - 1;
    }

    /**
     * 获取本周开始一刻时间戳
     * @param timestamp 指定时间戳
     * @return 时间戳
     */
    public static long toWeekFirstTimeMilli(long timestamp) {
        return toWeekFirstTimeMilli(timestamp, ZoneId.systemDefault());
    }

    /**
     * 获取本周开始一刻时间戳
     * @param timestamp 指定时间戳
     * @param zoneId 时区
     * @return 时间戳
     */
    public static long toWeekFirstTimeMilli(long timestamp, ZoneId zoneId) {
        return LocalDate.now(Clock.fixed(Instant.ofEpochMilli(timestamp), zoneId))
                .with(DayOfWeek.MONDAY)
                .atStartOfDay(zoneId)
                .toInstant()
                .toEpochMilli();
    }

    /**
     * 获取本周最后一秒时间戳
     * @param timestamp 指定时间戳
     * @return 时间戳
     */
    public static long toWeekLastTimeMilli(long timestamp) {
        return toWeekLastTimeMilli(timestamp, ZoneId.systemDefault());
    }

    /**
     * 获取本周最后一秒时间戳
     * @param timestamp 指定时间戳
     * @param zoneId 时区
     * @return 时间戳
     */
    public static long toWeekLastTimeMilli(long timestamp, ZoneId zoneId) {
        return LocalDate.now(Clock.fixed(Instant.ofEpochMilli(timestamp), zoneId))
                .with(DayOfWeek.MONDAY)
                .plusWeeks(1)
                .atStartOfDay(zoneId)
                .toInstant()
                .toEpochMilli() - 1;
    }

    /**
     * 获取本月第一天开始一刻时间戳
     * @param timestamp 指定时间戳
     * @return 时间戳
     */
    public static long toMonthFirstTimeMilli(long timestamp) {
        return toMonthFirstTimeMilli(timestamp, ZoneId.systemDefault());
    }

    /**
     * 获取本月第一天开始一刻时间戳
     * @param timestamp 指定时间戳
     * @param zoneId 时区
     * @return 时间戳
     */
    public static long toMonthFirstTimeMilli(long timestamp, ZoneId zoneId) {
        Clock clock = Clock.fixed(Instant.ofEpochMilli(timestamp), zoneId);
        Month nextMonth = MonthDay.now(clock).getMonth();
        return LocalDate.now(clock)
                .with(MonthDay.of(nextMonth, 1))
                .atStartOfDay(zoneId)
                .toInstant()
                .toEpochMilli();
    }

    /**
     * 获取当月开始一刻时间戳
     * @param timestamp 指定时间戳
     * @return 时间戳
     */
    public static long toMonthLastTimeMilli(long timestamp) {
        return toMonthLastTimeMilli(timestamp, ZoneId.systemDefault());
    }

    /**
     * 获取当月最后一秒时间戳
     * @param timestamp 指定时间戳
     * @param zoneId 时区
     * @return 时间戳
     */
    public static long toMonthLastTimeMilli(long timestamp, ZoneId zoneId) {
        Clock clock = Clock.fixed(Instant.ofEpochMilli(timestamp), zoneId);
        Month nextMonth = MonthDay.now(clock).getMonth();
        return LocalDate.now(clock)
                .with(MonthDay.of(nextMonth, 1))
                .plusMonths(1)
                .atStartOfDay(zoneId)
                .toInstant()
                .toEpochMilli() - 1;
    }

    public static String getStandardWeek(long timestamp) {
        return format(new Date(timestamp), PATTERN_WEEK);
    }

    public static String getSimplifyWeek(long timestamp) {
        String standWeek = getStandardWeek(timestamp);
        return "周" + standWeek.substring(2);
    }

    public static String getAmiableTime(){
        return getAmiableTime(System.currentTimeMillis());
    }

    public static String getAmiableTime(long timestamp){
        LocalDateTime localDateTime = toLocalDateTime(timestamp);
        int hour = localDateTime.getHour();
        if (hour <= 10) {
            return "早上";
        } else if (hour > 12 && hour <= 14) {
            return "中午";
        } else if (hour > 14 && hour <= 18) {
            return "下午";
        } else if (hour > 18) {
            return "晚上";
        }
        return "早上";
    }

    public static int getYmdInt(){
        return getYmdInt(System.currentTimeMillis());
    }

    public static int getYmdInt(long timestamp) {
        return Integer.parseInt(TimeUtils.format(timestamp, PATTERN_YMD_INT));
    }

    // 获取两个时间相差（毫秒）
    public static long getTimeDelta(long oldTime, long newTime) {
        return (newTime - oldTime);
    }

    // 获取两个时间相差（毫秒）
    public static long getTimeDelta(Date oldTime, Date newTime) {
        return getTimeDelta(oldTime.getTime(), newTime.getTime());
    }

    // 获取两个时间相差（毫秒）
    public static long getTimeDelta(LocalDateTime oldTime, LocalDateTime newTime) {
        return getTimeDelta(from(oldTime), from(newTime));
    }

    public static Duration getDuration(LocalDateTime d1, LocalDateTime d2) {
        return Duration.between(d1, d2);
    }

    public static LocalDate max(LocalDate... localDates) {
        if (ArrayUtils.isEmpty(localDates)) {
            return null;
        }
        List<LocalDate> localDateList = Arrays.asList(localDates);
        Collections.sort(localDateList, Comparator.reverseOrder());
        return localDateList.get(0);
    }

    public static LocalDate min(LocalDate... localDates) {
        if (ArrayUtils.isEmpty(localDates)) {
            return null;
        }
        List<LocalDate> localDateList = Arrays.asList(localDates);
        Collections.sort(localDateList, Comparator.naturalOrder());
        return localDateList.get(0);
    }


    public static final String PATTERN_DATETIME = "yyyy-MM-dd HH:mm:ss";
    public static final String PATTERN_DATE = "yyyy-MM-dd";
    public static final String PATTERN_TIME = "HH:mm:ss";
    public static final String PATTERN_YMD_INT = "yyyyMMdd";
    public static final String PATTERN_WEEK = "EEEE";
    public static final ZoneId CHINA_ZONE_ID = ZoneId.of("Asia/Shanghai");
}
