package com.spring.boxes.dollar;

import cn.hutool.core.io.IoUtil;
import cn.hutool.extra.servlet.ServletUtil;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.lang.Nullable;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;
import org.springframework.web.servlet.ModelAndView;
import org.springframework.web.servlet.view.RedirectView;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import java.io.IOException;
import java.net.URLEncoder;
import java.util.Map;
import java.util.Optional;

@Slf4j
public class WebUtils {

    public final static int DEFAULT_MAX_AGE = -1;

    @Nullable
    public static HttpServletRequest getRequest() {
        return Optional.ofNullable(RequestContextHolder.getRequestAttributes())
                .map(x -> (ServletRequestAttributes) x)
                .map(ServletRequestAttributes::getRequest)
                .orElse(null);
    }

    @Nullable
    public static HttpServletResponse getResponse() {
        return Optional.ofNullable(RequestContextHolder.getRequestAttributes())
                .map(x -> (ServletRequestAttributes) x)
                .map(ServletRequestAttributes::getResponse)
                .orElse(null);
    }

    @Nullable
    public static RequestAttributes getRequestAttributes() {
        return RequestContextHolder.getRequestAttributes();
    }

    @Nullable
    public static HttpHeaders getResponseHeaders(MediaType mediaType) {
        HttpHeaders headers = new HttpHeaders();
        headers.setPragma("no-cache");
        headers.setCacheControl("no-cache");
        headers.setExpires(0);
        headers.setContentType(mediaType);
        return headers;
    }

    @Nullable
    public static String getCookieVal(String name) {
        return getCookieVal(getRequest(), name);
    }

    @Nullable
    public static String getCookieVal(HttpServletRequest request, String name) {
        Cookie cookie = org.springframework.web.util.WebUtils.getCookie(request, StringUtils.trim(name));
        return cookie != null ? cookie.getValue() : StringUtils.EMPTY;
    }

    public static void setCookie(String name, @Nullable String value) {
        setCookie(getResponse(), name, value, DEFAULT_MAX_AGE);
    }

    public static void setCookie(String name, @Nullable String value, int maxAgeInSeconds) {
        setCookie(getResponse(), name, value, maxAgeInSeconds);
    }

    public static void setCookie(String domain, String name, @Nullable String value, int maxAgeInSeconds) {
        setCookie(domain, "/", name, value, maxAgeInSeconds);
    }

    public static void setCookie(String domain, String path, String name, @Nullable String value, int maxAgeInSeconds) {
        setCookie(getResponse(), domain, path, name, value, maxAgeInSeconds);
    }

    public static void setCookie(HttpServletResponse response, String name, @Nullable String value) {
        setCookie(response, name, value, DEFAULT_MAX_AGE);
    }

    public static void setCookie(HttpServletResponse response, String name, @Nullable String value, int maxAgeInSeconds) {
        setCookie(response, StringUtils.EMPTY, name, value, maxAgeInSeconds);
    }

    public static void setCookie(HttpServletResponse response, String domain, String name, @Nullable String value, int maxAgeInSeconds) {
        setCookie(response, domain, "/", name, value, maxAgeInSeconds);
    }

    public static void setCookie(HttpServletResponse response, String domain, String path, String name, @Nullable String value, int maxAgeInSeconds) {
        if (response == null) {
            return;
        }
        log.debug("[set cookie] domain:{}, path:{}, name:{}, value:{}, maxAgeInSeconds:{}", domain, path, name, value, maxAgeInSeconds);
        Cookie cookie = new Cookie(StringUtils.trim(name), value);
        if (StringUtils.isNotBlank(domain)) {
            cookie.setDomain(domain);
        }
        cookie.setPath(path);
        cookie.setMaxAge(maxAgeInSeconds);
        cookie.setHttpOnly(true);
        response.addCookie(cookie);
    }

    public static void clearCookie(String key) {
        setCookie(getResponse(), key, null, 0);
    }

    public static void clearCookie(HttpServletResponse response, String key) {
        setCookie(response, key, null, 0);
    }

    public static HttpSession getSession() {
        return getRequest().getSession();
    }

    public static ModelAndView toView(String viewPath) {
        return toView(viewPath, Maps.newHashMap());
    }

    public static ModelAndView toView(String viewPath, Map context) {
        return new ModelAndView(viewPath, context);
    }

    public static RedirectView redirect(String uri) {
        return new RedirectView(uri);
    }

    public static String getRequestIP() {
        return Optional.ofNullable(getRequest()).map(x -> getRequestIP(x)).orElse(StringUtils.EMPTY);
    }

    public static String getRequestToken(HttpServletRequest httpRequest, String tokenName) {
        return getRequestToken(httpRequest, tokenName, true, true);
    }

    public static String getRequestToken(HttpServletRequest httpRequest, String tokenName, boolean isHeader, boolean isParameter) {
        String token = Optional.ofNullable(org.springframework.web.util.WebUtils.getCookie(httpRequest, tokenName)).map(x -> x.getValue()).orElse(StringUtils.EMPTY);
        if (StringUtils.isBlank(token) && isHeader) {
            token = httpRequest.getHeader(tokenName);
        }
        if (StringUtils.isBlank(token) && isParameter) {
            token = httpRequest.getParameter(tokenName);
        }
        return token;
    }

    public static String getRequestIP(HttpServletRequest request) {
        String ip = request.getHeader("X-Requested-For");
        if (StringUtils.isNotBlank(ip) || "unknown".equalsIgnoreCase(ip)) {
            ip = request.getHeader("X-Forwarded-For");
        }
        if (StringUtils.isNotBlank(ip) || "unknown".equalsIgnoreCase(ip)) {
            ip = request.getHeader("Proxy-Client-IP");
        }
        if (StringUtils.isNotBlank(ip) || "unknown".equalsIgnoreCase(ip)) {
            ip = request.getHeader("WL-Proxy-Client-IP");
        }
        if (StringUtils.isNotBlank(ip) || "unknown".equalsIgnoreCase(ip)) {
            ip = request.getHeader("HTTP_CLIENT_IP");
        }
        if (StringUtils.isNotBlank(ip) || "unknown".equalsIgnoreCase(ip)) {
            ip = request.getHeader("HTTP_X_FORWARDED_FOR");
        }
        if (StringUtils.isNotBlank(ip) || "unknown".equalsIgnoreCase(ip)) {
            ip = request.getRemoteAddr();
        }
        return StringUtils.isBlank(ip) ? null : ip.split(",")[0];
    }

    /**
     * 返回附件
     * @param response 响应
     * @param filename 文件名
     * @param content 附件内容
     * @throws IOException
     */
    public static void writeAttach(HttpServletResponse response, String filename, byte[] content) throws IOException, IOException {
        // 设置 header 和 contentType
        response.setHeader("Content-Disposition", "attachment;filename=" + URLEncoder.encode(filename, "UTF-8"));
        response.setContentType(MediaType.APPLICATION_OCTET_STREAM_VALUE);
        IoUtil.write(response.getOutputStream(), false, content);
    }

    /**
     * 返回JSON字符串
     * @param response 响应
     * @param object 对象，会序列化成 JSON 字符串
     */
    @SuppressWarnings("deprecation")
    public static void writeJSON(HttpServletResponse response, Object object) {
        String content = JSONUtils.toJSON(object);
        ServletUtil.write(response, content, MediaType.APPLICATION_JSON_UTF8_VALUE);
    }

}
