package com.spring.boxes.dollar.support;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import com.googlecode.aviator.AviatorEvaluator;

import com.googlecode.aviator.Expression;
import com.spring.boxes.dollar.*;
import com.spring.boxes.dollar.enums.HashTypeEnum;
import com.spring.boxes.dollar.enums.ShuffleKey;
import com.spring.boxes.dollar.term.AbView;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;

import java.nio.charset.Charset;
import java.util.*;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;

@Slf4j
public class AbTool {

    /**
     * 执行场景命中
     *
     * @param input 场景入参
     * @param scene 场景模型
     * @return 场景命中结果
     */
    public static AbView.Output shuffle(AbView.Input input, AbView.SceneView scene) {
        if (null == input || $.isBlank(input.getSceneKey()) || MapUtils.isEmpty(input.getParam()) || !input.getParam().containsKey(input.getShuntKey())) {
            log.info("shuffle empty out! input: {}", input);
            return AbView.EMPTY_OUT;
        }

        if (null == scene || CollectionUtils.isEmpty(scene.getGroups())) {
            log.info("shuffle empty out! scene: {}", scene);
            return AbView.EMPTY_OUT;
        }

        if (!input.getSceneKey().equals(scene.getSceneKey())) {
            log.info("shuffle empty out! input: {}, scene:{}", input, scene);
            return AbView.EMPTY_OUT;
        }

        // do expression match
        compileExpressions(scene);

        // do shuffle with match
        List<AbView.SceneView.Item> items = shuffle(scene, input.getParam());

        return new AbView.Output(items);
    }


    private static List<AbView.SceneView.Item> shuffle(AbView.SceneView scene, Map<String, Object> matchParam) {
        List<AbView.SceneView.Item> items = Lists.newArrayList();
        for (AbView.SceneView.Group group : scene.getGroups()) {
            List<AbView.SceneView.Item> es = shuffle(group, matchParam);
            items.addAll(es);
        }
        return items;
    }


    private static List<AbView.SceneView.Item> shuffle(AbView.SceneView.Group g, Map<String, Object> matchParam) {
        if (null == g || MapUtils.isEmpty(matchParam) || $.isBlank(g.getShuntKey()) || null == matchParam.get(g.getShuntKey())
                || $.isBlank(String.valueOf(matchParam.get(g.getShuntKey()))) || CollectionUtils.isEmpty(g.getItems())) {
            return Lists.newArrayList();
        }

        // 分流值
        String shuffleVal = String.valueOf(matchParam.get(g.getShuntKey()));

        List<AbView.SceneView.Item> items = g.getItems().stream().
                sorted(Comparator.comparing(AbView.SceneView.Item::getItemWeight)).collect(Collectors.toList());

        List<AbView.SceneView.Item> list = new ArrayList<>();

        // 若配置了准入规则 且准入条件不满足 返回空
        if ($.isNotBlank(g.getExpressions()) && !filterExpression(g.getCompiledExpression(), matchParam)) {
            return list;
        }

        long sum = items.stream().mapToLong(AbView.SceneView.Item::getItemWeight).sum();
        AtomicLong now = new AtomicLong();
        // 分流命中
        for (AbView.SceneView.Item e : items) {
            HashTypeEnum hashEnum = HashTypeEnum.getHashType(g.getShuntAlgorithm());
            int hashVal = Math.abs(HashTypeEnum.getHashFunction(hashEnum).newHasher().putString(shuffleVal, Charset.defaultCharset()).hash().asInt());
            if (e.getItemWeight() > 0) {
                now.addAndGet(e.getItemWeight());
                String config = String.format("%s;%s-%s;%s;%s;", sum, now.get() - e.getItemWeight(), now.get() - 1, JoiningUtils.joinList(e.getWhiteList()), JoiningUtils.joinList(e.getBlackList()));
                if (ShuntUtils.isOnFor(config, hashVal)) {
                    list.add(e);
                }
            }
        }

        return MoreStream.distinct(list);
    }

    private static boolean filterExpression(Expression compiledExpressions, Map<String, Object> ft) {
        if (compiledExpressions != null) {
            //若没有命中规则 则返回true过滤该流量
            try {
                return (boolean) compiledExpressions.execute(ft);
            } catch (Exception e) {
                log.warn("执行aviator表达式失败,请检查表达式的配置和入参是否正确,原因:{} ", e.getCause(), e);
            }
        }
        return true;
    }

    /**
     * 判断分流白名单
     *
     * @param shuffleVal 分流值
     * @param item       对照组
     * @return 是否在对照组白名单
     */
    @Deprecated
    private static boolean isItemWhiteList(String shuffleVal, AbView.SceneView.Item item) {
        if (null == item || $.isBlank(shuffleVal) || CollectionUtils.isEmpty(item.getWhiteList())) {
            return false;
        }
        return item.getWhiteList().contains(shuffleVal);
    }

    /**
     * 执行渲染
     *
     * @param scene 对照组
     */
    private static void compileExpressions(AbView.SceneView scene) {
        if (null == scene) {
            return;
        }
        if (CollectionUtils.isEmpty(scene.getGroups())) {
            return;
        }
        for (AbView.SceneView.Group group : scene.getGroups()) {
            if (Objects.nonNull(group) && StringUtils.isNotBlank(group.getExpressions())) {
                group.setCompiledExpression(AviatorEvaluator.compile(group.getExpressions(), true));
            }
        }
    }

    public static List<AbView.SceneView.Group> sceneGroup() {
        List<AbView.SceneView.Group> groups = Lists.newArrayList();

        AbView.SceneView.Group group = new AbView.SceneView.Group();
        group.setSceneId(1);
        group.setGroupId(1001);
        group.setGroupName("实验分组1");
        group.setShuntKey(ShuffleKey.USER_ID.getKey());
        //group.setShuntAlgorithm(HashType.MD5.getLabel());
        group.setExpressions("(city_id==1||city_id==2||city_id==3)");
        group.setItems(sceneItem());

        groups.add(group);
        return groups;
    }

    public static List<AbView.SceneView.Item> sceneItem() {
        List<AbView.SceneView.Item> items = Lists.newArrayList();
        AbView.SceneView.Item item1 = new AbView.SceneView.Item();
        item1.setCarry(ImmutableMap.of("k1", String.valueOf(1), "k2", String.valueOf(2)));
        item1.setItemId(1);
        item1.setItemName("对照组1");
        item1.setGroupId(1001);
        item1.setItemWeight(30);
        item1.setSceneId(1);
        item1.setWhiteList(Lists.newArrayList("3"));

        AbView.SceneView.Item item2 = new AbView.SceneView.Item();
        item2.setCarry(ImmutableMap.of("k333", String.valueOf(3)));
        item2.setItemId(2);
        item2.setItemName("对照组2");
        item2.setItemWeight(70);
        item2.setGroupId(1001);
        item2.setSceneId(1);

        items.add(item1);
        items.add(item2);
        return items;
    }

}
