package com.spring.boxes.dollar.support;

import java.util.Arrays;

import com.spring.boxes.dollar.enums.TimePeriodEnum;
import org.joda.time.DateTime;

import lombok.extern.slf4j.Slf4j;

// 带过期逻辑的金币金额算法实现(交换法)
@Slf4j
public final class CurrencyAmount {

    // 计算过期周期
    public static int getPeriodMod() {
        return getPeriodMod(TimePeriodEnum.YEAR);
    }

    // 计算过期周期
    public static int getPeriodMod(TimePeriodEnum timeType) {
        return getPeriodMod(DateTime.now(), timeType);
    }

    // 计算过期周期
    public static int getPeriodMod(DateTime now, TimePeriodEnum timeType) {
        if (TimePeriodEnum.DAY == timeType) {
            return now.getDayOfWeek();
        }
        if (TimePeriodEnum.WEEK == timeType) {
            return now.getWeekOfWeekyear();
        }
        if (TimePeriodEnum.MONTH == timeType) {
            return now.getMonthOfYear();
        }
        // 默认为1年有效期
        return now.getYear();
    }

    // 当前余额总量：balance = balance_x + balance_y , x = (month - 1) % 3, y = month % 3
    public static long getCurrentAmount(long balance0, long balance1, long balance2, int modVal) {
        Long[] balance = new Long[] {balance0, balance1, balance2};
        return getCurrentAmount(balance, modVal);
    }

    public static long getCurrentAmount(Long[] balance, int cycleVal) {
        return balance[(cycleVal - 1) % 3] + balance[(cycleVal) % 3];
    }

    // 即将过期余额：balance_z , z = (month - 1) % 3
    public static long getExpiringAmount(long balance0, long balance1, long balance2, int modVal) {
        Long[] balance = new Long[] {balance0, balance1, balance2};
        return getExpiringAmount(balance, modVal);
    }

    public static long getExpiringAmount(Long[] balance, int modVal) {
        return balance[(modVal - 1) % 3];
    }

    // 清理过期余额：set balance_x = 0, x = (month - 2) % 3
    public static Long[] clearExpiredAmount(long balance0, long balance1, long balance2, int modVal) {
        Long[] balance = new Long[] {balance0, balance1, balance2};
        return clearExpiredAmount(balance, modVal);
    }

    public static Long[] clearExpiredAmount(Long[] balance, int modVal) {
        int index = (modVal - 2) % 3;
        if (index <= 0 || index > balance.length) {
            return balance;
        }
        balance[index] = 0L;
        return balance;
    }

    // 增加当前余额：set balance_x = balance_x + delta, x = month % 3
    public static Long[] incrCurrentAmount(long balance0, long balance1, long balance2, int modVal, long deltaAmount) {
        Long[] balance = new Long[] {balance0, balance1, balance2};
        return incrCurrentAmount(balance, modVal, deltaAmount);
    }

    public static Long[] incrCurrentAmount(Long[] balance, int cycleVal, long deltaAmount) {
        balance[(cycleVal) % 3] += deltaAmount;
        return balance;
    }

    // 消耗当前余额:
    //  a. start transaction
    //  b. select * where user_id = {} for update;
    // 	c. set balance_x = balance_x - min(delta, balance_x) and balance_y = balance_y - max(delta - balance_x, 0)
    //	   where balance_x + balance_y > delta, x = (month - 1) % 3, y = month % 3
    //  d. stop transaction
    public static Long[] decrCurrentAmount(long balance0, long balance1, long balance2, int modVal, long deltaAmount) {
        Long[] balance = new Long[] {balance0, balance1, balance2};
        return decrCurrentAmount(balance, modVal, deltaAmount);
    }

    public static Long[] decrCurrentAmount(Long[] balance, int modVal, long deltaAmount) {
        long currentTotal = getCurrentAmount(balance, modVal);
        if (deltaAmount > currentTotal) {
            throw new IllegalArgumentException("余额不足");
        }
        // x:即将过期
        int x = (modVal - 1) % 3, y = (modVal) % 3;
        // 拆分金额
        long decrX = Math.min(deltaAmount, balance[x]);
        long decrY = Math.max(deltaAmount - balance[x], 0);
        // 执行抵扣
        balance[x] -= decrX;
        balance[y] -= decrY;
        return balance;
    }

    /**
     * 当前总账: 200, 即将过期:0
     * 金额记录: [0, 200, 0]
     * 当前总账: 550, 即将过期:200
     * 金额记录: [0, 200, 350]
     * 当前总账: 420, 即将过期:350
     * 金额记录: [70, 0, 350]
     * 当前总账: 170, 即将过期:70
     * 金额记录: [70, 100, 0]
     */
    protected static void incrAmount(){
        Long[] balance = new Long[] {0L, 0L, 0L};
        incrCurrentAmount(balance, 1, 200);
        System.out.println("当前总账: " + getCurrentAmount(balance, 1) + ", 即将过期:" + getExpiringAmount(balance, 1));
        clearExpiredAmount(balance, 1);
        System.out.println("金额记录: " + Arrays.toString(balance));

        incrCurrentAmount(balance, 2, 350);
        System.out.println("当前总账: " + getCurrentAmount(balance, 2) + ", 即将过期:" + getExpiringAmount(balance, 2));
        clearExpiredAmount(balance, 2);
        System.out.println("金额记录: " + Arrays.toString(balance));

        incrCurrentAmount(balance, 3, 70);
        System.out.println("当前总账: " + getCurrentAmount(balance, 3) + ", 即将过期:" + getExpiringAmount(balance, 3));
        clearExpiredAmount(balance, 3);
        System.out.println("金额记录: " + Arrays.toString(balance));


        incrCurrentAmount(balance, 4, 100);
        System.out.println("当前总账: " + getCurrentAmount(balance, 4) + ", 即将过期:" + getExpiringAmount(balance, 4));
        clearExpiredAmount(balance, 4);
        System.out.println("金额记录: " + Arrays.toString(balance));
    }

    /**
     * 当前总账: 170, 即将过期:70
     * 当前总账: 120, 即将过期:20
     * 金额记录: [20, 100, 0]
     */
    // 抵扣金额小于过期金额，优先抵扣过期的
    protected static void decrAmount1(){
        Long[] balance = new Long[] {70L, 100L, 0L};
        System.out.println("当前总账: " + getCurrentAmount(balance, 4) + ", 即将过期:" + getExpiringAmount(balance, 4));
        decrCurrentAmount(balance, 4, 50);
        System.out.println("当前总账: " + getCurrentAmount(balance, 4) + ", 即将过期:" + getExpiringAmount(balance, 4));
        System.out.println("金额记录: " + Arrays.toString(balance));
    }


    /**
     * 当前总账: 170, 即将过期:70
     * 当前总账: 90, 即将过期:0
     * 金额记录: [0, 90, 0]
     */
    // 抵扣金额大于过期金额，拆分金额抵扣(过期全部抵扣)
    protected static void decrAmount2(){
        Long[] balance = new Long[] {70L, 100L, 0L};
        System.out.println("当前总账: " + getCurrentAmount(balance, 4) + ", 即将过期:" + getExpiringAmount(balance, 4));
        decrCurrentAmount(balance, 4, 10);
        System.out.println("当前总账: " + getCurrentAmount(balance, 4) + ", 即将过期:" + getExpiringAmount(balance, 4));
        System.out.println("金额记录: " + Arrays.toString(balance));
    }

}
