package com.spring.boxes.dollar.support;

import cn.dev33.satoken.exception.NotLoginException;
import cn.hutool.core.util.StrUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.core.NestedRuntimeException;
import org.springframework.http.HttpStatus;
import org.springframework.http.converter.HttpMessageNotReadableException;
import org.springframework.validation.BindException;
import org.springframework.validation.ObjectError;
import org.springframework.web.HttpMediaTypeNotSupportedException;
import org.springframework.web.HttpRequestMethodNotSupportedException;
import org.springframework.web.bind.MissingServletRequestParameterException;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import java.nio.file.AccessDeniedException;
import java.sql.SQLException;
import java.util.MissingFormatArgumentException;
import java.util.Objects;
import java.util.Optional;

import com.spring.boxes.dollar.support.ApiStatus.Status;
import org.springframework.web.servlet.NoHandlerFoundException;

import javax.validation.ConstraintViolation;
import javax.validation.ConstraintViolationException;
import javax.xml.bind.ValidationException;

// 扩展方式 继承该类 @ControllerAdvice;
@Slf4j
@ResponseBody
public class DefaultExceptionAdvice {

    @ResponseStatus(HttpStatus.OK)
    @ExceptionHandler({IllegalArgumentException.class})
    public ApiResult<?> handleException(IllegalArgumentException e) {
        return handler(String.format("参数错误: %s", e.getLocalizedMessage()), e);
    }

    @ExceptionHandler(NoHandlerFoundException.class)
    public ApiResult<?> noHandlerFoundExceptionHandler(NoHandlerFoundException ex) {
        log.warn("[noHandlerFoundExceptionHandler]", ex);
        return ApiResult.failure(String.format("请求地址不存在:%s", ex.getRequestURL()));
    }

    @ExceptionHandler(value = HttpMessageNotReadableException.class)
    public ApiResult<?> convertExceptionHandler(HttpMessageNotReadableException ex) {
        log.warn("[convertExceptionHandler]", ex);
        return ApiResult.failure(String.format("状态转码异常: %s", Optional.of(ex).map(NestedRuntimeException::getRootCause).map(Throwable::getMessage).orElse("")));
    }

    @ExceptionHandler(value = ConstraintViolationException.class)
    public ApiResult<?> constraintViolationExceptionHandler(ConstraintViolationException ex) {
        log.warn("[constraintViolationExceptionHandler]", ex);
        ConstraintViolation<?> constraintViolation = ex.getConstraintViolations().iterator().next();
        String message = "请求参数不正确";
        if(StrUtil.isNotEmpty(constraintViolation.getMessage())){
            message = constraintViolation.getMessage();
        }
        return ApiResult.failure(message);
    }

    @ExceptionHandler(value = ValidationException.class)
    public ApiResult<?> validationException(ValidationException ex) {
        log.warn("[constraintViolationExceptionHandler]", ex);
        // 无法拼接明细的错误信息，因为 Dubbo Consumer 抛出 ValidationException 异常时，是直接的字符串信息，且人类不可读
        return ApiResult.failure(String.format("参数校验不通过! %s", ex.getMessage()));
    }

    @ResponseStatus(HttpStatus.OK)
    @ExceptionHandler({HttpMediaTypeNotSupportedException.class})
    public ApiResult<?> handleException(HttpMediaTypeNotSupportedException e) {
        return handler("不支持当前媒体类型", e);
    }

    @ResponseStatus(HttpStatus.OK)
    @ExceptionHandler({HttpRequestMethodNotSupportedException.class})
    public ApiResult<?> handleException(HttpRequestMethodNotSupportedException e) {
        return handler("不支持当前请求方法", e);
    }

    @ResponseStatus(HttpStatus.FORBIDDEN)
    @ExceptionHandler({AccessDeniedException.class})
    public ApiResult<?> handleException(AccessDeniedException e) {
        return handler("没有权限", e);
    }

    @ResponseStatus(HttpStatus.OK)
    @ExceptionHandler({NotLoginException.class})
    public ApiResult<?> handleException(NotLoginException e) {
        String message = e.getLocalizedMessage();
        if(StringUtils.isBlank(message)){
            message = "会话失效,请重新登录";
        }
        int status = Status.UNAUTHORIZED.getStatus();
        return new ApiResult<>(status, message, null);
    }

    @ResponseStatus(HttpStatus.OK)
    @ExceptionHandler({SQLException.class})
    public ApiResult<?> handleException(SQLException e) {
        return handler("SQLException异常", e);
    }

    @ResponseStatus(HttpStatus.OK)
    @ExceptionHandler(ApiException.class)
    public ApiResult<?> handleException(ApiException e) {
        ApiStatus.Status status = e.getStatus();
        if (Objects.nonNull(status)) {
            return new ApiResult<>(status.getStatus(), e.getLocalizedMessage(), null);
        }
        return handler("业务异常", e);
    }

    @ResponseStatus(HttpStatus.OK)
    @ExceptionHandler(BindException.class)
    public ApiResult<?> paramExceptionHandler(BindException e) {
        ObjectError objectError = e.getBindingResult().getAllErrors().get(0);
        return ApiResult.failure(objectError.getDefaultMessage());
    }

    @ResponseStatus(HttpStatus.OK)
    @ExceptionHandler({MissingFormatArgumentException.class, MissingServletRequestParameterException.class})
    public ApiResult<?> handleException(MissingFormatArgumentException e) {
        return handler("参数缺失", e);
    }

    protected ApiResult<?> handler(String message, Exception e) {
        log.error(message, e);
        return ApiResult.failure(message);
    }
}
