package com.spring.boxes.dollar.support;

import static java.util.concurrent.TimeUnit.MILLISECONDS;

import java.util.function.Predicate;

import com.google.common.base.Predicates;
import com.google.common.util.concurrent.Uninterruptibles;

import lombok.experimental.UtilityClass;
import lombok.extern.slf4j.Slf4j;

@Slf4j
@UtilityClass
public class EasyRetry {

    public static <X extends Throwable> void runWithRetry(int maxRetryTimes, long retryPeriod, ThrowableRunnable<X> func) throws X {
        runWithRetry(maxRetryTimes, retryPeriod, func, Predicates.alwaysTrue());
    }

    public static <X extends Throwable> void runWithRetry(int maxRetryTimes, long retryPeriod, ThrowableRunnable<X> func, Predicate<Throwable> exceptionChecker) throws X {
        supplyWithRetry(maxRetryTimes, retryPeriod, () -> {
            func.run();
            return null;
        }, exceptionChecker);
    }

    public static <T, X extends Throwable> T supplyWithRetry(int maxRetryTimes, long retryPeriod, ThrowableSupplier<T, X> func) throws X {
        return supplyWithRetry(maxRetryTimes, retryPeriod, func, Predicates.alwaysTrue());
    }

    public static <T, X extends Throwable> T supplyWithRetry(int maxRetryTimes, long retryPeriod, ThrowableSupplier<T, X> func, Predicate<Throwable> exceptionChecker) throws X {
        int times = 0;
        Throwable lastThrowable;
        do {
            try {
                T result = func.get();
                log.debug("retry.success.{}", times);
                return result;
            } catch (Throwable e) {
                if (!exceptionChecker.test(e)) {
                    log.debug("retry.fail.{}", times);
                    throw e;
                } else {
                    if (e instanceof Error) {
                        log.error("", e);
                    }
                    if (retryPeriod > 0) {
                        Uninterruptibles.sleepUninterruptibly(retryPeriod, MILLISECONDS);
                    }
                    times++;
                    if (times <= maxRetryTimes) {
                        log.warn("try to retry for exception:{}, retry times:{}", e, times);
                    }
                    lastThrowable = e;
                }
            }
        } while (times <= maxRetryTimes);
        // noinspection unchecked
        throw (X) lastThrowable;
    }
}