/*--------------------------------------------------------------------------
 *  Copyright (c) 2010-2020, Elon.su All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the elon developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: Elon.su, you can also mail dennisit@163.com
 *--------------------------------------------------------------------------
 */
package com.spring.boxes.dollar.support;

import com.google.common.collect.Lists;
import org.apache.commons.collections4.ListUtils;
import org.apache.commons.lang3.StringUtils;

import java.util.*;
import java.util.stream.Collectors;

public abstract class EnumWrap<T extends EnumType> {

    public abstract List<T> valueList();

    public T valueGet(int value) {
        return valueGet(value, null);
    }

    public T valueGet(int value, T defVal) {
        List<T> enums = valueList();
        for (T em : enums) {
            if (em.getValue() == value) {
                return em;
            }
        }
        return defVal;
    }

    public T labelGet(String label) {
        return labelGet(label, null);
    }

    public T labelGet(String label, T defVal) {
        List<T> enums = valueList();
        for (T em : enums) {
            if (label.equalsIgnoreCase(em.getLabel())) {
                return em;
            }
        }
        return defVal;
    }

    public T nameGet(String name) {
        return nameGet(name, null);
    }

    public T nameGet(String name, T defVal) {
        List<T> enums = valueList();
        for (T em : enums) {
            if (StringUtils.trimToEmpty(name).toUpperCase().equals(em.getName())) {
                return em;
            }
        }
        return defVal;
    }

    public List<T> valueGet(List<Integer> values) {
        return ListUtils.emptyIfNull(values)
                .stream()
                .map(this::valueGet)
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
    }

    public List<T> labelGet(List<String> labels) {
        return ListUtils.emptyIfNull(labels)
                .stream()
                .filter(StringUtils::isNotBlank)
                .map(this::labelGet)
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
    }

    public List<T> nameGet(List<String> names) {
        return ListUtils.emptyIfNull(names)
                .stream()
                .filter(StringUtils::isNotBlank)
                .map(this::nameGet)
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
    }

    public List<Integer> toValues(List<T> enums) {
        return ListUtils.emptyIfNull(enums)
                .stream()
                .map(x -> x.getValue())
                .collect(Collectors.toList());
    }

    public List<Integer> allValues() {
        return ListUtils.emptyIfNull(valueList())
                .stream()
                .map(x -> x.getValue())
                .collect(Collectors.toList());
    }

    public List<String> toLabels(List<T> enums) {
        return ListUtils.emptyIfNull(enums)
                .stream()
                .map(EnumType::getLabel)
                .collect(Collectors.toList());
    }

    public List<String> toNames(List<T> enums) {
        return ListUtils.emptyIfNull(enums)
                .stream()
                .map(EnumType::getName)
                .collect(Collectors.toList());
    }

    public LinkedHashMap<Integer, String> toValueLabelPair() {
        LinkedHashMap<Integer, String> linkedHash = new LinkedHashMap<>();
        Optional.ofNullable(valueList()).orElse(Lists.newArrayList())
                .forEach(x -> {
                    linkedHash.put(x.getValue(), x.getLabel());
                });
        return linkedHash;
    }

    public LinkedHashMap<String, String> toNameLabelPair() {
        LinkedHashMap<String, String> linkedHash = new LinkedHashMap<>();
        Optional.ofNullable(valueList()).orElse(Lists.newArrayList())
                .forEach(x -> {
                    linkedHash.put(x.getName(), x.getLabel());
                });
        return linkedHash;
    }


}
