package com.spring.boxes.dollar.support;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.util.concurrent.Uninterruptibles.getUninterruptibly;
import static java.lang.String.format;
import static java.util.concurrent.TimeUnit.NANOSECONDS;

import java.time.Duration;
import java.util.*;
import java.util.concurrent.*;
import java.util.function.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

import javax.annotation.Nonnegative;
import javax.annotation.Nonnull;

import com.google.common.collect.Lists;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.ListUtils;

import com.google.common.collect.Iterators;
import com.google.common.util.concurrent.ExecutionError;
import com.google.common.util.concurrent.UncheckedExecutionException;
import com.google.common.util.concurrent.UncheckedTimeoutException;

import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.RandomUtils;
import org.springframework.lang.Nullable;
import org.springframework.util.ObjectUtils;

@Slf4j
public class MoreStream {

    public static <T> T getFirst(List<T> list) {
        if (CollectionUtils.isNotEmpty(list)) {
            return list.get(0);
        }
        return null;
    }

    public static <T> T getLast(List<T> list) {
        if (CollectionUtils.isNotEmpty(list)) {
            return list.get(list.size() - 1);
        }
        return null;
    }

    public static <E> E findAny(List<E> list){
        if(CollectionUtils.isEmpty(list)){
            return null;
        }
        return list.get(RandomUtils.nextInt(0, list.size()));
    }

    public static boolean containsAny(Object source, Object... targets) {
        return Arrays.asList(targets).contains(source);
    }

    public static boolean isAnyEmpty(Collection<?>... collections) {
        return Arrays.stream(collections).anyMatch(CollectionUtils::isEmpty);
    }

    public static <T> List<T> filterList(Collection<T> from, Predicate<T> predicate) {
        if (CollectionUtils.isEmpty(from)) {
            return new ArrayList<>();
        }
        return from.stream().filter(predicate).collect(Collectors.toList());
    }

    /**
     * 数组元素转SET集合
     * @param es 数组元素
     * @param <E> 泛型
     * @return Set结果
     */
    @SafeVarargs
    public static <E> Set<E> ofImmutableSet(E... es) {
        Objects.requireNonNull(es, "args es is null.");
        return Arrays.stream(es).collect(Collectors.toSet());
    }

    /**
     * 数组元素转List集合
     * @param es 数组元素
     * @param <E> 泛型
     * @return Set结果
     */
    @SafeVarargs
    public static <E> List<E> ofImmutableList(E... es) {
        Objects.requireNonNull(es, "args es is null.");
        return Arrays.stream(es).collect(Collectors.toList());
    }

    /**
     * 分组数据统计
     * @param list 原始集合
     * @param func 分组字段
     * @param <T> Key泛型
     * @param <K> 集合泛型
     * @return 分组结果
     */
    public static <T, K> Map<K, List<T>>  groupBy(Collection<T> list, Function<? super T, ? extends K> func) {
        return CollectionUtils.emptyIfNull(list).stream().collect(Collectors.groupingBy(func));
    }

    /**
     * 集合获取前边几个元素
     *
     * @param list 目标集合
     * @param size 目标数量
     * @param <T>  泛型支持
     * @return 集合TOP顶元素
     */
    public static <T> List<T> limit(List<T> list, int size) {
        return ListUtils.emptyIfNull(list).stream().limit(size).collect(Collectors.toList());
    }

    public static <T> List<T> distinct(List<T> list) {
        return ListUtils.emptyIfNull(list).stream().distinct().collect(Collectors.toList());
    }

    public static <T, R> List<T> distinct(Collection<T> from, Function<T, R> keyMapper) {
        if (CollectionUtils.isEmpty(from)) {
            return new ArrayList<>();
        }
        return distinct(from, keyMapper, (t1, t2) -> t1);
    }

    public static <T, R> List<T> distinct(Collection<T> from, Function<T, R> keyMapper, BinaryOperator<T> cover) {
        if (CollectionUtils.isEmpty(from)) {
            return new ArrayList<>();
        }
        return new ArrayList<>(convertMap(from, keyMapper, Function.identity(), cover).values());
    }

    public static <T, U> List<U> convertList(Collection<T> from, Function<T, U> func) {
        if (CollectionUtils.isEmpty(from)) {
            return new ArrayList<>();
        }
        return from.stream().map(func).filter(Objects::nonNull).collect(Collectors.toList());
    }

    public static <T, U> List<U> convertList(Collection<T> from, Function<T, U> func, Predicate<T> filter) {
        if (CollectionUtils.isEmpty(from)) {
            return new ArrayList<>();
        }
        return from.stream().filter(filter).map(func).filter(Objects::nonNull).collect(Collectors.toList());
    }

    public static <T, U> Set<U> convertSet(Collection<T> from, Function<T, U> func) {
        if (CollectionUtils.isEmpty(from)) {
            return new HashSet<>();
        }
        return from.stream().map(func).filter(Objects::nonNull).collect(Collectors.toSet());
    }

    public static <T, U> Set<U> convertSet(Collection<T> from, Function<T, U> func, Predicate<T> filter) {
        if (CollectionUtils.isEmpty(from)) {
            return new HashSet<>();
        }
        return from.stream().filter(filter).map(func).filter(Objects::nonNull).collect(Collectors.toSet());
    }

    public static <T, K> Map<K, T> convertMap(Collection<T> from, Function<T, K> keyFunc) {
        if (CollectionUtils.isEmpty(from)) {
            return new HashMap<>();
        }
        return convertMap(from, keyFunc, Function.identity());
    }

    public static <T, K> Map<K, T> convertMap(Collection<T> from, Function<T, K> keyFunc, Supplier<? extends Map<K, T>> supplier) {
        if (CollectionUtils.isEmpty(from)) {
            return supplier.get();
        }
        return convertMap(from, keyFunc, Function.identity(), supplier);
    }

    public static <T, K, V> Map<K, V> convertMap(Collection<T> from, Function<T, K> keyFunc, Function<T, V> valueFunc) {
        if (CollectionUtils.isEmpty(from)) {
            return new HashMap<>();
        }
        return convertMap(from, keyFunc, valueFunc, (v1, v2) -> v1);
    }

    public static <T, K, V> Map<K, V> convertMap(Collection<T> from, Function<T, K> keyFunc, Function<T, V> valueFunc, BinaryOperator<V> mergeFunction) {
        if (CollectionUtils.isEmpty(from)) {
            return new HashMap<>();
        }
        return convertMap(from, keyFunc, valueFunc, mergeFunction, HashMap::new);
    }

    public static <T, K, V> Map<K, V> convertMap(Collection<T> from, Function<T, K> keyFunc, Function<T, V> valueFunc, Supplier<? extends Map<K, V>> supplier) {
        if (CollectionUtils.isEmpty(from)) {
            return supplier.get();
        }
        return convertMap(from, keyFunc, valueFunc, (v1, v2) -> v1, supplier);
    }

    public static <T, K, V> Map<K, V> convertMap(Collection<T> from, Function<T, K> keyFunc, Function<T, V> valueFunc, BinaryOperator<V> mergeFunction, Supplier<? extends Map<K, V>> supplier) {
        if (CollectionUtils.isEmpty(from)) {
            return new HashMap<>();
        }
        return from.stream().collect(Collectors.toMap(keyFunc, valueFunc, mergeFunction, supplier));
    }

    /**
     * 打散集合
     *
     * @param list 原始数据集
     * @param <T>  泛型
     * @return 随机结果集
     */
    public static <T> List<T> randomList(Collection<T> list) {
        List<T> copy = new ArrayList<>(list);
        List<T> result = new ArrayList<>(list.size());
        for (int i = 0; i < list.size(); i++) {
            T t = copy.get(RandomUtils.nextInt(0, copy.size()));
            result.add(t);
        }
        return result;
    }

    /**
     * 带数量获取随机子集
     *
     * @param list 原始数据集
     * @param size 目标数量
     * @param <T>  泛型
     * @return 随机结果集
     */
    public static <T> List<T> randomList(List<T> list, int size) {
        List<T> tmp = Lists.newArrayList(list);
        if (size == list.size()) {
            return tmp;
        }

        List<T> result = Lists.newArrayList();
        for (int i = 0; i < size; i++) {
            int index = RandomUtils.nextInt(0, tmp.size());
            result.add(tmp.remove(index));
        }

        return result;
    }

    /**
     * 集合元素合并
     *
     * @param coll 多个目标集合
     * @param <T>  泛型支持
     * @return 水平抽到同一个集合中
     */
    @SafeVarargs
    public static <T> List<T> listMerge(Collection<T>... coll) {
        if(ArrayUtils.getLength(coll) == 0){
            return Lists.newArrayList();
        }
        List<T> target = Lists.newArrayList();
        Arrays.stream(coll).filter(CollectionUtils::isNotEmpty).forEach(target::addAll);
        return target;
    }

    /**
     * 进行map集合中相同元素的value值聚合
     *
     * @param target 目标Map
     * @param maps   往目标Map上追加的多个Map
     * @param <K>    泛型约束Key
     * @param <V>    泛型约束Map
     * @return 多个mapKey聚合后的结果, key相同的元素, value值进行合并
     */
    @SafeVarargs
    public static <K, V> Map<K, List<V>> keyMerged(Map<K, List<V>> target, Map<K, List<V>>... maps) {
        Map<K, List<V>> base = MapUtils.emptyIfNull(target);
        if (ArrayUtils.isEmpty(maps)) {
            return base;
        }

        // 待追加的多个map
        List<Map<K, List<V>>> adding = Arrays.stream(maps)
                .map(MapUtils::emptyIfNull)
                .collect(Collectors.toList());

        // 逐个Map元素追加
        adding.forEach(x -> {
            x.keySet().forEach(
                    key -> base.merge(key, x.get(key), (v1, v2)
                            -> Stream.of(v1, v2).flatMap(Collection::stream).collect(Collectors.toList())
                    )
            );
        });

        return base;
    }

    /**
     * Check whether the given Array contains the given element.
     *
     * @param array   the Array to check
     * @param element the element to look for
     * @param <T>     The generic tag
     * @return {@code true} if found, {@code false} else
     */
    public static <T> boolean arrayContains(@Nullable T[] array, final T element) {
        if (array == null) {
            return false;
        }
        return Arrays.stream(array).anyMatch(x -> ObjectUtils.nullSafeEquals(x, element));
    }

    /**
     * 从目标集合中抽取指定字段并去重
     *
     * @param coll   原始集合
     * @param mapper 字段抽取
     * @param <T>    原始集合元素类型
     * @param <R>    目标结果元素类型
     * @return 去重后的结果集合
     */
    public static <T, R> List<R> toList(Collection<T> coll, Function<? super T, ? extends R> mapper) {
        return CollectionUtils.emptyIfNull(coll)
                .stream()
                .map(mapper)
                .collect(Collectors.toList());
    }

    /**
     * 从目标集合中抽取指定字段并去重
     *
     * @param coll   原始集合
     * @param mapper 字段抽取
     * @param <T>    原始集合元素类型
     * @param <R>    目标结果元素类型
     * @return 去重后的结果集合
     */
    public static <T, R> List<R> toListWithDistinct(Collection<T> coll, Function<? super T, ? extends R> mapper) {
        return CollectionUtils.emptyIfNull(coll)
                .stream()
                .map(mapper)
                .distinct()
                .collect(Collectors.toList());
    }

    @SafeVarargs
    public static <K, V> Map<K, V> getMapValues(Map<K, V> map, K... key) {
        if (MapUtils.isEmpty(map) || ArrayUtils.getLength(key) == 0) {
            return map;
        }
        Map<K, V> target = new HashMap<>();
        Arrays.stream(key)
                .filter(Objects::nonNull)
                .forEach(e -> {
                    V value = map.get(e);
                    if (Objects.nonNull(value)) {
                        target.put(e, value);
                    }
                });
        return target;
    }

    public static <K, V> V getMapValue(Map<K, V> map, K key) {
        if (Objects.isNull(key)) {
            return null;
        }
        return MapUtils.emptyIfNull(map).get(key);
    }

    /**
     * 讲目标list集合转换为map
     *
     * @param coll        原始集合
     * @param keyMapper   key处理
     * @param valueMapper map处理
     * @param <T>         原始集合元素类型
     * @param <K>         目标集合Key类型
     * @param <V>         目标集合Value类型
     * @return 抽取后的Map集合集
     */
    public static <T, K, V> Map<K, V> toMap(Collection<T> coll, Function<? super T, ? extends K> keyMapper,
                                            Function<? super T, ? extends V> valueMapper) {
        return CollectionUtils.emptyIfNull(coll)
                .stream()
                .collect(Collectors.toMap(keyMapper, valueMapper, (o, n) -> o));
    }

    /**
     * 键值对转换为并发HashMap
     *
     * @param keysValues 键值对
     * @param <K>        key泛型
     * @param <V>        value泛型
     * @return 兼职堆
     */
    public static <K, V> ConcurrentHashMap<K, V> ofConcurrentHashMap(Object... keysValues) {
        if (keysValues.length % 2 != 0) {
            throw new IllegalArgumentException("wrong number of arguments for met, keysValues length can not be odd");
        }
        ConcurrentHashMap<K, V> keyValueMap = new ConcurrentHashMap<>();
        for (int i = keysValues.length - 2; i >= 0; i -= 2) {
            Object key = keysValues[i];
            Object value = keysValues[i + 1];
            keyValueMap.put((K) key, (V) value);
        }
        return keyValueMap;
    }


    /**
     * 将一个Stream按一定的大小进行批次分组，返回一个按组的新的Stream
     *
     * @param stream 输入值的Stream
     * @param size   分组的大小
     * @param <T>    输入值的类型泛型
     * @return 生成的{@link Stream}
     */
    public static <T> Stream<List<T>> partition(Stream<T> stream, int size) {
        Iterable<List<T>> iterable = () -> Iterators.partition(stream.iterator(), size);
        return StreamSupport.stream(iterable.spliterator(), false);
    }

    /**
     * 单体数据处理
     *
     * @param supplier 数据供给
     * @param mapper   数据转换
     * @param <T>      供给输出类型
     * @param <R>      结果数据类型
     * @return 结果集合
     */
    public static <T, R> R supplyOne(@NonNull Supplier<T> supplier, @NonNull Function<T, ? extends R> mapper) {
        T data = supplier.get();
        return Optional.ofNullable(data)
                .map(mapper)
                .orElse(null);
    }

    /**
     * 批量数据处理
     *
     * @param supplier 数据供给
     * @param mapper   数据转换
     * @param <T>      供给输出类型
     * @param <R>      结果数据类型
     * @return 结果集合
     */
    public static <T, R> List<R> supplyList(@NonNull Supplier<Collection<T>> supplier, @NonNull Function<T, ? extends R> mapper) {
        Collection<T> list = supplier.get();
        return CollectionUtils.emptyIfNull(list)
                .stream()
                .map(mapper)
                .collect(Collectors.toList());
    }

    /**
     * 返回Future结果
     *
     * @param future           要获取值的{@link Future}
     * @param <T>              返回结果类型
     * @param exceptionHandler 异常抛出
     * @return 目标返回结果
     */
    public static <T> T getFuture(CompletableFuture<T> future, Consumer<? super Throwable> exceptionHandler) {
        return getFuture(future, exceptionHandler, null);
    }

    /**
     * 返回Future结果
     *
     * @param future       要获取值的{@link Future}
     * @param <T>          返回结果类型
     * @param defaultValue 发生异常时返回的兜底值
     * @return 目标返回结果
     */
    public static <T> T getFuture(CompletableFuture<T> future, T defaultValue) {
        return getFuture(future, e -> log.error(e.getLocalizedMessage(), e), defaultValue);
    }


    /**
     * 返回Future结果
     *
     * @param future           要获取值的{@link Future}
     * @param <T>              返回结果类型
     * @param exceptionHandler 异常处理
     * @param defaultValue     发生异常时返回的兜底值
     * @return 目标返回结果
     */
    public static <T> T getFuture(CompletableFuture<T> future, Consumer<? super Throwable> exceptionHandler, T defaultValue) {
        return future.handle((result, ex) -> {
            if (ex != null) {
                exceptionHandler.accept(ex);
                return defaultValue;
            }
            return result;
        }).join();
    }

    public static <X extends Throwable> void checkWithThrow(boolean expression, @NonNull Supplier<X> exception) throws X {
        if (!expression) {
            throw exception.get();
        }
    }

    public static <X extends Throwable> void checkWithThrow(boolean expression, Function<String, X> exception,
                                                            String errorMessage, Object... errorMessageArgs) throws X {
        if (!expression) {
            throw exception.apply(format(errorMessage, errorMessageArgs));
        }
    }

    public static <T> T getFuture(@Nonnull Future<? extends T> future, @Nonnull Duration duration) {
        checkNotNull(duration);
        return getFuture(future, duration.toNanos(), NANOSECONDS);
    }

    public static <T> T getFuture(@Nonnull Future<? extends T> future, @Nonnegative long timeout, @Nonnull TimeUnit unit) {
        return getFuture(future, timeout, unit, false);
    }

    /**
     * 获取并返回一个{@link Future}的操作结果值，并将可能出现的异常以非检查异常的方式抛出
     *
     * @param future          要获取值的{@link Future}
     * @param <T>             返回结果类型
     * @param timeout         超时时间
     * @param unit            超时时间单位
     * @param cancelOnTimeout 执行超时后是否取消{@link Future}的执行
     * @return 目标返回结果
     * @throws UncheckedTimeoutException                  操作超时异常
     * @throws java.util.concurrent.CancellationException 任务取消异常
     * @throws ExecutionError                             执行时发生的{@link Error}异常
     * @throws UncheckedExecutionException                其他执行异常
     */
    public static <T> T getFuture(@Nonnull Future<? extends T> future, @Nonnegative long timeout, @Nonnull TimeUnit unit, boolean cancelOnTimeout) {
        checkArgument(timeout > 0);
        checkNotNull(future);
        try {
            return getUninterruptibly(future, timeout, unit);
        } catch (ExecutionException e) {
            Throwable cause = e.getCause();
            if (cause instanceof Error) {
                throw new ExecutionError((Error) cause);
            } else {
                throw new UncheckedExecutionException(cause);
            }
        } catch (TimeoutException e) {
            if (cancelOnTimeout) {
                future.cancel(false);
            }
            throw new UncheckedTimeoutException(e);
        }
    }


}
