package com.spring.boxes.dollar.support;

import com.google.common.collect.Lists;
import com.spring.boxes.dollar.JSONUtils;
import io.swagger.annotations.ApiModelProperty;
import lombok.Data;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

@Data
public class Pagination<T> implements Serializable {

    // 默认每页数据大小
    public static final Integer DEFAULT_PAGE_SIZE = 20;

    @ApiModelProperty("页码数")
    private long page;

    @ApiModelProperty("每页大小")
    private long size;

    @ApiModelProperty("总共页码数")
    private long pageTotal;

    @ApiModelProperty("记录总数")
    private long rowTotal;

    @ApiModelProperty("当前页数据集")
    private List<T> items = Lists.newArrayList();

    /**
     * 查询参数(用于透传)
     */
    private Map query;

    public static <T> Pagination<T> empty() {
        return new Pagination<>();
    }

    public Pagination() {
        this(new ArrayList<T>(), 0L, 1L);
    }

    public Pagination(List<T> list, long rowTotal, long page) {
        setPage(list, rowTotal, page, DEFAULT_PAGE_SIZE);
    }

    public Pagination(List<T> list, long rowTotal, long page, long size) {
        setPage(list, rowTotal, page, size);
    }

    private void setPage(List<T> list, long rowTotal, long page, long size) {
        page = Math.abs(page);
        size = Math.abs(size);
        long pageCount = (int) (rowTotal - 1) / size + 1;
        if (page >= pageCount) {
            page = pageCount;
        }
        this.setItems(list);
        this.setPageTotal(pageCount);
        this.setPage(page);
        this.setSize(size);
        this.setRowTotal(rowTotal);
    }

    public long getPageTotal() {
        if (this.getSize() > 0) {
            return ((this.getRowTotal() - 1) / this.getSize() + 1);
        }
        return 0;
    }

    public boolean isFirstPage() {
        return 1 == this.getPage();
    }

    public boolean isLastPage() {
        return this.getPage() == this.getPageTotal();
    }

    public boolean isNextPage() {
        return ((this.getPageTotal() > 1) && (this.getPage() < this.getPageTotal()));
    }

    public boolean isPrePage() {
        return ((this.getPageTotal() > 1) && (this.getPage() > 1));
    }

    @Override
    public String toString() {
        return JSONUtils.toJSON(this);
    }
}
