package com.spring.boxes.dollar.support;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.IService;
import com.google.common.collect.Lists;
import org.springframework.util.Assert;

import java.util.List;
import java.util.Optional;

public class PaginationPlus {

    /**
     * 分页模型转换
     *
     * @param page 页码
     * @param size 页值
     * @param <T>  结果模型
     * @return 分页对象
     */
    public static <T> Page<T> page(int page, int size) {
        return new Page<>(Math.abs(page), Math.abs(size));
    }

    /**
     * 将mybatisplus分页模型转换成vplus分页模型
     *
     * @param page mybatisplus分页模型
     * @param <T>  对象集合
     * @return 目标分页实体
     */
    public static <T> Pagination<T> builder(IPage<T> page) {
        if (null == page) {
            return Pagination.empty();
        }

        return new Pagination<>(page.getRecords(), page.getTotal(), (int) page.getCurrent(), (int) page.getSize());
    }


    /**
     * 构建分页查询对象
     *
     * @param service 服务接口
     * @param page    页码
     * @param size    页量
     * @param wrapper 查询条件
     * @param <T>     泛型约束
     * @return page对象
     */
    public static <T> Pagination<T> selectPagination(IService<T> service, int page, int size, QueryWrapper<T> wrapper) {
        Assert.notNull(service, "服务接口实现不能为空");
        Assert.notNull(wrapper, "查询条件不能为空");
        return builder(service.page(page(page, size), wrapper));
    }


    /**
     * 获取分页数据集合
     *
     * @param page 分页实体
     * @param <T>  对象元素
     * @return page页中的集合
     */
    public static <T> List<T> items(IPage<T> page) {
        return Optional.ofNullable(page).map(IPage::getRecords).orElse(Lists.newArrayList());
    }


    /**
     * 获取分页数据集合
     *
     * @param pagination 分页实体
     * @param <T>        对象元素
     * @return page页中的集合
     */
    public static <T> List<T> items(Pagination<T> pagination) {
        return Optional.ofNullable(pagination).map(Pagination::getItems).orElse(Lists.newArrayList());
    }


    /**
     * 抽取分页查询后的list集合
     *
     * @param service 业务接口
     * @param page    页码
     * @param size    页量
     * @param wrapper 查询条件
     * @param <T>     查询实体
     * @return page页中的集合
     */
    public static <T> List<T> items(IService<T> service, int page, int size, QueryWrapper<T> wrapper) {
        Assert.notNull(service, "服务接口实现不能为空");
        Assert.notNull(wrapper, "查询条件不能为空");

        return items(service.page(page(page, size), wrapper));
    }

}
