package com.spring.boxes.dollar.support;

import java.util.List;
import java.util.concurrent.TimeUnit;

import com.google.common.collect.Lists;
import com.spring.boxes.dollar.TimeUtils;
import com.spring.boxes.dollar.enums.TimePeriodEnum;
import com.spring.boxes.dollar.term.TimeZone;

import lombok.experimental.UtilityClass;
import lombok.extern.slf4j.Slf4j;

@Slf4j
@UtilityClass
public class PunchRange {

    /**
     * 获取可定制周期的签到Key
     *
     * @param appId 应用编号
     * @param userId 用户编号
     * @param timestamp 时间戳
     * @param period 周期
     * @return 签到Key
     */
    public static String getPeriodPunchKey(long appId, long userId, TimePeriodEnum period, long timestamp) {
        if (TimePeriodEnum.WEEK == period) {
            // 用户ID放在最后能减少KV热点问题
            long firstTimeMill = TimeUtils.toWeekFirstTimeMilli(timestamp);
            return getPatternPunchKey(appId, userId, firstTimeMill);
        }
        if (TimePeriodEnum.MONTH == period) {
            // 用户ID放在最后能减少KV热点问题
            long firstTimeMill = TimeUtils.toMonthFirstTimeMilli(timestamp);
            return getPatternPunchKey(appId, userId, firstTimeMill);
        }
        throw new IllegalArgumentException("不支持的活动周期");
    }
    
    /**
     * 构建签到缓存Key
     * @param appId 应用编号
     * @param userId 用户编号
     * @param timestamp 签到周期的第一天时间戳
     * @return 格式化的缓存Key
     */
    // 用户ID放在最后能减少KV热点问题
    public static String getPatternPunchKey(long appId, long userId, long timestamp){
        return String.format("_p:punch:m:%s:%s:%s", timestamp, appId, userId);
    }

    /**
     * 统计区间内每天的起始时间戳
     *
     * @param period 周期
     * @param timestamp 参照时间
     * @return 周期内每天的签到状态
     */
    public static List<Long> getDailyTimeMillis(TimePeriodEnum period, long timestamp){
        List<Long> dailyTimestamp = Lists.newArrayList();
        TimeZone timeZone = getTimeZone(period, timestamp);
        for(long i = timeZone.getStartTime(); i <= timeZone.getStopTime(); i += TimeUnit.DAYS.toMillis(1L)) {
            dailyTimestamp.add(TimeUtils.toDailyFirstTimeMilli(i));
        }
        return dailyTimestamp;
    }

    /**
     * 根据参照时间动态计算周期区间
     *
     * @param period 时间周期
     * @param timestamp 传入时间
     * @return 传入时间动态扩展的周期区间
     */
    public static TimeZone getTimeZone(TimePeriodEnum period, long timestamp) {
        if (TimePeriodEnum.WEEK == period) {
            long startTimestamp = TimeUtils.toWeekFirstTimeMilli(timestamp);
            long stopTimestamp = TimeUtils.toWeekLastTimeMilli(timestamp);
            return new TimeZone(startTimestamp, stopTimestamp);
        }
        if (TimePeriodEnum.MONTH == period) {
            long startTimestamp = TimeUtils.toMonthFirstTimeMilli(timestamp);
            long stopTimestamp = TimeUtils.toMonthLastTimeMilli(timestamp);
            return new TimeZone(startTimestamp, stopTimestamp);
        }
        throw new IllegalArgumentException("不支持的活动周期");
    }
}
