package com.spring.boxes.dollar.support;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.atomic.AtomicInteger;

import com.google.common.annotations.Beta;
import com.google.common.collect.Maps;
import com.google.common.collect.Range;
import com.google.common.collect.RangeMap;
import com.google.common.collect.TreeRangeMap;
import com.spring.boxes.dollar.term.RatePair;

// 概率抽奖辅助工具
public class RangRate {

    /**
     * 获取概率命中结果
     * @param itemRate 抽奖商品ID-概率
     * @return 命中的结果
     * @param <ID> 奖品ID
     */
    public static <ID> ID getRateValue(List<RatePair<ID>> itemRate) {
        Map<ID, Integer> itemRates = Maps.newHashMap();
        for(RatePair<ID> ratePair : itemRate){
            itemRates.put(ratePair.getValue(), ratePair.getRate());
        }
        return getRateValue(itemRates);
    }

    /**
     * 获取概率命中结果
     * @param itemRate 抽奖商品ID-概率
     * @return 命中的结果
     * @param <ID> 奖品ID
     */
    public static <ID> ID getRateValue(Map<ID, Integer> itemRate) {
        RangeMap<Integer, ID> rangeMap = toRangeMap(itemRate);
        return getRateValue(rangeMap);
    }

    /**
     * 获取概率命中结果
     * @param rateItem 概率区间-对应参与抽奖集合
     * @return 命中的结果
     * @param <ID> 奖品ID
     */
    public static <ID> ID getRateValue(RangeMap<Integer, ID> rateItem) {
        int winRange = rateItem.span().upperEndpoint();
        int random = ThreadLocalRandom.current().nextInt(winRange);
        return rateItem.get(random);
    }

    /**
     * 抽奖元素集合转换成概率区间
     * @param itemRate 抽奖元素集
     * @return 概率重叠区间
     * @param <ID> 奖品ID
     */
    @Beta
    public static <ID> RangeMap<Integer, ID> toRangeMap(Map<ID, Integer> itemRate) {
        RangeMap<Integer, ID> rangeMap = TreeRangeMap.create();
        AtomicInteger rateSum = new AtomicInteger(0);
        for (Entry<ID, Integer> entry : itemRate.entrySet()) {
            int rate = entry.getValue();
            rangeMap.put(Range.closedOpen(rateSum.get(), rateSum.addAndGet(rate)), entry.getKey());
        }
        return rangeMap;
    }

    public static void main(String[] args) {
        List<RatePair<String>> itemList = Arrays.asList(
                new RatePair<>(10, "001"),
                new RatePair<>(65, "002"),
                new RatePair<>(35, "003")
        );
        System.out.println(getRateValue(itemList));
    }
}
