package com.spring.boxes.dollar.support.locale;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Maps;
import lombok.Getter;

import java.util.EnumSet;
import java.util.Map;
import java.util.Set;

@Getter
public enum I18NCurrencyEnum {
    /**
     * 人民币币种
     */
    CNY("CNY", "¥", "人民币"),

    /**
     * 美元币种
     */
    USD("USD", "$", "美元"),

    /**
     * 港币币种
     */
    HKD("HKD", "HK$", "港币"),

    /**
     * 日元币种
     */
    JPY("JPY", "¥", "日币"),

    /**
     * 新币币种
     */
    SGD("SGD", "$", "新币"),

    /**
     * 欧元
     */
    EUR("EUR", "€", "欧元"),

    /**
     * 韩币
     */
    KRW("KRW", "KRW", "韩币"),
    ;

    private static final Map<String, I18NCurrencyEnum> CURRENCY_ENUM_MAP;
    private static final Set<I18NCurrencyEnum> MIN_UNIT_YUAN;

    static {
        Map<String, I18NCurrencyEnum> temp = Maps.newHashMapWithExpectedSize(128);
        for (I18NCurrencyEnum currencyEnum : EnumSet.allOf(I18NCurrencyEnum.class)) {
            temp.put(currencyEnum.currency, currencyEnum);
        }
        CURRENCY_ENUM_MAP = ImmutableMap.copyOf(temp);
        MIN_UNIT_YUAN = ImmutableSet.of(KRW, JPY);
    }

    /**
     * 币种
     */
    private final String currency;

    /**
     * 币种符号
     */
    private final String symbol;

    /**
     * 简述
     */
    private final String brief;

    I18NCurrencyEnum(String currency, String symbol, String brief) {
        this.currency = currency;
        this.symbol = symbol;
        this.brief = brief;
    }

    /**
     * 根据currency 获取  GlobalCurrencyEnum枚举值
     *
     * @param currency 币种参数
     * @return GlobalCurrencyEnum by currency mapping
     */
    public static I18NCurrencyEnum getCurrencyEnumBy(String currency) {
        return CURRENCY_ENUM_MAP.get(currency);
    }

    /**
     * 获取所有已存在的币种枚举
     *
     * @return key：币种 value：币种对应的枚举
     */
    public static Map<String, I18NCurrencyEnum> getCurrencyEnumMap() {
        return CURRENCY_ENUM_MAP;
    }

    /**
     * 获取最小结算币种是元的地区
     */
    public static Set<I18NCurrencyEnum> getMinUnitYuanCurrency() {
        return MIN_UNIT_YUAN;
    }

    /**
     * 判断是否是最小币种元
     *
     * @param currency 目前韩币和日币最小单位元，返回ture。null和其他返回false
     */
    public static boolean isMinUnitYuanCurrency(String currency) {
        if (currency == null) {
            return false;
        }

        return MIN_UNIT_YUAN.stream().anyMatch(p -> p.getCurrency().equals(currency.toUpperCase()));
    }
}

