package com.spring.boxes.dollar.support.locale;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Maps;

import java.util.EnumSet;
import java.util.Map;

public enum I18NLanguageEnum {

    /**
     * 英文
     */
    EN("en", "英文"),

    /**
     * 日文
     */
    JP("ja", "日文"),

    /**
     * 英文
     */
    ZH("zh", "中文"),

    /**
     * 中文
     */
    ZH_CN("zh-CN", "中文简体"),

    /**
     * 新加坡简体中文
     */
    ZH_SG("zh-SG", "简体中文(新加坡)"),

    /**
     * 马来西亚简体中文
     */
    ZH_MY("zh-MY", "简体中文(马来西亚)"),

    /**
     * 香港繁体中文
     */
    ZH_TW("zh-TW", "繁体中文"),

    /**
     * 台湾繁体中文
     */
    ZH_TW2("zh-TW2", "繁体中文(台湾)"),

    /**
     * 澳门繁体中文
     */
    ZH_MO("zh-MO", "繁体中文(澳门)"),

    /**
     * 新加坡英文
     */
    EN_SG("en-SG", "英文(新加坡)"),

    /**
     * 马来西亚英文
     */
    EN_MY("en-MY", "英文(马来西亚)"),

    /**
     * 日文
     */
    JA("ja", "日文"),
    /**
     * 意大利语
     */
    ITALIAN("ITALIAN", "意大利语"),

    ;

    private final static Map<String, I18NLanguageEnum> LANGUAGE_ENUM_MAP;

    static {
        Map<String, I18NLanguageEnum> temp = Maps.newHashMapWithExpectedSize(128);
        for (I18NLanguageEnum languageEnum : EnumSet.allOf(I18NLanguageEnum.class)) {
            temp.put(languageEnum.language, languageEnum);
        }
        LANGUAGE_ENUM_MAP = ImmutableMap.copyOf(temp);
    }

    private final String language;
    private final String brief;

    I18NLanguageEnum(String language, String brief) {

        this.language = language;
        this.brief = brief;
    }

    /**
     * 根据language获取GlobalLanguageEnum
     *
     * @param language 语言入参
     * @return GlobalLanguageEnum by language mapping
     */
    public static I18NLanguageEnum getByLanguage(String language) {
        return LANGUAGE_ENUM_MAP.get(language);
    }

    /**
     * 获取所有的语言枚举
     *
     * @return key：语言，如en value：对应语言的枚举
     */
    public static Map<String, I18NLanguageEnum> getLanguageEnumMap() {
        return LANGUAGE_ENUM_MAP;
    }

    /**
     * 获取语言
     *
     * @return 如：en
     */
    public String getLanguage() {
        return language;
    }

    /**
     * 获取描述字符串
     *
     * @return 如：英文
     */
    public String getBrief() {
        return brief;
    }
}
