package com.spring.boxes.dollar.support.logger;

import static org.slf4j.spi.LocationAwareLogger.DEBUG_INT;
import static org.slf4j.spi.LocationAwareLogger.ERROR_INT;
import static org.slf4j.spi.LocationAwareLogger.INFO_INT;
import static org.slf4j.spi.LocationAwareLogger.TRACE_INT;
import static org.slf4j.spi.LocationAwareLogger.WARN_INT;

import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import javax.annotation.Nullable;

import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Triple;
import org.slf4j.Logger;
import org.slf4j.Marker;
import org.slf4j.spi.LocationAwareLogger;

import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.LoadingCache;
import com.spring.boxes.dollar.support.ratelimit.SimpleRateLimiter;

public final class RateLogger implements Logger {

    public static final double DEFAULT_PERMITS_PER_SECOND = 1;
    private final Logger logger;
    private final LocationAwareLogger locationAwareLogger;

    // 私有构造
    private RateLogger(Logger logger, double permitsPerSecond, boolean perFormatString) {
        this.logger = logger;
        if (logger instanceof LocationAwareLogger) {
            this.locationAwareLogger = (LocationAwareLogger) logger;
        } else {
            this.locationAwareLogger = null;
        }
        this.rateLimiter = SimpleRateLimiter.create(permitsPerSecond);
        if (perFormatString) {
            this.perFormatStringRateLimiter = Caffeine.newBuilder().maximumSize(100).build(s -> SimpleRateLimiter.create(permitsPerSecond));
        } else {
            this.perFormatStringRateLimiter = null;
        }
    }

    /**
     * @param logger 要封装的logger实例
     */
    public static RateLogger rateLogger(Logger logger) {
        return rateLogger(logger, DEFAULT_PERMITS_PER_SECOND);
    }

    /**
     * @param logger 要封装的logger实例
     * @param permitsPer 打log的每秒允许个数，例如传入0.2，就意味着五秒打一条log
     */
    public static RateLogger rateLogger(Logger logger, double permitsPer) {
        return rateLogger(logger, permitsPer, false);
    }

    /**
     * @param logger 要封装的logger实例
     * @param permitsPer 打log的每秒允许个数，例如传入0.2，就意味着五秒打一条log
     * @param perFormatString 如果为 {@code true}，则按照每个 formatString 为单位而不是整个 logger 为单位执行采样
     */
    public static RateLogger rateLogger(Logger logger, double permitsPer, boolean perFormatString) {
        String name = logger.getName();
        Triple<String, Double, Boolean> key = Triple.of(name, permitsPer, perFormatString);
        RateLogger rateLogger = CACHE.get(key);
        if (Objects.nonNull(rateLogger)) {
            return rateLogger;
        }
        return CACHE.computeIfAbsent(key, it -> new RateLogger(logger, it.getMiddle(), it.getRight()));
    }

    @Override
    public String getName() {
        return logger.getName();
    }

    @Override
    public boolean isTraceEnabled() {
        return logger.isTraceEnabled();
    }

    @Override
    public void trace(String s) {
        if (isTraceEnabled()) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(null, FQCN, TRACE_INT, msg(s), args(limiter), null);
                } else {
                    logger.trace(msg(s), args(limiter));
                }
            }
        }
    }

    @Override
    public void trace(String s, Object o) {
        if (isTraceEnabled()) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(null, FQCN, TRACE_INT, msg(s), args(limiter, o), null);
                } else {
                    logger.trace(msg(s), args(limiter, o));
                }
            }
        }
    }

    @Override
    public void trace(String s, Object o, Object o1) {
        if (isTraceEnabled()) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(null, FQCN, TRACE_INT, msg(s), args(limiter, o, o1), null);
                } else {
                    logger.trace(msg(s), args(limiter, o, o1));
                }
            }
        }
    }

    @Override
    public void trace(String s, Object... objects) {
        if (isTraceEnabled()) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(null, FQCN, TRACE_INT, msg(s), args(limiter, objects), null);
                } else {
                    logger.trace(msg(s), args(limiter, objects));
                }
            }
        }
    }

    @Override
    public void trace(String s, Throwable throwable) {
        if (isTraceEnabled()) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(null, FQCN, TRACE_INT, msg(s), args(limiter), throwable);
                } else {
                    logger.trace(msg(s), args(limiter));
                }
            }
        }
    }

    @Override
    public boolean isTraceEnabled(Marker marker) {
        return logger.isTraceEnabled(marker);
    }

    @Override
    public void trace(Marker marker, String s) {
        if (isTraceEnabled(marker)) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(marker, FQCN, TRACE_INT, msg(s), args(limiter), null);
                } else {
                    logger.trace(marker, msg(s), args(limiter));
                }
            }
        }
    }

    @Override
    public void trace(Marker marker, String s, Object o) {
        if (isTraceEnabled(marker)) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(marker, FQCN, TRACE_INT, msg(s), args(limiter, o), null);
                } else {
                    logger.trace(marker, msg(s), args(limiter, o));
                }
            }
        }
    }

    @Override
    public void trace(Marker marker, String s, Object o, Object o1) {
        if (isTraceEnabled(marker)) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(marker, FQCN, TRACE_INT, msg(s), args(limiter, o, o1), null);
                } else {
                    logger.trace(marker, msg(s), args(limiter, o, o1));
                }
            }
        }
    }

    @Override
    public void trace(Marker marker, String s, Object... objects) {
        if (isTraceEnabled(marker)) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(marker, FQCN, TRACE_INT, msg(s), args(limiter, objects), null);
                } else {
                    logger.trace(marker, msg(s), args(limiter, objects));
                }
            }
        }
    }

    @Override
    public void trace(Marker marker, String s, Throwable throwable) {
        if (isTraceEnabled(marker)) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(marker, FQCN, TRACE_INT, msg(s), args(limiter), throwable);
                } else {
                    logger.trace(marker, msg(s), args(limiter));
                }
            }
        }
    }

    @Override
    public boolean isDebugEnabled() {
        return logger.isDebugEnabled();
    }

    @Override
    public void debug(String s) {
        if (isDebugEnabled()) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(null, FQCN, DEBUG_INT, msg(s), args(limiter), null);
                } else {
                    logger.debug(msg(s), args(limiter));
                }
            }
        }
    }

    @Override
    public void debug(String s, Object o) {
        if (isDebugEnabled()) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(null, FQCN, DEBUG_INT, msg(s), args(limiter, o), null);
                } else {
                    logger.debug(msg(s), args(limiter, o));
                }
            }
        }
    }

    @Override
    public void debug(String s, Object o, Object o1) {
        if (isDebugEnabled()) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(null, FQCN, DEBUG_INT, msg(s), args(limiter, o, o1), null);
                } else {
                    logger.debug(msg(s), args(limiter, o, o1));
                }
            }
        }
    }

    @Override
    public void debug(String s, Object... objects) {
        if (isDebugEnabled()) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(null, FQCN, DEBUG_INT, msg(s), args(limiter, objects), null);
                } else {
                    logger.debug(msg(s), args(limiter, objects));
                }
            }
        }
    }

    @Override
    public void debug(String s, Throwable throwable) {
        if (isDebugEnabled()) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(null, FQCN, DEBUG_INT, msg(s), args(limiter), throwable);
                } else {
                    logger.debug(msg(s), args(limiter, throwable));
                }
            }
        }
    }

    @Override
    public boolean isDebugEnabled(Marker marker) {
        return logger.isDebugEnabled(marker);
    }

    @Override
    public void debug(Marker marker, String s) {
        if (isDebugEnabled(marker)) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(marker, FQCN, DEBUG_INT, msg(s), args(limiter), null);
                } else {
                    logger.debug(marker, msg(s), args(limiter));
                }
            }
        }
    }

    @Override
    public void debug(Marker marker, String s, Object o) {
        if (isDebugEnabled(marker)) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(marker, FQCN, DEBUG_INT, msg(s), args(limiter, o), null);
                } else {
                    logger.debug(marker, msg(s), args(limiter, o));
                }
            }
        }
    }

    @Override
    public void debug(Marker marker, String s, Object o, Object o1) {
        if (isDebugEnabled(marker)) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(marker, FQCN, DEBUG_INT, msg(s), args(limiter, o, o1), null);
                } else {
                    logger.debug(marker, msg(s), args(limiter, o, o1));
                }
            }
        }
    }

    @Override
    public void debug(Marker marker, String s, Object... objects) {
        if (isDebugEnabled(marker)) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(marker, FQCN, DEBUG_INT, msg(s), args(limiter, objects), null);
                } else {
                    logger.debug(marker, msg(s), args(limiter, objects));
                }
            }
        }
    }

    @Override
    public void debug(Marker marker, String s, Throwable throwable) {
        if (isDebugEnabled(marker)) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(marker, FQCN, DEBUG_INT, msg(s), args(limiter), throwable);
                } else {
                    logger.debug(marker, msg(s), args(limiter));
                }
            }
        }
    }

    @Override
    public boolean isInfoEnabled() {
        return logger.isInfoEnabled();
    }

    @Override
    public void info(String s) {
        if (isInfoEnabled()) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(null, FQCN, INFO_INT, msg(s), args(limiter), null);
                } else {
                    logger.info(msg(s), args(limiter));
                }
            }
        }
    }

    @Override
    public void info(String s, Object o) {
        if (isInfoEnabled()) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(null, FQCN, INFO_INT, msg(s), args(limiter, o), null);
                } else {
                    logger.info(msg(s), args(limiter, o));
                }
            }
        }
    }

    @Override
    public void info(String s, Object o, Object o1) {
        if (isInfoEnabled()) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(null, FQCN, INFO_INT, msg(s), args(limiter, o, o1), null);
                } else {
                    logger.info(msg(s), args(limiter, o, o1));
                }
            }
        }
    }

    @Override
    public void info(String s, Object... objects) {
        if (isInfoEnabled()) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(null, FQCN, INFO_INT, msg(s), args(limiter, objects), null);
                } else {
                    logger.info(msg(s), args(limiter, objects));
                }
            }
        }
    }

    @Override
    public void info(String s, Throwable throwable) {
        if (isInfoEnabled()) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(null, FQCN, INFO_INT, msg(s), args(limiter), throwable);
                } else {
                    logger.info(msg(s), args(limiter));
                }
            }
        }
    }

    @Override
    public boolean isInfoEnabled(Marker marker) {
        return logger.isInfoEnabled(marker);
    }

    @Override
    public void info(Marker marker, String s) {
        if (isInfoEnabled(marker)) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(marker, FQCN, INFO_INT, msg(s), args(limiter), null);
                } else {
                    logger.info(marker, msg(s), args(limiter));
                }
            }
        }
    }

    @Override
    public void info(Marker marker, String s, Object o) {
        if (isInfoEnabled(marker)) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(marker, FQCN, INFO_INT, msg(s), args(limiter, o), null);
                } else {
                    logger.info(marker, msg(s), args(limiter, o));
                }
            }
        }
    }

    @Override
    public void info(Marker marker, String s, Object o, Object o1) {
        if (isInfoEnabled(marker)) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(marker, FQCN, INFO_INT, msg(s), args(limiter, o, o1), null);
                } else {
                    logger.info(marker, msg(s), args(limiter, o, o1));
                }
            }
        }
    }

    @Override
    public void info(Marker marker, String s, Object... objects) {
        if (isInfoEnabled(marker)) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(marker, FQCN, INFO_INT, msg(s), args(limiter, objects), null);
                } else {
                    logger.info(marker, msg(s), args(limiter, objects));
                }
            }
        }
    }

    @Override
    public void info(Marker marker, String s, Throwable throwable) {
        if (isInfoEnabled(marker)) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(marker, FQCN, INFO_INT, msg(s), args(limiter), throwable);
                } else {
                    logger.info(marker, msg(s), args(limiter));
                }
            }
        }
    }

    @Override
    public boolean isWarnEnabled() {
        return logger.isWarnEnabled();
    }

    @Override
    public void warn(String s) {
        if (isWarnEnabled()) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(null, FQCN, WARN_INT, msg(s), args(limiter), null);
                } else {
                    logger.warn(msg(s), args(limiter));
                }
            }
        }
    }

    @Override
    public void warn(String s, Object o) {
        if (isWarnEnabled()) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(null, FQCN, WARN_INT, msg(s), args(limiter, o), null);
                } else {
                    logger.warn(msg(s), args(limiter, o));
                }
            }
        }
    }

    @Override
    public void warn(String s, Object... objects) {
        if (isWarnEnabled()) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(null, FQCN, WARN_INT, msg(s), args(limiter, objects), null);
                } else {
                    logger.warn(msg(s), args(limiter, objects));
                }
            }
        }
    }

    @Override
    public void warn(String s, Object o, Object o1) {
        if (isWarnEnabled()) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(null, FQCN, WARN_INT, msg(s), args(limiter, o, o1), null);
                } else {
                    logger.warn(msg(s), args(limiter, o, o1));
                }
            }
        }
    }

    @Override
    public void warn(String s, Throwable throwable) {
        if (isWarnEnabled()) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(null, FQCN, WARN_INT, msg(s), args(limiter), throwable);
                } else {
                    logger.warn(msg(s), args(limiter));
                }
            }
        }
    }

    @Override
    public boolean isWarnEnabled(Marker marker) {
        return logger.isWarnEnabled(marker);
    }

    @Override
    public void warn(Marker marker, String s) {
        if (isWarnEnabled(marker)) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(marker, FQCN, WARN_INT, msg(s), args(limiter), null);
                } else {
                    logger.warn(marker, msg(s), args(limiter));
                }
            }
        }
    }

    @Override
    public void warn(Marker marker, String s, Object o) {
        if (isWarnEnabled(marker)) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(marker, FQCN, WARN_INT, msg(s), args(limiter, o), null);
                } else {
                    logger.warn(marker, msg(s), args(limiter, o));
                }
            }
        }
    }

    @Override
    public void warn(Marker marker, String s, Object o, Object o1) {
        if (isWarnEnabled(marker)) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(marker, FQCN, WARN_INT, msg(s), args(limiter, o, o1), null);
                } else {
                    logger.warn(marker, msg(s), args(limiter, o, o1));
                }
            }
        }
    }

    @Override
    public void warn(Marker marker, String s, Object... objects) {
        if (isWarnEnabled(marker)) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(marker, FQCN, WARN_INT, msg(s), args(limiter, objects), null);
                } else {
                    logger.warn(marker, msg(s), args(limiter, objects));
                }
            }
        }
    }

    @Override
    public void warn(Marker marker, String s, Throwable throwable) {
        if (isWarnEnabled(marker)) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(marker, FQCN, WARN_INT, msg(s), args(limiter), throwable);
                } else {
                    logger.warn(marker, msg(s), args(limiter));
                }
            }
        }
    }

    @Override
    public boolean isErrorEnabled() {
        return logger.isErrorEnabled();
    }

    @Override
    public void error(String s) {
        if (isErrorEnabled()) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(null, FQCN, ERROR_INT, msg(s), args(limiter), null);
                } else {
                    logger.error(msg(s), args(limiter));
                }
            }
        }
    }

    @Override
    public void error(String s, Object o) {
        if (isErrorEnabled()) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(null, FQCN, ERROR_INT, msg(s), args(limiter, o), null);
                } else {
                    logger.error(msg(s), args(limiter, o));
                }
            }
        }
    }

    @Override
    public void error(String s, Object o, Object o1) {
        if (isErrorEnabled()) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(null, FQCN, ERROR_INT, msg(s), args(limiter, o, o1), null);
                } else {
                    logger.error(msg(s), args(limiter, o, o1));
                }
            }
        }
    }

    @Override
    public void error(String s, Object... objects) {
        if (isErrorEnabled()) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(null, FQCN, ERROR_INT, msg(s), args(limiter, objects), null);
                } else {
                    logger.error(msg(s), args(limiter, objects));
                }
            }
        }
    }

    @Override
    public void error(String s, Throwable throwable) {
        if (isErrorEnabled()) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(null, FQCN, ERROR_INT, msg(s), args(limiter), throwable);
                } else {
                    logger.error(msg(s), args(limiter));
                }
            }
        }
    }

    @Override
    public boolean isErrorEnabled(Marker marker) {
        return logger.isErrorEnabled(marker);
    }

    @Override
    public void error(Marker marker, String s) {
        if (isErrorEnabled(marker)) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(marker, FQCN, ERROR_INT, msg(s), args(limiter), null);
                } else {
                    logger.error(marker, msg(s), args(limiter));
                }
            }
        }
    }

    @Override
    public void error(Marker marker, String s, Object o) {
        if (isErrorEnabled(marker)) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(marker, FQCN, ERROR_INT, msg(s), args(limiter, o), null);
                } else {
                    logger.error(marker, msg(s), args(limiter, o));
                }
            }
        }
    }

    @Override
    public void error(Marker marker, String s, Object o, Object o1) {
        if (isErrorEnabled(marker)) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(marker, FQCN, ERROR_INT, msg(s), args(limiter, o, o1), null);
                } else {
                    logger.error(marker, msg(s), args(limiter, o, o1));
                }
            }
        }
    }

    @Override
    public void error(Marker marker, String s, Object... objects) {
        if (isErrorEnabled(marker)) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(marker, FQCN, ERROR_INT, msg(s), args(limiter, objects), null);
                } else {
                    logger.error(marker, msg(s), args(limiter, objects));
                }
            }
        }
    }

    @Override
    public void error(Marker marker, String s, Throwable throwable) {
        if (isErrorEnabled(marker)) {
            SimpleRateLimiter limiter = getLimiter(s);
            if (limiter.tryAcquire()) {
                if (locationAwareLogger != null) {
                    locationAwareLogger.log(marker, FQCN, ERROR_INT, msg(s), args(limiter), throwable);
                } else {
                    logger.error(marker, msg(s), args(limiter));
                }
            }
        }
    }

    private String msg(String message) {
        return "[IGNORED={}]" + message;
    }

    private Object[] args(SimpleRateLimiter limiter, Object... args) {
        Object[] result;
        long skip = limiter.getSkipCountAndClear();
        if (ArrayUtils.getLength(args) == 0) {
            result = new Object[] {skip};
        } else {
            result = ArrayUtils.addAll(new Object[] {skip}, args);
        }
        return result;
    }

    private SimpleRateLimiter getLimiter(@Nullable String message) {
        if(Objects.isNull(perFormatStringRateLimiter)){
            return rateLimiter;
        }
        if(StringUtils.isBlank(message)){
            return rateLimiter;
        }
        return perFormatStringRateLimiter.get(message);
    }

    private final SimpleRateLimiter rateLimiter;
    private final LoadingCache<String, SimpleRateLimiter> perFormatStringRateLimiter;
    private static final String FQCN = RateLogger.class.getName();
    // 直接使用Map来cache RateLogger。Logger的数量是有限的，LogBack也是使用了Map来Cache，所以没必要用一个支持evict的Cache。
    private static final ConcurrentMap<Triple<String, Double, Boolean>, RateLogger> CACHE = new ConcurrentHashMap<>();

}
