package com.spring.boxes.dollar.support.privacy.jackson;

import cn.hutool.core.util.DesensitizedUtil;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.databind.BeanProperty;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.JsonSerializer;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.databind.ser.ContextualSerializer;
import com.spring.boxes.dollar.StringUtils;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.NoArgsConstructor;

import java.io.IOException;
import java.util.Objects;

@Data
@NoArgsConstructor
@AllArgsConstructor
@EqualsAndHashCode(callSuper = true)
public class PrivacyFieldSerializer extends JsonSerializer<String> implements ContextualSerializer {

    private PrivacyType privacyType;

    @Override
    public void serialize(final String origin, final JsonGenerator jsonGenerator, final SerializerProvider serializerProvider) throws IOException {
        if (StringUtils.isNotBlank(origin) && Objects.nonNull(privacyType)) {
            switch (privacyType) {
                case DEFAULT:
                    jsonGenerator.writeString(new PrivacySupplier.DefaultPrivacySupplier().accept(origin));
                    break;
                case ID_CARD:
                    jsonGenerator.writeString(DesensitizedUtil.idCardNum(origin, 1, 1));
                    break;
                case EMAIL:
                    jsonGenerator.writeString(DesensitizedUtil.email(origin));
                    break;
                case PHONE:
                    jsonGenerator.writeString(DesensitizedUtil.mobilePhone(origin));
                    break;
                case FIXED_PHONE:
                    jsonGenerator.writeString(DesensitizedUtil.fixedPhone(origin));
                    break;
                case NAME:
                    jsonGenerator.writeString(DesensitizedUtil.chineseName(origin));
                    break;
                case ADDRESS:
                    jsonGenerator.writeString(DesensitizedUtil.address(origin, 5));
                    break;
                case BANK_CARD:
                    jsonGenerator.writeString(DesensitizedUtil.bankCard(origin));
                    break;
                case PASSPORT:
                    jsonGenerator.writeString(new PrivacySupplier.DefaultPrivacySupplier().accept(origin));
                    break;
                case CAR_LICENSE:
                    jsonGenerator.writeString(DesensitizedUtil.carLicense(origin));
                    break;
                default:
                    throw new IllegalArgumentException("Unknown privacy type enum " + privacyType);
            }
        }
    }

    @Override
    public JsonSerializer<?> createContextual(SerializerProvider serializerProvider, BeanProperty beanProperty) throws JsonMappingException {
        if (Objects.nonNull(beanProperty)) {
            if (Objects.equals(beanProperty.getType().getRawClass(), String.class)) {
                PrivacyField privacyEncrypt = beanProperty.getAnnotation(PrivacyField.class);
                if (Objects.isNull(privacyEncrypt)) {
                    privacyEncrypt = beanProperty.getContextAnnotation(PrivacyField.class);
                }
                if (Objects.nonNull(privacyEncrypt)) {
                    return new PrivacyFieldSerializer(privacyEncrypt.type());
                }
            }
            return serializerProvider.findValueSerializer(beanProperty.getType(), beanProperty);
        }
        return serializerProvider.findNullValueSerializer(null);
    }
}
