package com.spring.boxes.dollar.support.reflect;

import java.lang.reflect.Array;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;

import com.google.common.collect.Lists;

/**
 * 类型泛化工具
 */
@Slf4j
public class TypeReflect {

    private static final Map<String, Class<?>> TYPES = new HashMap<>();
    private static final String ARRAY_PREFIX = "[";
    private static final String REFERENCE_TYPE_PREFIX = "L";
    private static final String REFERENCE_TYPE_SUFFIX = ";";

    public static List<Class<?>> forNames(List<String> names) {
        return forNames(Thread.currentThread().getContextClassLoader(), names);
    }

    public static List<Class<?>> forNames(ClassLoader loader, List<String> names) {
        if (CollectionUtils.isEmpty(names)) {
            return Lists.newArrayList();
        }
        return names.stream().map(e -> {
            try {
                return forName(loader, e);
            } catch (Exception ex) {
                log.error(ex.getLocalizedMessage(), ex);
            }
            return null;
        }).collect(Collectors.toList());
    }

    public static Class<?> forName(String name) {
        return forName(Thread.currentThread().getContextClassLoader(), name);
    }

    public static Class<?> forName(ClassLoader loader, String name) {
        if (TYPES.containsKey(name)) {
            return TYPES.get(name);
        }
        if (isArray(name)) {
            int dimension = 0;
            while (isArray(name)) {
                ++dimension;
                name = name.substring(1);
            }
            Class<?> type = forName(loader, name);
            int[] dimensions = new int[dimension];
            for (int i = 0; i < dimension; i++) {
                dimensions[i] = 0;
            }
            return Array.newInstance(type, dimensions).getClass();
        }
        if (isReferenceType(name)) {
            name = name.substring(1, name.length() - 1);
        }
        try {
            return Class.forName(name, false, loader);
        } catch (Exception ex) {
            log.error(ex.getLocalizedMessage(), ex);
        }
        return null;
    }

    private static boolean isArray(String type) {
        return type != null && type.startsWith(ARRAY_PREFIX);
    }

    private static boolean isReferenceType(String type) {
        return type != null && type.startsWith(REFERENCE_TYPE_PREFIX) && type.endsWith(REFERENCE_TYPE_SUFFIX);
    }

    static {
        TYPES.put(boolean.class.getName(), boolean.class);
        TYPES.put(byte.class.getName(), byte.class);
        TYPES.put(short.class.getName(), short.class);
        TYPES.put(int.class.getName(), int.class);
        TYPES.put(long.class.getName(), long.class);
        TYPES.put(float.class.getName(), float.class);
        TYPES.put(double.class.getName(), double.class);
        TYPES.put(void.class.getName(), void.class);
        TYPES.put("Z", boolean.class);
        TYPES.put("B", byte.class);
        TYPES.put("C", char.class);
        TYPES.put("D", double.class);
        TYPES.put("F", float.class);
        TYPES.put("I", int.class);
        TYPES.put("J", long.class);
        TYPES.put("S", short.class);
    }
}
