package com.spring.boxes.dollar.term;

import java.io.Serializable;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.function.Function;

import com.google.common.base.Preconditions;
import com.spring.boxes.dollar.JSONUtils;

import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.NoArgsConstructor;

// 支撑json到object、list、map、
@Data
@NoArgsConstructor
@AllArgsConstructor
public class StringValue implements Serializable {

    private String type;

    private Class<?> type1;

    private Class<?> type2;

    private Class<?> type3;

    public static StringValue newMap(Class<?> keyType, Class<?> valueType) {
        return new StringValue(_MAP, Map.class, keyType, valueType);
    }

    public static StringValue newObj(Class<?> valueType) {
        if (valueType == String.class || valueType == Number.class) {
            throw new UnsupportedOperationException("only for object json serialize");
        }
        if (valueType == List.class) {
            throw new UnsupportedOperationException("should be use _list type");
        }
        if (valueType == Set.class) {
            throw new UnsupportedOperationException("should be use _set type");
        }
        if (valueType == Map.class) {
            throw new UnsupportedOperationException("should be use _map type");
        }
        return new StringValue(_OBJ, Map.class, valueType, Void.class);
    }

    public static StringValue newSet(Class<?> valueType) {
        return new StringValue(_SET, Set.class, valueType, Void.class);
    }

    public static StringValue newList(Class<?> valueType) {
        return new StringValue(_LIST, List.class, valueType, Void.class);
    }

    public static <T> T ofValue(String source, Function<String, T> function) {
        return function.apply(source);
    }

    public static <T> T ofValue(String source, StringValue settingValue, BiFunction<String, StringValue, T> function) {
        return function.apply(source, settingValue);
    }

    public static Object ofValue(String source, StringValue stringValue) {
        if (Objects.isNull(source)) {
            return null;
        }
        if (_OBJ.equals(stringValue.getType())) {
            return JSONUtils.fromJSON(source, stringValue.getType1());
        }
        if (_LIST.equals(stringValue.getType())) {
            return JSONUtils.fromJSON(source, List.class, stringValue.getType2());
        }
        if (_SET.equals(stringValue.getType())) {
            return JSONUtils.fromJSON(source, Set.class, stringValue.getType2());
        }
        if (_MAP.equals(stringValue.getType())) {
            return JSONUtils.fromJSON(source, Map.class, stringValue.getType2(), stringValue.getType3());
        }
        return null;
    }

    public static <T> List<T> ofList(String source, StringValue stringValue) {
        Preconditions.checkArgument(Objects.nonNull(stringValue), "empty stringValue");
        Preconditions.checkArgument(_LIST.equals(stringValue.getType()),
                String.format("stringValue mismatch _list (%s)", stringValue.getType()));
        return JSONUtils.fromJSON(source, List.class, stringValue.getType2());
    }

    public static <T> Set<T> ofSet(String source, StringValue stringValue) {
        Preconditions.checkArgument(Objects.nonNull(stringValue), "empty stringValue");
        Preconditions.checkArgument(_SET.equals(stringValue.getType()),
                String.format("stringValue mismatch _set (%s)", stringValue.getType()));
        return JSONUtils.fromJSON(source, Set.class, stringValue.getType2());
    }

    @SuppressWarnings("unchecked")
    public static <T> T ofObj(String source, StringValue stringValue) {
        Preconditions.checkArgument(Objects.nonNull(stringValue), "empty stringValue");
        Preconditions.checkArgument(_OBJ.equals(stringValue.getType()),
                String.format("stringValue mismatch _obj (%s)", stringValue.getType()));
        return (T) JSONUtils.fromJSON(source, stringValue.getType1());
    }

    @SuppressWarnings("unchecked")
    public static <K, V> Map<K, V> ofMap(String source, StringValue stringValue) {
        Preconditions.checkArgument(Objects.nonNull(stringValue), "empty stringValue");
        Preconditions.checkArgument(_MAP.equals(stringValue.getType()),
                String.format("stringValue mismatch _map (%s)", stringValue.getType()));
        return (Map<K, V>) JSONUtils.fromJSON(source, Map.class, stringValue.getType2(), stringValue.getType3());
    }

    public static final String _OBJ = "_obj";
    public static final String _MAP = "_map";
    public static final String _SET = "_set";
    public static final String _LIST = "_list";


}
