/*
 * Copyright (c) 2005-2023 Xceptance Software Technologies GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.xceptance.xlt.api.actions;

import java.net.URL;
import java.util.List;

import org.htmlunit.HttpMethod;
import org.htmlunit.WebRequest;
import org.htmlunit.util.NameValuePair;

import com.xceptance.common.util.ParameterCheckUtils;
import com.xceptance.xlt.api.htmlunit.LightWeightPage;
import com.xceptance.xlt.engine.SessionImpl;
import com.xceptance.xlt.engine.XltWebClient;
import com.xceptance.xlt.engine.resultbrowser.RequestHistory;

/**
 * AbstractLightWeightPageAction is the base class for all HTML/XML-based actions which do simple page processing only.
 * In contrast to {@link AbstractHtmlPageAction}, the loaded page is represented as a single unparsed string. This makes
 * it harder to pick certain elements of the page, but completely avoids the client-side overhead to parse the page and
 * represent it as a tree of elements. Especially with complex pages, the overhead may be significantly lower this way.
 * So this class should be used, if either the page processing is simple enough or the test clients turn out to be the
 * bottleneck.
 * 
 * @author Jörg Werner (Xceptance Software Technologies GmbH)
 */
public abstract class AbstractLightWeightPageAction extends AbstractWebAction
{
    /**
     * The (unparsed) content of the web page loaded by this action.
     */
    private LightWeightPage page;

    /**
     * The URL that was used to load the page.
     */
    private URL url;

    /**
     * Creates a new AbstractLightWeightPageAction object and gives it the passed timer name. This constructor is
     * typically used for an intermediate action in a sequence of actions, i.e. it has a previous action.
     * 
     * @param previousAction
     *            the action that preceded the current action
     * @param timerName
     *            the name of the timer that is associated with this action
     */
    protected AbstractLightWeightPageAction(final AbstractWebAction previousAction, final String timerName)
    {
        super(previousAction, timerName);
    }

    /**
     * Creates a new AbstractLightWeightPageAction object and gives it the passed timer name. This constructor is
     * typically used for the first action in a sequence of actions, i.e. it has no previous action.
     * 
     * @param timerName
     *            the name of the timer that is associated with this action
     */
    protected AbstractLightWeightPageAction(final String timerName)
    {
        this(null, timerName);
    }

    /**
     * Returns the action that was passed as the previous action to the constructor. Allows to access data collected
     * during the previous action.
     * 
     * @return the previous action (may be null)
     */
    @Override
    public AbstractLightWeightPageAction getPreviousAction()
    {
        return (AbstractLightWeightPageAction) super.getPreviousAction();
    }

    /**
     * Returns the light-weight page generated by this action.
     * 
     * @return the page
     */
    public LightWeightPage getLightWeightPage()
    {
        return page;
    }

    /**
     * Returns the unparsed content of the web page loaded by this action.
     * 
     * @return the content of the loaded page as string
     */
    public String getContent()
    {
        final LightWeightPage page = getLightWeightPage();
        return page != null ? page.getContent() : null;
    }

    /**
     * Returns the response code of the last loaded page.
     * 
     * @return the HTTP response code
     */
    public int getHttpResponseCode()
    {
        final LightWeightPage page = getLightWeightPage();
        return page != null ? page.getHttpResponseCode() : 0;
    }

    /**
     * Returns the URL that was used to load the page. Subsequent actions may use this value to create new URLs based on
     * this URL.
     * 
     * @return the URL
     */
    public URL getURL()
    {
        return url;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void run() throws Throwable
    {
        try
        {
            super.run();
        }
        finally
        {
            /*
             * Dump the page not before the very end of this action. This way, all requests that are executed after one
             * of the loadPageByXXX() methods are correctly associated with this action.
             */
            dumpPage(getLightWeightPage());
        }
    }

    /**
     * Sets the given light-weight page object to be the result of this action. Typically, the light-weight page is
     * generated by one of the <code>loadPage()</code> methods and need not be set explicitly.
     * 
     * @param page
     *            the page to set
     */
    public void setLightWeightPage(final LightWeightPage page)
    {
        ParameterCheckUtils.isNotNull(page, "page");

        this.page = page;
    }

    /**
     * Loads the page using HTTP GET from the passed URL and stores it internally. The URL is stored for further
     * reference.
     * 
     * @param url
     *            the target URL
     * @throws Exception
     *             if an error occurred while loading the page
     */
    protected void loadPage(final URL url) throws Exception
    {
        loadPage(url, HttpMethod.GET, EMPTY_PARAMETER_LIST);
    }

    /**
     * Creates an URL from the passed string, loads the page using HTTP GET and stores it internally. The URL is stored
     * for further reference.
     * 
     * @param urlAsString
     *            an URL as string
     * @throws Exception
     *             if an error occurred while loading the page
     */
    protected void loadPage(final String urlAsString) throws Exception
    {
        final URL url = new URL(urlAsString);

        loadPage(url);
    }

    /**
     * Loads the page using the given request method from the passed URL and stores it internally. The specified request
     * parameters are added either to the request body if the request method is {@link HttpMethod#POST} or appended to
     * the query string of the target URL for any other request method.
     * 
     * @param url
     *            the target URL
     * @param method
     *            the HTTP request method to be used
     * @param requestParameters
     *            the list of custom parameters to add
     * @throws Exception
     *             if an error occurred while loading the page
     */
    protected void loadPage(final URL url, final HttpMethod method, final List<NameValuePair> requestParameters) throws Exception
    {
        final WebRequest webRequest = createWebRequestSettings(url, method, requestParameters);

        this.url = webRequest.getUrl();
        setLightWeightPage(((XltWebClient) getWebClient()).getLightWeightPage(webRequest));
    }

    /**
     * Dumps the given HTML page object to the request history.
     * 
     * @param page
     *            the page to dump
     */
    private void dumpPage(final LightWeightPage page)
    {
        final RequestHistory requestHistory = SessionImpl.getCurrent().getRequestHistory();

        if (page != null)
        {
            requestHistory.add(page);
        }
        else
        {
            // dump at least an empty page to let the action appear in the result browser
            final String timerName = ((XltWebClient) getWebClient()).getTimerName();
            requestHistory.add(timerName);
        }
    }
}
