package com.xzchaoo.commons.basic.concurrent;

import java.time.Duration;
import java.util.Objects;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

import javax.annotation.concurrent.ThreadSafe;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 一个异步串行执行的任务.
 * 用户代码调用 {@link #trigger()} 尝试触发异步任务执行. 如果当前异步任务没有在执行或在调度中, 那么会使用{@link #executor}调度它.
 * 使用CAS确保在多线程环境行task是串行执行的.
 *
 * <p>created at 2021/1/27
 *
 * @author xiangfeng.xzc
 */
@ThreadSafe
public class AsyncThrottleSerializingTask {
    private static final Logger LOGGER = LoggerFactory.getLogger(AsyncThrottleSerializingTask.class);

    private final Executor      executor;
    private final Runnable      task;
    private final AtomicInteger wip = new AtomicInteger();

    public AsyncThrottleSerializingTask(Runnable task, Executor executor) {
        this.task = Objects.requireNonNull(task);
        this.executor = Objects.requireNonNull(executor);
    }

    public AsyncThrottleSerializingTask(Runnable task, ScheduledExecutorService executor, Duration delay) {
        this.task = Objects.requireNonNull(task);
        Objects.requireNonNull(executor);
        // 我们只支持到mill精度
        long mills = delay.toMillis();
        this.executor = (t) -> executor.schedule(t, mills, TimeUnit.MILLISECONDS);
    }

    public void trigger() {
        if (wip.getAndIncrement() == 0) {
            // 如果被reject, 那我也没办法了, 能不能加点冷却时间
            executor.execute(this::run0);
        }
    }

    private void run0() {
        int delta = wip.get();
        do {
            try {
                task.run();
            } catch (Throwable e) {
                LOGGER.error("Exception when executing task {}", task, e);
            }
            delta = wip.addAndGet(-delta);
        } while (delta != 0);
    }
}
