package com.xzchaoo.commons.basic.concurrent;

import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.BooleanSupplier;

/**
 * <p>参考disruptor里的LiteBlockingWaitStrategy, 用于可能会用到condition但使用频率非常少的情况.
 * 比如在某些场景(比如优雅关机时), 我想等待一个ConcurrentHashMap非空, 不浪费太多资源实时性又高的方法是condition机制(非自旋).
 * 在Stop方法里等待该Condition; 然后在修改CHM的方法里可能要signal该condition.
 * 如果不使用该类, 那么每次修改CHM后都可能要通知condition(因为condition没有属性可以表明它是否有线程在等待), 此时该类就可以起到优化效果了.
 * <p>created at 2021/2/13
 *
 * @author xiangfeng.xzc
 */
@SuppressWarnings("AlibabaLockShouldWithTryFinally")
public class LiteCondition {
    /**
     * 如果其他线程读到needNotify为true表明有线程在等待condition.
     */
    private final AtomicBoolean needNotify = new AtomicBoolean();
    private final Lock lock;
    private final Condition condition;
    
    public LiteCondition() {
        this(new ReentrantLock());
    }

    public LiteCondition(Lock lock) {
        this.lock = lock;
        this.condition = lock.newCondition();
    }

    public static LiteCondition create(Lock lock) {
        return new LiteCondition(lock);
    }

    public void wait(BooleanSupplier b) {
        lock.lock();
        try {
            for (; ; ) {
                needNotify.set(true);
                if (b.getAsBoolean()) {
                    break;
                }
                condition.awaitUninterruptibly();
            }
        } finally {
            lock.unlock();
        }
    }

    public long waitNanos(BooleanSupplier b, long nanos) throws InterruptedException {
        lock.lock();
        try {
            for (; ; ) {
                needNotify.set(true);
                if (b.getAsBoolean()) {
                    return nanos;
                }
                nanos = condition.awaitNanos(nanos);
                if (nanos <= 0) {
                    return nanos;
                }
            }
        } finally {
            lock.unlock();
        }
    }

    public void signal() {
        if (needNotify.get() && needNotify.compareAndSet(true, false)) {
            lock.lock();
            try {
                condition.signal();
            } finally {
                lock.unlock();
            }
        }
    }

    public void signalAll() {
        // 经过JMH测试: 如果 needNotify 在极少数的情况下才为true
        // 那么 get 的吞吐量是 compareAndSet 的三倍+: 因此先用get短路一下是很快的
        // 理论上只有stop时才会进入才分支, 这只会在生命周期里进入一次, 频率是非常少的
        if (needNotify.get() && needNotify.compareAndSet(true, false)) {
            lock.lock();
            try {
                condition.signalAll();
            } finally {
                lock.unlock();
            }
        }
    }
}
