package com.xzchaoo.commons.basic.config;

import java.util.Objects;
import java.util.concurrent.CopyOnWriteArrayList;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * <p>created at 2020-08-07
 *
 * @author xiangfeng.xzc
 */
public abstract class AbstractConfig implements Config {
    private static final Logger LOGGER = LoggerFactory
            .getLogger(AbstractConfig.class);

    protected final CopyOnWriteArrayList<Listener> listeners
            = new CopyOnWriteArrayList<>();

    protected final Manager manager;

    protected AbstractConfig(Manager manager) {
        this.manager = Objects.requireNonNull(manager);
    }

    @Override
    public double getDouble(String key, double defaultValue) {
        String value = getString(key);
        return value != null ? Double.parseDouble(value) : defaultValue;
    }

    @Override
    public Double getDouble(String key) {
        String value = getString(key);
        return value != null ? Double.parseDouble(value) : null;
    }

    @Override
    public int getInt(String key, int defaultValue) {
        String value = getString(key);
        return value != null ? Integer.parseInt(value) : defaultValue;
    }

    @Override
    public Integer getInteger(String key) {
        String value = getString(key);
        return value != null ? Integer.parseInt(value) : null;
    }
    @Override
    public Long getLong(String key) {
        String value = getString(key);
        return value != null ? Long.parseLong(value) : null;
    }

    @Override
    public long getLong(String key, long defaultValue) {
        String value = getString(key);
        return value != null ? Long.parseLong(value) : defaultValue;
    }

    @Override
    public String getString(String key, String defaultValue) {
        String value = getString(key);
        return value != null ? value : defaultValue;
    }

    @Override
    public Boolean getBoolean(String key) {
        String value = getString(key);
        return value != null ? Boolean.parseBoolean(value) : null;
    }

    @Override
    public boolean getBoolean(String key, boolean defaultValue) {
        String value = getString(key);
        return value != null ? Boolean.parseBoolean(value) : defaultValue;
    }

    @Override
    public void addListener(Listener listener) {
        listeners.add(listener);
    }

    @Override
    public Manager manager() {
        return manager;
    }

    @Override
    public String name() {
        throw new UnsupportedOperationException();
    }

    @Override
    public void removeListener(Listener listener) {
        listeners.remove(listener);
    }

    @Override
    public void lockAndListen(Runnable runnable, Listener listener) {
        lock();
        try {
            runnable.run();
            listeners.add(listener);
        } finally {
            unlock();
        }
    }

    protected void fireChange() {
        lock();
        try {
            Event event = new Event(Event.Type.CHANGED);
            for (Listener listener : listeners) {
                try {
                    listener.onEvent(event);
                } catch (Throwable e) {
                    LOGGER.error("Exception caught when calling onEvent on "
                            + "listener {}", listener, e);
                }
            }
        } finally {
            unlock();
        }
    }

    protected final void lock() {
        manager.lock().lock();
    }

    protected final void unlock() {
        manager.lock().unlock();
    }
}
